// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "MyMachine.h"

MyMachine::MyMachine(void)
{
	wheel = new H3D::Helper::Mesh::H3D();
	chassis = new H3D::Helper::Mesh::H3D();
}

MyMachine::~MyMachine(void)
{
	delete chassis;
	delete wheel;
}

bool
MyMachine::load(const std::string &machine, dWorldID world, dSpaceID space, const float *position)
{
	unsigned int i;
	dMass mass;
	dQuaternion q;
	const float *anchor;
	ALuint buffer;

	info.speed = 70.0f;
	info.steer = 20.0f;

	body[0] = dBodyCreate(world);
	dMassSetBox(&mass, 1.0f, 2.5f, 1.0f, 5.5f);
	dMassAdjust(&mass, 1.0f);
	dBodySetMass(body[0], &mass);
	geom[0] = dCreateBox(space, 2.5f, 1.0f, 5.5f);
	dGeomSetBody(geom[0], body[0]);
	dBodySetPosition(body[0], position[0], position[1], position[2]);

	dQFromAxisAndAngle(q, 0.0f, 1.0f, 0.0f, M_PI * 0.5f);
	dMassSetSphere(&mass, 0.05f, 0.25f);
	dMassAdjust(&mass, 0.05f);
	for(i = 1; i < 5; i++)
	{
		body[i] = dBodyCreate(world);
		dBodySetQuaternion(body[i], q);
		dBodySetMass(body[i], &mass);
		geom[i] = dCreateSphere(space, 0.25f);
		dGeomSetBody(geom[i], body[i]);
	}
	dBodySetPosition(body[1], position[0] - 1.5f, position[1] - 0.5f, position[2] + 1.25f);
	dBodySetPosition(body[2], position[0] + 1.5f, position[1] - 0.5f, position[2] + 1.25f);
	dBodySetPosition(body[3], position[0] - 1.5f, position[1] - 0.5f, position[2] - 1.5f);
	dBodySetPosition(body[4], position[0] + 1.5f, position[1] - 0.5f, position[2] - 1.5f);

	for(i = 0; i < 4; i++)
	{
		joint[i] = dJointCreateHinge2(world, NULL);
		dJointAttach(joint[i], body[0], body[i + 1]);
		anchor = dBodyGetPosition(body[i + 1]);
		dJointSetHinge2Anchor(joint[i], anchor[0], anchor[1], anchor[2]);
		dJointSetHinge2Axis1(joint[i], 0.0f, 1.0f, 0.0f);
		dJointSetHinge2Axis2(joint[i], 1.0f, 0.0f, 0.0f);
		dJointSetHinge2Param(joint[i], dParamSuspensionERP, 0.4f);
		dJointSetHinge2Param(joint[i], dParamSuspensionCFM, 0.8f);
	}
	for(i = 0; i < 2; i++)
	{
		dJointSetHinge2Param(joint[i], dParamLoStop, 0);
		dJointSetHinge2Param(joint[i], dParamHiStop, 0);
	}

	chassis->load(PACKAGE_DATADIR STRPS + machine);
	chassis->glLoad(PACKAGE_DATADIR STRPS "textures");
	wheel->load(PACKAGE_DATADIR STRPS "wheel");
	wheel->glLoad(PACKAGE_DATADIR STRPS "textures");

	alGenSources(1, &engine);
	alGenBuffers(1, &buffer);
	H3D::Sound::H3D* sound = H3D::Sound::VORBIS::Load(PACKAGE_DATADIR STRPS "sounds" STRPS "engine.ogg");
	alBufferData(buffer, H3D::Helper::OpenAL::Sound::Format(sound->info.nchannels), sound->content.samples, sound->info.nframes * sound->info.nchannels * sizeof(short), sound->info.rate);
	sound->unload();
	delete sound;
	alSourcei(engine, AL_BUFFER, buffer);
	alSourcei(engine, AL_LOOPING, AL_TRUE);
	alSource3f(engine, AL_POSITION, position[0], position[1], position[2]);
	alSourcei(engine, AL_ROLLOFF_FACTOR, 0.1f);
	alSourcePlay(engine);

	return true;
}

void
MyMachine::unload(void)
{
	ALuint buffer;
	unsigned int i;

	alGetSourcei(engine, AL_BUFFER, (int*)&buffer);
	alDeleteSources(1, &engine);
	alDeleteBuffers(1, &buffer);

	wheel->glUnload();
	wheel->unload();
	chassis->glUnload();
	chassis->unload();

	for(i = 0; i < 4; i++) dJointDestroy(joint[i]);
	for(i = 0; i < 5; i++) {dGeomDestroy(geom[i]); dBodyDestroy(body[i]);}
}

void
MyMachine::frame(void)
{
	H3D::Helper::Mesh::H3D *mesh;
	const float *p, *r;
	float m[16], v;
	unsigned int i;

	v = 0;
	for(i = 0; i < 4; i++)
		v += H3D::Math::Vector::NormFunc(dBodyGetAngularVel(body[i + 1]));
	alSourcef(engine, AL_PITCH, H3D::Math::Clamp(v / 80.0f, 0.0f, 1.5f) + 0.5f);

	p = dBodyGetPosition(body[0]);
	alSource3f(engine, AL_POSITION, p[0], p[1], p[2]);
	p = dBodyGetLinearVel(body[0]);
	alSource3f(engine, AL_VELOCITY, p[0], p[1], p[2]);
	r = dBodyGetRotation(body[0]);
	alSource3f(engine, AL_DIRECTION, -r[2], -r[6], -r[10]);

	for(i = 0; i < 5; i++)
	{
		mesh = i ? wheel : chassis;
		p = dBodyGetPosition(body[i]);
        	r = dBodyGetRotation(body[i]);
		H3D::Helper::OpenGL::OpenDE::Matrix(p, r, m);
		glPushMatrix();
			glMultMatrixf(m);
			mesh->draw();
		glPopMatrix();
	}
}

void
MyMachine::enable(void)
{
	alSourcePlay(engine);
	for(unsigned int i = 0; i < 5; i++)
	{
		dBodyEnable(body[i]);
		dGeomEnable(geom[i]);
	}
}

void
MyMachine::disable(void)
{
	for(unsigned int i = 0; i < 5; i++)
	{
		dGeomDisable(geom[i]);
		dBodyDisable(body[i]);
	}
	alSourceStop(engine);
}

void
MyMachine::drive(bool up, bool down, bool left, bool right, bool turbo)
{
	unsigned int i;
	const float speed = info.speed * (turbo ? 2.0f : 1.0f) * (up ? 1.0f : (down ? -1.0f : 0.0f));
	const float steer = info.steer * (left ? -1.0f : (right ? 1.0f : 0.0f));
	if(steer)
	{
		for(i = 2; i < 4; i++)
		{
			dJointSetHinge2Param(joint[i], dParamVel, steer);
			dJointSetHinge2Param(joint[i], dParamFMax, 4.0f);
			dJointSetHinge2Param(joint[i], dParamLoStop, -0.75f);
			dJointSetHinge2Param(joint[i], dParamHiStop, 0.75f);
			dJointSetHinge2Param(joint[i], dParamFudgeFactor, 0.1f);
		}
	}
	else
	{
		for(i = 2; i < 4; i++)
		{
			dJointSetHinge2Param(joint[i], dParamLoStop, 0.0f);
			dJointSetHinge2Param(joint[i], dParamHiStop, 0.0f);
			dJointSetHinge2Param(joint[i], dParamFudgeFactor, 0.1f);
		}
	}
	for(i = 0; i < 4; i++)
	{
		dJointSetHinge2Param(joint[i], dParamVel2, speed);
		dJointSetHinge2Param(joint[i], dParamFMax2, 2.0f);
	}
	for(i = 0; i < 2; i++)
	{
		dJointSetHinge2Param(joint[i], dParamLoStop, 0.0f);
		dJointSetHinge2Param(joint[i], dParamHiStop, 0.0f);
		dJointSetHinge2Param(joint[i], dParamFudgeFactor, 0.01f);
	}
}

void
MyMachine::place(const float *position, const float *rotation)
{
	float matrix[12], wposition[3];

	H3D::Helper::OpenDE::Rotation(rotation, matrix);
	dBodySetPosition(body[0], position[0], position[1], position[2]);
	dBodySetRotation(body[0], matrix);
	dBodySetLinearVel(body[0], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(body[0], 0.0f, 0.0f, 0.0f);

	H3D::Math::Vector::Load(wposition, -0.5f, - 0.5f, 1.25f);
	H3D::Math::Vector::MatrixProductInPlace(wposition, rotation);
	H3D::Math::Vector::AddInPlace(wposition, position);
	dBodySetPosition(body[1], wposition[0], wposition[1], wposition[2]);
	dBodySetRotation(body[1], matrix);
	dBodySetLinearVel(body[1], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(body[1], 0.0f, 0.0f, 0.0f);

	H3D::Math::Vector::Load(wposition, 1.5f, -0.5f, 1.25f);
	H3D::Math::Vector::MatrixProductInPlace(wposition, rotation);
	H3D::Math::Vector::AddInPlace(wposition, position);
	dBodySetPosition(body[2], wposition[0], wposition[1], wposition[2]);
	dBodySetRotation(body[2], matrix);
	dBodySetLinearVel(body[2], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(body[2], 0.0f, 0.0f, 0.0f);

	H3D::Math::Vector::Load(wposition, -1.5f, -0.5f, -1.5f);
	H3D::Math::Vector::MatrixProductInPlace(wposition, rotation);
	H3D::Math::Vector::AddInPlace(wposition, position);
	dBodySetPosition(body[3], wposition[0], wposition[1], wposition[2]);
	dBodySetRotation(body[3], matrix);
	dBodySetLinearVel(body[3], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(body[3], 0.0f, 0.0f, 0.0f);

	H3D::Math::Vector::Load(wposition, 1.5f, -0.5f, -1.5f);
	H3D::Math::Vector::MatrixProductInPlace(wposition, rotation);
	H3D::Math::Vector::AddInPlace(wposition, position);
	dBodySetPosition(body[4], wposition[0], wposition[1], wposition[2]);
	dBodySetRotation(body[4], matrix);
	dBodySetLinearVel(body[4], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(body[4], 0.0f, 0.0f, 0.0f);
}
