// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "H3D/Math.h"
#include <stdio.h>
#include <string.h>

namespace H3D
{
	namespace Math
	{
		namespace Point
		{
			const float Zero[3] = {0, 0, 0};

			void
			Print(const float *a)
			{
				printf("(%f, %f, %f)\n", a[0], a[1], a[2]);
			}

			void
			Copy(const float *a, float *result)
			{
				memcpy(result, a, 3 * sizeof(float));
			}

			void
			Load(float *a, float x, float y, float z)
			{
				a[0] = x; a[1] = y; a[2] = z;
			}

			void
			Add(const float *a, const float *b, float *result)
			{
				unsigned int i;
				for(i = 0; i < 3; i++)
					result[i] = a[i] + b[i];
			}

			void
			AddInPlace(float *a, const float *b)
			{
				unsigned int i;
				for(i = 0; i < 3; i++)
					a[i] += b[i];
			}

			void
			Sub(const float *a, const float *b, float *result)
			{
				unsigned int i;
				for(i = 0; i < 3; i++)
					result[i] = a[i] - b[i];
			}

			void
			SubInPlace(float *a, const float *b)
			{
				unsigned int i;
				for(i = 0; i < 3; i++)
					a[i] -= b[i];
			}
		}

		namespace Vector
		{
			const float X[3] = {1.0f, 0.0f, 0.0f};
			const float Y[3] = {0.0f, 1.0f, 0.0f};
			const float Z[3] = {0.0f, 0.0f, 1.0f};
			const float NegX[3] = {-1.0f, 0.0f, 0.0f};
			const float NegY[3] = {0.0f, -1.0f, 0.0f};
			const float NegZ[3] = {0.0f, 0.0f, -1.0f};

			void
			Print(const float *a)
			{
				printf("(%f, %f, %f)\n", a[0], a[1], a[2]);
			}

			void
			Copy(const float *a, float *result)
			{
				memcpy(result, a, 3 * sizeof(float));
			}

			void
			Load(float *a, float x, float y, float z)
			{
				a[0] = x; a[1] = y; a[2] = z;
			}

			void
			Add(const float *a, const float *b, float *result)
			{
				unsigned int i;
				for(i = 0; i < 3; i++)
					result[i] = a[i] + b[i];
			}

			void
			AddInPlace(float *a, const float *b)
			{
				unsigned int i;
				for(i = 0; i < 3; i++)
					a[i] += b[i];
			}

			void
			Sub(const float *a, const float *b, float *result)
			{
				unsigned int i;
				for(i = 0; i < 3; i++)
					result[i] = a[i] - b[i];
			}

			void
			SubInPlace(float *a, const float *b)
			{
				unsigned int i;
				for(i = 0; i < 3; i++)
					a[i] -= b[i];
			}

			void
			ScalarProduct(const float *a, float b, float *result)
			{
				unsigned int i;
				for(i = 0; i < 3; i++)
					result[i] = a[i] * b;
			}

			void
			ScalarProductInPlace(float *a, float b)
			{
				unsigned int i;
				for(i = 0; i < 3; i++)
					a[i] *= b;
			}

			void
			Norm(const float *a, float *result)
			{
				*result = sqrt(a[0] * a[0] + a[1] * a[1] + a[2] * a[2]);
			}

			float
			NormFunc(const float *a)
			{
				return sqrt(a[0] * a[0] + a[1] * a[1] + a[2] * a[2]);
			}

			void
			DotProduct(const float *a, const float *b, float *result)
			{
				*result = a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
			}

			float
			DotProductFunc(const float *a, const float *b)
			{
				return a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
			}

			void
			CrossProduct(const float *a, const float *b, float *result)
			{
				result[0] = a[1] * b[2] - a[2] * b[1];
				result[1] = a[2] * b[0] - a[0] * b[2];
				result[2] = a[0] * b[1] - a[1] * b[0];
			}

			void
			CrossProductInPlace(float *a, const float *b)
			{
				float result[3];
				CrossProduct(a, b, result);
				Copy(result, a);
			}

			void
			Normal(const float *a, float *result)
			{
				float norm;
				Norm(a, &norm);
				ScalarProduct(a, 1.0f / norm, result);
			}

			void
			NormalInPlace(float *a)
			{
				ScalarProductInPlace(a, 1.0f / NormFunc(a));
			}

			void
			MatrixProduct(const float *a, const float *b, float *result)
			{
				result[0] = b[0] * a[0] + b[4] * a[1] + b[8] * a[2] + b[12];
				result[1] = b[1] * a[0] + b[5] * a[1] + b[9] * a[2] + b[13];
				result[2] = b[2] * a[0] + b[6] * a[1] + b[10] * a[2] + b[14];
			}

			void
			MatrixProductInPlace(float *a, const float *b)
			{
				float result[3];
				MatrixProduct(a, b, result);
				Copy(result, a);
			}

			void
			Rotate(const float *a, const float *axis, float angle, float *result)
			{
				const float c = cos(DegToRad(angle));
				const float s = sin(DegToRad(angle));
				const float matrix[16] =
				{
					axis[0] * axis[0] * (1.0f - c) + c, axis[1] * axis[0] * (1.0f - c) + axis[2] * s, axis[0] * axis[2] * (1.0f - c) - axis[1] * s, 0.0f,
					axis[0] * axis[1] * (1.0f - c) - axis[2] * s, axis[1] * axis[1] * (1.0f - c) + c, axis[1] * axis[2] * (1.0f - c) + axis[0] * s, 0.0f,
					axis[0] * axis[2] * (1.0f - c) + axis[1] * s, axis[1] * axis[2] * (1.0f - c) - axis[0] * s, axis[2] * axis[2] * (1.0f - c) + c, 0.0f,
					0.0f, 0.0f, 0.0f, 1.0f
				};
				MatrixProduct(a, matrix, result);
			}

			void
			RotateInPlace(float *a, const float *axis, float angle)
			{
				float result[3];
				Rotate(a, axis, angle, result);
				Copy(result, a);
			}

			void
			Angle(const float *a, const float *b, float *angle)
			{
				float dp, na, nb;
				DotProduct(a, b, &dp);
				Norm(a, &na);
				Norm(b, &nb);
				*angle = RadToDeg(acos(dp / (na * nb)));
			}

			float
			AngleFunc(const float *a, const float *b)
			{
				float dp, na, nb;
				DotProduct(a, b, &dp);
				Norm(a, &na);
				Norm(b, &nb);
				return RadToDeg(acos(dp / (na * nb)));
			}
		}

		namespace Matrix
		{
			const float I[16] =
			{
				1.0f, 0.0f, 0.0f, 0.0f,
				0.0f, 1.0f, 0.0f, 0.0f,
				0.0f, 0.0f, 1.0f, 0.0f,
				0.0f, 0.0f, 0.0f, 1.0f,
			};

			void
			Print(const float *a)
			{
				printf("[(%f, %f, %f, %f), (%f, %f, %f, %f), (%f, %f, %f, %f), (%f, %f, %f, %f)]\n", a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9], a[10], a[11], a[12], a[13], a[14], a[15]);
			}

			void
			Copy(const float *a, float *result)
			{
				memcpy(result, a, 16 * sizeof(float));
			}

			void
			Load(float *x, float a, float b, float c, float d, float e, float f, float g, float h, float i, float j, float k, float l, float m, float n, float o, float p)
			{
				x[0] = a; x[4] = e; x[8] = i; x[12] = m;
				x[1] = b; x[5] = f; x[9] = j; x[13] = n;
				x[2] = c; x[6] = g; x[10] = k; x[14] = o;
				x[3] = d; x[7] = h; x[11] = l; x[15] = p;
			}

			void
			LoadIdentity(float *a)
			{
				unsigned int i;
				for(i = 0; i < 16; i++)
					a[i] = (i % 5) == 0 ? 1.0f : 0.0f;
			}

			void
			Add(const float *a, const float *b, float *result)
			{
				unsigned int i;
				for(i = 0; i < 16; i++)
					result[i] = a[i] + b[i];
			}

			void
			AddInPlace(float *a, const float *b)
			{
				unsigned int i;
				for(i = 0; i < 16; i++)
					a[i] += b[i];
			}

			void
			Sub(const float *a, const float *b, float *result)
			{
				unsigned int i;
				for(i = 0; i < 16; i++)
					result[i] = a[i] - b[i];
			}

			void
			SubInPlace(float *a, const float *b)
			{
				unsigned int i;
				for(i = 0; i < 16; i++)
					a[i] -= b[i];
			}

			void
			ScalarProduct(const float *a, float b, float *result)
			{
				unsigned int i;
				for(i = 0; i < 16; i++)
					result[i] = a[i] * b;
			}

			void
			ScalarProductInPlace(float *a, float b)
			{
				unsigned int i;
				for(i = 0; i < 16; i++)
					a[i] *= b;
			}

			void
			Product(const float *a, const float *b, float *result)
			{
				unsigned int i, j;
				for(j = 0; j < 4; j++)
					for(i = 0; i < 4; i++)
						result[j * 4 + i] = a[0 * 4 + i] * b[j * 4 + 0] + a[1 * 4 + i] * b[j * 4 + 1] + a[2 * 4 + i] * b[j * 4 + 2] + a[3 * 4 + i] * b[j * 4 + 3];
			}

			void
			ProductInPlace(float *a, const float *b)
			{
				float result[16];
				Product(a, b, result);
				Copy(result, a);
			}

			void
			VectorProduct(const float *a, const float *b, float *result)
			{
				result[0] = a[0] * b[0] + a[4] * b[1] + a[8] * b[2] + a[12];
				result[1] = a[1] * b[0] + a[5] * b[1] + a[9] * b[2] + a[13];
				result[2] = a[2] * b[0] + a[6] * b[1] + a[10] * b[2] + a[14];
			}

			void
			VectorProductInPlace(const float *a, float *b)
			{
				float result[3];
				VectorProduct(a, b, result);
				Vector::Copy(result, b);
			}

			void
			Transpose(const float *a, float *result)
			{
				unsigned int i, j;
				for(j = 0; j < 4; j++)
					for(i = 0; i < 4; i++)
						result[j * 4 + i] = a[i * 4 + j];
			}

			void
			TransposeInPlace(float *a)
			{
				float result[16];
				Transpose(a, result);
				Copy(result, a);
			}

			void
			Scale(const float *a, const float *b, float *result)
			{
				const float c[16] =
				{
					b[0], 0.0f, 0.0f, 0.0f,
					0.0f, b[1], 0.0f, 0.0f,
					0.0f, 0.0f, b[2], 0.0f,
					0.0f, 0.0f, 0.0f, 1.0f
				};
				Product(a, c, result);
			}

			void
			ScaleInPlace(float *a, const float *b)
			{
				float result[16];
				Scale(a, b, result);
				Copy(result, a);
			}

			void
			Rotate(const float *a, const float *axis, float angle, float *result)
			{
				const float c = cos(DegToRad(angle));
				const float s = sin(DegToRad(angle));
				const float b[16] = 
				{
					axis[0] * axis[0] * (1.0f - c) + c, axis[1] * axis[0] * (1.0f - c) + axis[2] * s, axis[0] * axis[2] * (1.0f - c) - axis[1] * s, 0.0f,
					axis[0] * axis[1] * (1.0f - c) - axis[2] * s, axis[1] * axis[1] * (1.0f - c) + c, axis[1] * axis[2] * (1.0f - c) + axis[0] * s, 0.0f,
					axis[0] * axis[2] * (1.0f - c) + axis[1] * s, axis[1] * axis[2] * (1.0f - c) - axis[0] * s, axis[2] * axis[2] * (1.0f - c) + c, 0.0f,
					0.0f, 0.0f, 0.0f, 1.0f
				};
				Product(a, b, result);
			}

			void
			RotateInPlace(float *a, const float *axis, float angle)
			{
				float result[16];
				Rotate(a, axis, angle, result);
				Copy(result, a);
			}

			void
			Translate(const float *a, const float *b, float *result)
			{
				const float c[16] =
				{
					1.0f, 0.0f, 0.0f, 0.0f,
					0.0f, 1.0f, 0.0f, 0.0f,
					0.0f, 0.0f, 1.0f, 0.0f,
					b[0], b[1], b[2], 1.0f
				};
				Product(a, c, result);
			}

			void
			TranslateInPlace(float *a, const float *b)
			{
				float result[16];
				Translate(a, b, result);
				Copy(result, a);
			}

			void
			LookAt(const float *eye, const float *center, const float *up, float *result)
			{
				float tmp[3], forward[3];
				Vector::Sub(center, eye, tmp);
				Vector::Normal(tmp, forward);
				Look(eye, forward, up, result);
			}

			void
			Look(const float *position, const float *forward, const float *up, float *result)
			{
				float right[3];
				Vector::CrossProduct(forward, up, right);
				const float rotation[16] =
				{
					right[0], up[0], -forward[0], 0.0f,
					right[1], up[1], -forward[1], 0.0f,
					right[2], up[2], -forward[2], 0.0f,
					0.0f, 0.0f, 0.0f, 1.0f
				};
				Vector::ScalarProduct(position, -1.0f, right);
				Translate(rotation, right, result);
			}

			void
			Rotation(const float *forward, const float *up, float *result)
			{
				unsigned int i;
				float right[3];
				Vector::CrossProduct(forward, up, right);
				for(i = 0; i < 3; i++)
				{
					result[0 * 4 + i] = right[i];
					result[1 * 4 + i] = up[i];
					result[2 * 4 + i] = -forward[i];
					result[3 * 4 + i] = result[i * 4 + 3] = 0.0f;
				}
				result[15] = 1.0f;
			}

			void
			GetPosition(const float *a, float *result)
			{
				result[0] = a[12];
				result[1] = a[13];
				result[2] = a[14];
			}

			void
			GetRight(const float *a, float *result)
			{
				float tmp[3] = {a[0], a[4], a[8]};
				Vector::Normal(tmp, result);
			}

			void
			GetUp(const float *a, float *result)
			{
				float tmp[3] = {a[1], a[5], a[9]};
				Vector::Normal(tmp, result);
			}

			void
			GetForward(const float *a, float *result)
			{
				float tmp[3] = {-a[2], -a[6], -a[10]};
				Vector::Normal(tmp, result);
			}
		}

		namespace Triangle
		{
			void
			Print(const float *a)
			{
				printf("[(%f, %f, %f), (%f, %f, %f), (%f, %f, %f)]", a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
			}

			void
			Copy(const float *a, float *result)
			{
				memcpy(result, a, 3 * 3 * sizeof(float));
			}

			void
			GetBoundSphere(const float *a, float *result)
			{
				result[0] = (a[0 * 3 + 0] + a[1 * 3 + 0] + a[2 * 3 + 0]) / 3.0f;
				result[1] = (a[0 * 3 + 1] + a[1 * 3 + 1] + a[2 * 3 + 1]) / 3.0f;
				result[2] = (a[0 * 3 + 2] + a[1 * 3 + 2] + a[2 * 3 + 2]) / 3.0f;
  				result[3] = sqrt(pow(result[0] - a[0 * 3 + 0], 2) + pow(result[1] - a[0 * 3 + 1], 2) + pow(result[2] - a[0 * 3 + 2], 2));
			}
		}

		namespace Sphere
		{
			void
			Print(const float *a)
			{
				printf("[(%f, %f, %f), %f]\n", a[0], a[1], a[2], a[3]);
			}

			void
			Copy(const float *a, float *result)
			{
				memcpy(result, a, 4 * sizeof(float));
			}

			void
			Load(float *a, float x, float y, float z, float r)
			{
				a[0] = x; a[1] = y; a[2] = z; a[3] = r;
			}

			void
			Merge(const float *a, const float *b, float *result)
			{
				float distancer[3];
				float distance;
				distancer[0] = b[0] - a[0];
				distancer[1] = b[1] - a[1];
				distancer[2] = b[2] - a[2];
				distance = sqrt(pow(distancer[0], 2) + pow(distancer[1], 2) + pow(distancer[2], 2));
				if(distance + b[3] > a[3])
				{
					if(distance + a[3] > b[3])
					{
						distancer[0] /= distance;
						distancer[1] /= distance;
						distancer[2] /= distance;
						result[3] = (distance + a[3] + b[3]) / 2.0f;
						result[0] = a[0] + distancer[0] * (result[3] - a[3]);
						result[1] = a[1] + distancer[1] * (result[3] - a[3]);
						result[2] = a[2] + distancer[2] * (result[3] - a[3]);
					}
					else Copy(b, result);
				}
				else Copy(a, result);
			}

			void
			MergeInPlace(float *a, const float *b)
			{
				float result[4];
				Merge(a, b, result);
				Copy(result, a);
			}
		}

		namespace Frustum
		{
			bool
			Point(const float *frustum, const float *point)
			{
				unsigned int f;
				for(f = 0; f < 6; f++)
					if(frustum[f * 4 + 0] * point[0] + frustum[f *4 + 1] * point[1] + frustum[f * 4 + 2] * point[2] + frustum[f * 4 + 3] <= 0)
						return false;
				return true;
			}

			bool
			Sphere(const float *frustum, const float *sphere)
			{
				unsigned int p;
				for(p = 0; p < 6; p++)
					if(frustum[p * 4 + 0] * sphere[0] + frustum[p *4 + 1] * sphere[1] + frustum[p *4 + 2] * sphere[2] + frustum[p * 4 + 3] < -sphere[3])
						return true;
				return false;
			}

			bool
			Triangle(const float *frustum, const float *triangle)
			{
				unsigned int f, p;
				for(f = 0; f < 6; f++)
				{
					for(p = 0; p < 3; p++)
						if(frustum[f * 4 + 0] * triangle[p * 3 + 0] + frustum[f * 4 + 1] * triangle[p * 3 + 1] + frustum[f * 4 + 2] * triangle[p * 3 + 2] + frustum[f * 4 + 3] > 0)
							break;
					if(p == 3) return false;
				}
				return true;
			}

			bool
			Cube(const float *frustum, const float *cube)
			{
				unsigned int p;
				for(p = 0; p < 6; p++)
				{
					if(frustum[p * 4 + 0] * (cube[0] - cube[3]) + frustum[p * 4 + 1] * (cube[1] - cube[3]) + frustum[p * 4 + 2] * (cube[2] - cube[3]) + frustum[p * 4 + 3] > 0)
						continue;
					if(frustum[p * 4 + 0] * (cube[0] + cube[3]) + frustum[p * 4 + 1] * (cube[1] - cube[3]) + frustum[p * 4 + 2] * (cube[2] - cube[3]) + frustum[p * 4 + 3] > 0)
						continue;
					if(frustum[p * 4 + 0] * (cube[0] - cube[3]) + frustum[p * 4 + 1] * (cube[1] + cube[3]) + frustum[p * 4 + 2] * (cube[2] - cube[3]) + frustum[p * 4 + 3] > 0)
						continue;
					if(frustum[p * 4 + 0] * (cube[0] + cube[3]) + frustum[p * 4 + 1] * (cube[1] + cube[3]) + frustum[p * 4 + 2] * (cube[2] - cube[3]) + frustum[p * 4 + 3] > 0)
						continue;
					if(frustum[p * 4 + 0] * (cube[0] - cube[3]) + frustum[p * 4 + 1] * (cube[1] - cube[3]) + frustum[p * 4 + 2] * (cube[2] + cube[3]) + frustum[p * 4 + 3] > 0)
						continue;
					if(frustum[p * 4 + 0] * (cube[0] + cube[3]) + frustum[p * 4 + 1] * (cube[1] - cube[3]) + frustum[p * 4 + 2] * (cube[2] + cube[3]) + frustum[p * 4 + 3] > 0)
						continue;
					if(frustum[p * 4 + 0] * (cube[0] - cube[3]) + frustum[p * 4 + 1] * (cube[1] + cube[3]) + frustum[p * 4 + 2] * (cube[2] + cube[3]) + frustum[p * 4 + 3] > 0)
						continue;
					if(frustum[p * 4 + 0] * (cube[0] + cube[3]) + frustum[p * 4 + 1] * (cube[1] + cube[3]) + frustum[p * 4 + 2] * (cube[2] + cube[3]) + frustum[p * 4 + 3] > 0)
						continue;
					return false;
				}
				return true;
			}

			bool
			Polygon(const float *frustum, const float *points, unsigned int npoints)
			{
				unsigned int f, p;
				for(f = 0; f < 6; f++)
				{
					for(p = 0; p < npoints; p++)
						if(frustum[f * 4 + 0] * points[p * 3] + frustum[f * 4 + 1] * points[p * 3 + 1] + frustum[f * 4 + 2] * points[p * 3 + 2] + frustum[f * 4 + 3] > 0)
							break;
					if(p == npoints) return false;
				}
				return true;
			}
		}
	}
}
