// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "H3D/Helper.h"
#include <stdlib.h>
#include <math.h>
#include <GL/glu.h>
#include "H3D/Math.h"
#include "H3D/Image.h"

namespace H3D
{
	namespace Helper
	{
		namespace FreeType
		{
			void
			StringMetrics(FT_Face face, const std::wstring& string, unsigned int *w, unsigned int *h)
			{
				const wchar_t *c;

				*w = *h = 0;
				for(c = &string[0]; *c; c++)
				{
					FT_Load_Char(face, *c, FT_LOAD_DEFAULT);
					*w += face->glyph->metrics.horiAdvance / 64;
					*h = Math::Max(*h, (unsigned int) face->glyph->metrics.height / 64);
				}
			}
		}

		namespace OpenGL
		{
			namespace FreeType
			{
				void
				TexGlyph2D(unsigned int w, unsigned int h, FT_GlyphSlot glyph)
				{
					unsigned int i, j;
					GLubyte *pixels;

					pixels = (GLubyte *) malloc(2 * w * h * sizeof(GLubyte));

					memset(pixels, 0, w * h * sizeof(GLubyte));
					for(j = 0; j < (unsigned int) glyph->bitmap.rows; j++)
        					for(i = 0; i < (unsigned int) glyph->bitmap.width; i++)
								pixels[2 * (j * w + i)] = pixels[2 * (j * w + i) + 1] = glyph->bitmap.buffer[(glyph->bitmap.rows - j - 1) * glyph->bitmap.pitch + i];
					glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, w, h, 0, GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, pixels);

					free(pixels);
				}

				void
				DrawTextureGlyph(FT_GlyphSlot glyph)
				{
					unsigned int w, h;
					float tw, th;

					w = Math::NextTwoPower((unsigned int) glyph->bitmap.width);
					h = Math::NextTwoPower((unsigned int) glyph->bitmap.rows);

					TexGlyph2D(w, h, glyph);

					tw = (float) glyph->bitmap.width / w;
					th = (float) glyph->bitmap.rows / h;

					w = glyph->metrics.width / 64;
					h = glyph->metrics.height / 64;

					glPushMatrix();
					glTranslatef((float) glyph->metrics.horiBearingX / 64, (float) - (glyph->metrics.height - glyph->metrics.horiBearingY) / 64, 0.0);
					glBegin(GL_QUADS);
					glTexCoord2f(0, 0); glVertex2i(0, 0);
					glTexCoord2f(0, th); glVertex2i(0, h);
					glTexCoord2f(tw, th); glVertex2i(w, h);
					glTexCoord2f(tw, 0); glVertex2i(w, 0);
					glEnd();
					glPopMatrix();
				}

				void
				DrawOutlineGlyph(FT_GlyphSlot glyph)
				{
					unsigned int i, j;

					if(glyph->outline.n_contours)
					{
						glBegin(GL_LINE_LOOP);
						for(i = 0, j = 0; i < (unsigned int) glyph->outline.n_points; i++)
						{
							glVertex2i(glyph->outline.points[i].x / 64, glyph->outline.points[i].y / 64);
							if(i == (unsigned int) glyph->outline.contours[j])
							{
								glEnd();
								if(j < (unsigned int) glyph->outline.n_contours - 1)
								{
									j++;
									glBegin(GL_LINE_LOOP);
								}
							}
						}
					}
				}

				void
				DrawWireGlyph(FT_GlyphSlot glyph, unsigned int z)
				{
					unsigned int i, j;

					glPushMatrix();
					glTranslatef(0, 0, - (float) z);
					DrawOutlineGlyph(glyph);
					glPopMatrix();

					if(glyph->outline.n_contours)
					{
						glBegin(GL_LINES);
						for(i = 0, j = 0; i < (unsigned int) glyph->outline.n_points; i++)
						{
							glVertex3i(glyph->outline.points[i].x / 64, glyph->outline.points[i].y / 64, - (int) z);
							glVertex3i(glyph->outline.points[i].x / 64, glyph->outline.points[i].y / 64, 0);
						}
						glEnd();
					}

					DrawOutlineGlyph(glyph);
				}

				void
				DrawFillGlyph(FT_GlyphSlot glyph, GLUtesselator *tess)
				{
					unsigned int i, j;
					GLdouble *vertices;

					if(glyph->outline.n_contours)
					{
						vertices = (GLdouble *) malloc(glyph->outline.n_points * 3 * sizeof(GLdouble));

						gluTessBeginPolygon(tess, NULL);
						gluTessBeginContour(tess);
						for(i = 0, j = 0; i < (unsigned int) glyph->outline.n_points; i++)
						{
							vertices[i * 3 + 0] = (double) glyph->outline.points[i].x / 64;
							vertices[i * 3 + 1] = (double) glyph->outline.points[i].y / 64;
							vertices[i * 3 + 2] = 0;
							gluTessVertex(tess, &vertices[i * 3], &vertices[i * 3]);
							if(i == (unsigned int) glyph->outline.contours[j])
							{
								gluTessEndContour(tess);
								if(j < (unsigned int) glyph->outline.n_contours - 1)
								{
									j++;
									gluTessBeginContour(tess);
								}
							}
						}
						gluTessEndPolygon(tess);

						free(vertices);
					}
				}

				void
				DrawSolidGlyph(FT_GlyphSlot glyph, unsigned int z, GLUtesselator *tess)
				{
					unsigned int i, j;

					glPushMatrix();
					glTranslatef(0, 0, - (float) z);
					DrawFillGlyph(glyph, tess);
					glPopMatrix();

					if(glyph->outline.n_contours)
					{
						glBegin(GL_QUAD_STRIP);
						for(i = 0, j = 0; i < (unsigned int) glyph->outline.n_points; i++)
						{
							glVertex3i(glyph->outline.points[i].x / 64, glyph->outline.points[i].y / 64, - (int) z);
							glVertex3i(glyph->outline.points[i].x / 64, glyph->outline.points[i].y / 64, 0);
							if(i == (unsigned int) glyph->outline.contours[j])
							{
								glEnd();
								if(j < (unsigned int) glyph->outline.n_contours - 1)
								{
									j++;
									glBegin(GL_QUAD_STRIP);
								}
							}
						}
						glEnd();
					}

					DrawFillGlyph(glyph, tess);
				}

				void
				DrawGlyph(FT_GlyphSlot glyph, DrawMode mode, unsigned int z, GLUtesselator *tess)
				{
					switch(mode)
					{
						case TEXTURE: DrawTextureGlyph(glyph); break;
						case OUTLINE: DrawOutlineGlyph(glyph); break;
						case WIRE: DrawWireGlyph(glyph, z); break;
						case FILL: DrawFillGlyph(glyph, tess); break;
						case SOLID: DrawSolidGlyph(glyph, z, tess); break;
					}
				}

				void
				DrawString(FT_Face face, DrawMode mode, const std::wstring& string, unsigned int z, GLUtesselator *tess)
				{
					const wchar_t *c;
					FT_Int32 flags;

					flags = mode == TEXTURE ? FT_LOAD_RENDER : FT_LOAD_DEFAULT;

					glPushMatrix();
					for(c = &string[0]; *c; c++)
					{
						FT_Load_Char(face, *c, flags);
						DrawGlyph(face->glyph, mode, z, tess);
						glTranslatef((float) face->glyph->metrics.horiAdvance / 64, 0, 0);
					}
					glPopMatrix();
				}
			}

			namespace Image
			{
				::H3D::Image::H3D *
				Read(int x, int y, unsigned int width, unsigned int height, GLenum format)
				{
					::H3D::Image::H3D *image = new ::H3D::Image::H3D();
					image->info.width = width;
					image->info.height = height;
					switch(format)
					{
						case GL_RGB: image->info.bpp = 3; break;
						case GL_RGBA: image->info.bpp = 4; break;
					}
					image->content.pixels = new unsigned char[image->info.width * image->info.height * image->info.bpp];
					glReadPixels(x, y, image->info.width, image->info.height, format, GL_UNSIGNED_BYTE, image->content.pixels);
					return image;

				}
			}

			namespace OpenDE
			{
				void
				Matrix(const float *position, const float *rotation, float *matrix)
				{
 					matrix[0] = rotation[0]; matrix[4] = rotation[1]; matrix[8] = rotation[2]; matrix[12] = position[0];
					matrix[1] = rotation[4]; matrix[5] = rotation[5]; matrix[9] = rotation[6]; matrix[13] = position[1];
					matrix[2] = rotation[8]; matrix[6] = rotation[9]; matrix[10] = rotation[10]; matrix[14] = position[2];
					matrix[3] = 0.0f; matrix[7] = 0.0f; matrix[11] = 0.0f; matrix[15] = 1.0f;
				}
			}

			namespace Frustum
			{
				void
				Load(float *frustum)
				{
					float proj[16], modl[16], clip[16], t;

					glGetFloatv(GL_PROJECTION_MATRIX, proj);
					glGetFloatv(GL_MODELVIEW_MATRIX, modl);

					/* Combine the two matrices (multiply projection by modelview) */
					clip[ 0] = modl[ 0] * proj[ 0] + modl[ 1] * proj[ 4] + modl[ 2] * proj[ 8] + modl[ 3] * proj[12];
					clip[ 1] = modl[ 0] * proj[ 1] + modl[ 1] * proj[ 5] + modl[ 2] * proj[ 9] + modl[ 3] * proj[13];
					clip[ 2] = modl[ 0] * proj[ 2] + modl[ 1] * proj[ 6] + modl[ 2] * proj[10] + modl[ 3] * proj[14];
					clip[ 3] = modl[ 0] * proj[ 3] + modl[ 1] * proj[ 7] + modl[ 2] * proj[11] + modl[ 3] * proj[15];

					clip[ 4] = modl[ 4] * proj[ 0] + modl[ 5] * proj[ 4] + modl[ 6] * proj[ 8] + modl[ 7] * proj[12];
					clip[ 5] = modl[ 4] * proj[ 1] + modl[ 5] * proj[ 5] + modl[ 6] * proj[ 9] + modl[ 7] * proj[13];
					clip[ 6] = modl[ 4] * proj[ 2] + modl[ 5] * proj[ 6] + modl[ 6] * proj[10] + modl[ 7] * proj[14];
					clip[ 7] = modl[ 4] * proj[ 3] + modl[ 5] * proj[ 7] + modl[ 6] * proj[11] + modl[ 7] * proj[15];

					clip[ 8] = modl[ 8] * proj[ 0] + modl[ 9] * proj[ 4] + modl[10] * proj[ 8] + modl[11] * proj[12];
					clip[ 9] = modl[ 8] * proj[ 1] + modl[ 9] * proj[ 5] + modl[10] * proj[ 9] + modl[11] * proj[13];
					clip[10] = modl[ 8] * proj[ 2] + modl[ 9] * proj[ 6] + modl[10] * proj[10] + modl[11] * proj[14];
					clip[11] = modl[ 8] * proj[ 3] + modl[ 9] * proj[ 7] + modl[10] * proj[11] + modl[11] * proj[15];

					clip[12] = modl[12] * proj[ 0] + modl[13] * proj[ 4] + modl[14] * proj[ 8] + modl[15] * proj[12];
					clip[13] = modl[12] * proj[ 1] + modl[13] * proj[ 5] + modl[14] * proj[ 9] + modl[15] * proj[13];
					clip[14] = modl[12] * proj[ 2] + modl[13] * proj[ 6] + modl[14] * proj[10] + modl[15] * proj[14];
					clip[15] = modl[12] * proj[ 3] + modl[13] * proj[ 7] + modl[14] * proj[11] + modl[15] * proj[15];

					/* Extract the numbers for the RIGHT plane */
					frustum[0] = clip[ 3] - clip[ 0];
					frustum[1] = clip[ 7] - clip[ 4];
					frustum[2] = clip[11] - clip[ 8];
					frustum[3] = clip[15] - clip[12];

					/* Normalize the result */
					t = sqrt(frustum[0] * frustum[0] + frustum[1] * frustum[1] + frustum[2] * frustum[2]);
					frustum[0] /= t;
					frustum[1] /= t;
					frustum[2] /= t;
					frustum[3] /= t;

					/* Extract the numbers for the LEFT plane */
					frustum[4] = clip[ 3] + clip[ 0];
					frustum[5] = clip[ 7] + clip[ 4];
					frustum[6] = clip[11] + clip[ 8];
					frustum[7] = clip[15] + clip[12];

					/* Normalize the result */
					t = sqrt(frustum[4] * frustum[4] + frustum[5] * frustum[5] + frustum[6] * frustum[6] );
					frustum[4] /= t;
					frustum[5] /= t;
					frustum[6] /= t;
					frustum[7] /= t;

					/* Extract the BOTTOM plane */
					frustum[8] = clip[ 3] + clip[ 1];
					frustum[9] = clip[ 7] + clip[ 5];
					frustum[10] = clip[11] + clip[ 9];
					frustum[11] = clip[15] + clip[13];

					/* Normalize the result */
					t = sqrt(frustum[8] * frustum[8] + frustum[9] * frustum[9] + frustum[10] * frustum[10]);
					frustum[8] /= t;
					frustum[9] /= t;
					frustum[10] /= t;
					frustum[11] /= t;

					/* Extract the TOP plane */
					frustum[12] = clip[ 3] - clip[ 1];
					frustum[13] = clip[ 7] - clip[ 5];
					frustum[14] = clip[11] - clip[ 9];
					frustum[15] = clip[15] - clip[13];

					/* Normalize the result */
					t = sqrt(frustum[12] * frustum[12] + frustum[13] * frustum[13] + frustum[14] * frustum[14]);
					frustum[12] /= t;
					frustum[13] /= t;
					frustum[14] /= t;
					frustum[15] /= t;

					/* Extract the FAR plane */
					frustum[16] = clip[ 3] - clip[ 2];
					frustum[17] = clip[ 7] - clip[ 6];
					frustum[18] = clip[11] - clip[10];
					frustum[19] = clip[15] - clip[14];

					/* Normalize the result */
					t = sqrt(frustum[16] * frustum[16] + frustum[17] * frustum[17] + frustum[18] * frustum[18]);
					frustum[16] /= t;
					frustum[17] /= t;
					frustum[18] /= t;
					frustum[19] /= t;

					/* Extract the NEAR plane */
					frustum[20] = clip[ 3] + clip[ 2];
					frustum[21] = clip[ 7] + clip[ 6];
					frustum[22] = clip[11] + clip[10];
					frustum[23] = clip[15] + clip[14];

					/* Normalize the result */
					t = sqrt(frustum[20] * frustum[20] + frustum[21] * frustum[21] + frustum[22] * frustum[22]);
					frustum[20] /= t;
					frustum[21] /= t;
					frustum[22] /= t;
					frustum[23] /= t;
				}
			}

		}

		namespace OpenDE
		{
			void
			Rotation(const float *matrix, float *rotation)
			{
				rotation[0] = matrix[0]; rotation[1] = matrix[4]; rotation[2] = matrix[8];
				rotation[4] = matrix[1]; rotation[5] = matrix[5]; rotation[6] = matrix[9];
				rotation[8] = matrix[2]; rotation[9] = matrix[6]; rotation[10] = matrix[10];
			}
		}

		namespace Mesh
		{
			H3D::H3D(void)
			{
			}

			H3D::~H3D(void)
			{
			}

			void
			H3D::glLoad(const std::string& path)
			{
				unsigned int i, j;
				textures = (GLuint *) malloc(info.ntextures * sizeof(GLuint));
				glGenTextures(info.ntextures, textures);
				for(i = 0; i < info.ntextures; i++)
				{
					::H3D::Image::H3D* image = new ::H3D::Image::H3D();
					if(!image->load(path + STRPS + content.textures[i].name))
					{
						delete image;
						image = ::H3D::Image::NAME::Load(path + STRPS + content.textures[i].name);
					}
					if(image)
					{
						glBindTexture(GL_TEXTURE_2D, textures[i]);
						glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
						glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
						//glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_NEAREST);
						glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
						glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
						//gluBuild2DMipmaps(GL_TEXTURE_2D, ::H3D::Helper::OpenGL::Image::Format(image->info.bpp), image->info.width, image->info.height, ::H3D::Helper::OpenGL::Image::Format(image->info.bpp), GL_UNSIGNED_BYTE, image->content.pixels);
						glTexImage2D(GL_TEXTURE_2D, 0, ::H3D::Helper::OpenGL::Image::Format(image->info.bpp), image->info.width, image->info.height, 0, ::H3D::Helper::OpenGL::Image::Format(image->info.bpp), GL_UNSIGNED_BYTE, image->content.pixels);
						image->unload();
						delete image;
					}
				}
				
				list = glGenLists(1);
				glNewList(list, GL_COMPILE);
				for(i = 0; i < info.nfaces; i++)
				{
					float ambient[4] = {content.materials[content.faces[i].material].ambient[0], content.materials[content.faces[i].material].ambient[1], content.materials[content.faces[i].material].ambient[2], content.materials[content.faces[i].material].alpha};
					float specular[4] = {content.materials[content.faces[i].material].specular[0], content.materials[content.faces[i].material].specular[1], content.materials[content.faces[i].material].specular[2], content.materials[content.faces[i].material].alpha};
					float diffuse[4] = {content.materials[content.faces[i].material].diffuse[0], content.materials[content.faces[i].material].diffuse[1], content.materials[content.faces[i].material].diffuse[2], content.materials[content.faces[i].material].alpha};
					glBindTexture(GL_TEXTURE_2D, textures[content.materials[content.faces[i].material].texture]);
					glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, ambient);
					glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, diffuse);
					glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular);
					glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, ambient);
					glBegin(GL_TRIANGLES);
					for(j = 0; j < 3; j++)
					{
						glTexCoord2fv(content.faces[i].texcoords[j]);
						glNormal3fv(content.vertices[content.faces[i].vertices[j]].normal);
						glVertex3fv(content.vertices[content.faces[i].vertices[j]].coord);
					}
					glEnd();
				}
				glEndList();
			}

			void
			H3D::glUnload(void)
			{
				glDeleteLists(list, 1);
				glDeleteTextures(info.ntextures, textures);
				free(textures);
			}

			void
			H3D::deLoad(dSpaceID space)
			{
				data = dGeomTriMeshDataCreate();
				dGeomTriMeshDataBuildSingle1(data, content.vertices[0].coord, sizeof(Vertex), info.nvertices, content.faces[0].vertices, info.nfaces * 3, sizeof(Face), content.faces[0].normal);
				geom = dCreateTriMesh(space, data, NULL, NULL, NULL);
			}

			void
			H3D::deUnload(void)
			{
				dGeomDestroy(geom);
				dGeomTriMeshDataDestroy(data);
			}

			void
			H3D::draw(void)
			{
				glCallList(list);
			}
		}
	}
}
