// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#pragma once

#include <H3D/Macros.h>
#include <math.h>

namespace H3D
{
	namespace Math
	{
		const float PI = 3.14159265358979323846f;

		namespace Point
		{
			extern H3D_API const float Zero[3];

			H3D_API
			void
			Print(const float *a);

			H3D_API
			void
			Copy(const float *a, float *result);

			H3D_API
			void
			Load(float *a, float x, float y, float z);

			H3D_API
			void
			Add(const float *a, const float *b, float *result);

			H3D_API
			void
			AddInPlace(float *a, const float *b);

			H3D_API
			void
			Sub(const float *a, const float *b, float *result);

			H3D_API
			void
			SubInPlace(float *a, const float *b);

			H3D_API
			inline
			void
			LoadZero(float *a)
			{
				Load(a, 0.0f, 0.0f, 0.0f);
			}

		}

		namespace Vector
		{
			extern H3D_API const float X[3];
			extern H3D_API const float Y[3];
			extern H3D_API const float Z[3];
			extern H3D_API const float NegX[3];
			extern H3D_API const float NegY[3];
			extern H3D_API const float NegZ[3];

			H3D_API
			void
			Print(const float *a);

			H3D_API
			void
			Copy(const float *a, float *result);

			H3D_API
			void
			Load(float *a, float x, float y, float z);

			H3D_API
			void
			Add(const float *a, const float *b, float *result);

			H3D_API
			void
			AddInPlace(float *a, const float *b);

			H3D_API
			void
			Sub(const float *a, const float *b, float *result);

			H3D_API
			void
			SubInPlace(float *a, const float *b);

			H3D_API
			void
			ScalarProduct(const float *a, float b, float *result);

			H3D_API
			void
			ScalarProductInPlace(float *a, float b);

			H3D_API
			void
			Norm(const float *a, float *result);

			H3D_API
			float
			NormFunc(const float *a);

			H3D_API
			void
			DotProduct(const float *a, const float *b, float *result);

			H3D_API
			float
			DotProductFunc(const float *a, const float *b);

			H3D_API
			void
			CrossProduct(const float *a, const float *b, float *result);

			H3D_API
			void
			CrossProductInPlace(float *a, const float *b);

			H3D_API
			void
			Normal(const float *a, float *result);

			H3D_API
			void
			NormalInPlace(float *a);

			H3D_API
			void
			MatrixProduct(const float *a, const float *b, float *result);

			H3D_API
			void
			MatrixProductInPlace(float *a, const float *b);

			H3D_API
			void
			Rotate(const float *a, const float *axis, float angle, float *result);

			H3D_API
			void
			RotateInPlace(float *a, const float *axis, float angle);

			H3D_API
			void
			Angle(const float *a, const float *b, float *angle);

			H3D_API
			float
			AngleFunc(const float *a, const float *b);

			H3D_API
			inline
			void
			LoadX(float *a)
			{
				Load(a, 1.0f, 0.0f, 0.0f);
			}

			H3D_API
			inline
			void
			LoadY(float *a)
			{
				Load(a, 0.0f, 1.0f, 0.0f);
			}

			H3D_API
			inline
			void
			LoadZ(float *a)
			{
				Load(a, 0.0f, 0.0f, 1.0f);
			}

			H3D_API
			inline
			void
			LoadNegX(float *a)
			{
				Load(a, -1.0f, 0.0f, 0.0f);
			}

			H3D_API
			inline
			void
			LoadNegY(float *a)
			{
				Load(a, 0.0f, -1.0f, 0.0f);
			}

			H3D_API
			inline
			void
			LoadNegZ(float *a)
			{
				Load(a, 0.0f, 0.0f, -1.0f);
			}
		}

		namespace Matrix
		{
			extern H3D_API const float I[16];

			H3D_API
			void
			Print(const float *a);

			H3D_API
			void
			Copy(const float *a, float *result);

			H3D_API
			void
			Load(float *x, float a, float b, float c, float d, float e, float f, float g, float h, float i, float j, float k, float l, float m, float n, float o, float p);

			H3D_API
			void
			LoadIdentity(float *a);

			H3D_API
			void
			Add(const float *a, const float *b, float *result);

			H3D_API
			void
			AddInPlace(float *a, const float *b);

			H3D_API
			void
			Sub(const float *a, const float *b, float *result);

			H3D_API
			void
			SubInPlace(float *a, const float *b);

			H3D_API
			void
			ScalarProduct(const float *a, float b, float *result);

			H3D_API
			void
			ScalarProductInPlace(float *a, float b);

			H3D_API
			void
			Product(const float *a, const float *b, float *result);

			H3D_API
			void
			ProductInPlace(float *a, const float *b);

			H3D_API
			void
			VectorProduct(const float *a, const float *b, float *result);

			H3D_API
			void
			VectorProductInPlace(const float *a, float *b);

			H3D_API
			void
			Transpose(const float *a, float *result);

			H3D_API
			void
			TransposeInPlace(float *a);

			H3D_API
			void
			Scale(const float *a, const float *b, float *result);

			H3D_API
			void
			ScaleInPlace(float *a, const float *b);

			H3D_API
			void
			Rotate(const float *a, const float *axis, float angle, float *result);

			H3D_API
			void
			RotateInPlace(float *a, const float *axis, float angle);

			H3D_API
			void
			Translate(const float *a, const float *b, float *result);

			H3D_API
			void
			TranslateInPlace(float *a, const float *b);

			H3D_API
			void
			LookAt(const float *eye, const float *center, const float *up, float *result);

			H3D_API
			void
			Look(const float *position, const float *forward, const float *up, float *result);

			H3D_API
			void
			Rotation(const float *forward, const float *up, float *result);

			H3D_API
			void
			GetPosition(const float *a, float *result);

			H3D_API
			void
			GetRight(const float *a, float *result);

			H3D_API
			void
			GetUp(const float *a, float *result);

			H3D_API
			void
			GetForward(const float *a, float *result);
		}

		namespace Triangle
		{
			H3D_API
			void
			Print(const float *a);

			H3D_API
			void
			Copy(const float *a, float *result);

			H3D_API
			void
			GetBoundSphere(const float *a, float *result);
		}

		namespace Sphere
		{
			H3D_API
			void
			Print(const float *a);

			H3D_API
			void
			Copy(const float *a, float *result);

			H3D_API
			void
			Load(float *a, float x, float y, float z, float r);

			H3D_API
			void
			Merge(const float *a, const float *b, float *result);

			H3D_API
			void
			MergeInPlace(float *a, const float *b);
		}

		namespace Frustum
		{
			H3D_API
			bool
			Point(const float *frustum, const float *point);

			H3D_API
			bool
			Sphere(const float *frustum, const float *sphere);

			H3D_API
			bool
			Triangle(const float *frustum, const float *triangle);

			H3D_API
			bool
			Cube(const float *frustum, const float *cube);

			H3D_API
			bool
			Polygon(const float *frustum, const float *points, unsigned int npoints);
		}

		H3D_API
		inline
		float
		RadToDeg(float r)
		{
			return r * (180.0f / PI);
		}

		H3D_API
		inline
		float
		DegToRad(float r)
		{
			return r * (PI / 180.0f);
		}

		H3D_API
		inline
		float
		NextTwoPower(float r)
		{
				return pow(2.0f, ceil(log(r) / log(2.0f)));
		}

		H3D_API
		inline
		unsigned int
		NextTwoPower(unsigned int r)
		{
				return (unsigned int) pow(2.0, ceil(log((float) r) / log(2.0)));
		}

		H3D_API
		inline
		float
		Clamp(float x, float a, float b)
		{
			return x < a ? a : (b < x ? b : x);
		}

		H3D_API
		inline
		float
		Max(float a, float b)
		{
			return a > b ? a : b;
		}

		H3D_API
		inline
		unsigned int
		Max(unsigned int a, unsigned int b)
		{
			return a > b ? a : b;
		}
	}
}
