// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "MyMachines.h"
#include <sstream>
#include <X11/keysym.h>
#include "MyTrack.h"
#include "MyMachine.h"
#include "MyBox.h"
#include "MyWidget.h"
#include "config.h"

MyMachines::MyMachines(void)
{
	ALuint buffer;

	srand(H3D::System::Ticks());

	glEnable(GL_COLOR_MATERIAL);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glEnable(GL_TEXTURE_2D);
	glClearColor(0, 0, 0.5, 0);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_BLEND);
	FT_Init_FreeType(&content.freetype);
	FT_New_Face(content.freetype, PACKAGE_DATADIR STRPS "VeraBd.ttf", 0, &content.face);
	FT_Set_Pixel_Sizes(content.face, 64, 64);

	content.tess = gluNewTess();
	gluTessCallback(content.tess, GLU_TESS_VERTEX, (void (*)())&glVertex3dv);
	gluTessCallback(content.tess, GLU_TESS_BEGIN, (void (*)())&glBegin);
	gluTessCallback(content.tess, GLU_TESS_END, (void (*)())&glEnd);

	glGenTextures(1, &content.menu.background);
	glBindTexture(GL_TEXTURE_2D, content.menu.background);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	H3D::Image::H3D *image = H3D::Image::JPEG::import(PACKAGE_DATADIR STRPS "textures" STRPS "Analog-Wind-1.jpg");
	if(image)
	{
		glTexImage2D(GL_TEXTURE_2D, 0, H3D::Helper::OpenGL::Image::Format(image->info.bpp), image->info.width, image->info.height, 0, H3D::Helper::OpenGL::Image::Format(image->info.bpp), GL_UNSIGNED_BYTE, image->content.pixels);
		image->unload();
		delete image;
	}

	content.world = dWorldCreate();
	content.space = dHashSpaceCreate(0);
	content.joints = dJointGroupCreate(0);
	dWorldSetGravity(content.world, 0.0f, -9.8f, 0.0f);

	content.menu.selector = new H3D::Helper::Mesh::H3D();
	content.menu.selector->load(PACKAGE_DATADIR STRPS "wheel");
	content.menu.selector->glLoad(PACKAGE_DATADIR STRPS "textures");
	content.menu.selectoranim = 0;

	alDistanceModel(AL_NONE);
	alGenSources(1, &content.menu.music);
	alGenBuffers(1, &buffer);
	H3D::Sound::H3D* sound = H3D::Sound::VORBIS::import(PACKAGE_DATADIR STRPS "sounds" STRPS "engine.ogg");
	alBufferData(buffer, H3D::Helper::OpenAL::Sound::Format(sound->info.nchannels), sound->content.samples, sound->info.nframes * sound->info.nchannels * sizeof(short), sound->info.rate);
	sound->unload();
	delete sound;
	alSourcei(content.menu.music, AL_BUFFER, buffer);
	alSourcei(content.menu.music, AL_LOOPING, AL_TRUE);
	alSourcei(content.menu.music, AL_SOURCE_RELATIVE, AL_TRUE);

	content.machines[0] = "red";
	content.machines[1] = "green";
	content.machines[2] = "blue";
	content.machines[3] = "yellow";
	content.widgets[0] = "box";
	content.widgets[1] = "turbo";
	content.widgets[2] = "fake";
	content.main.entries[0] = L"Play";
	content.main.entries[1] = L"Options";
	content.main.entries[2] = L"Help";
	content.main.selection = 0;
	content.options.entries[0] = L"Machines";
	content.options.entries[1] = L"Players";
	content.options.entries[2] = L"Items";
	content.options.entries[3] = L"Points";
	content.options.entries[4] = L"Track";
	content.options.entries[5] = L"Machine";
	content.options.tracks[0] = "crayons";
	content.options.tracks[1] = "garden";
	content.options.tracks[2] = "land";
	content.options.tracks[3] = "bathtub";
	content.options.tracks[4] = "square";
	content.options.tracks[5] = "circle";
	content.options.machines[0] = "car";
	content.options.machines[1] = "formula 1";
	content.options.machines[2] = "block";
	content.options.nmachines = 4;
	content.options.nplayers = 1;
	content.options.nitems = 4;
	content.options.track = 0;
	content.options.machine = 0;
	content.options.npoints = 3;
	content.options.selection = 0;
	content.help.entries[0] = L"Drive the machines in the various tracks";
	content.help.entries[1] = L"Items can be box, turbo or fake";
	content.help.entries[2] = L"Player 1 keys: up, down, left, right, return";
	content.help.entries[3] = L"Player 2 keys: w, s, a, d, q";
	content.help.entries[4] = L"Player 3 keys: i, k, j, l, u";
	content.help.entries[5] = L"Player 4 keys: t, g, f, h, t";
	content.help.entries[6] = L"Have fun";
}

MyMachines::~MyMachines(void)
{
	ALuint buffer;

	dJointGroupDestroy(content.joints);
	dSpaceDestroy(content.space);
	dWorldDestroy(content.world);

	content.menu.selector->glUnload();
	content.menu.selector->unload();
	delete content.menu.selector;

	gluDeleteTess(content.tess);
	glDeleteTextures(1, &content.menu.background);

	FT_Done_Face(content.face);
	FT_Done_FreeType(content.freetype);

	alGetSourcei(content.menu.music, AL_BUFFER, (ALint*)&buffer);
	alDeleteSources(1, &content.menu.music);
	alDeleteBuffers(1, &buffer);
}

void
MyMachines::resize(unsigned int width, unsigned int height)
{
	screen.width = width; screen.height = height;
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glViewport(0, 0, screen.width, screen.height);
	gluPerspective(45.0f, (float) screen.width / screen.height, 1.0f, 250.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

void
MyMachines::keyPress(XEvent &event)
{
	switch(state)
	{
		case INTRO: keyPressIntro(event); break;
		case MAIN: keyPressMain(event); break;
		case LOAD: keyPressLoad(event); break;
		case PLAY: keyPressPlay(event); break;
		case SCORE: keyPressScore(event); break;
		case OPTIONS: keyPressOptions(event); break;
		case HELP: keyPressHelp(event); break;
	}
}

void
MyMachines::keyRelease(XEvent &event)
{
	switch(state)
	{
		case INTRO: keyReleaseIntro(event); break;
		case MAIN: keyReleaseMain(event); break;
		case LOAD: keyReleaseLoad(event); break;
		case PLAY: keyReleasePlay(event); break;
		case SCORE: keyReleaseScore(event); break;
		case OPTIONS: keyReleaseOptions(event); break;
		case HELP: keyReleaseHelp(event); break;
	}
}

void
MyMachines::frame(void)
{
	switch(state)
	{
		case INTRO: frameIntro(); break;
		case MAIN: frameMain(); break;
		case LOAD: frameLoad(); break;
		case PLAY: framePlay(); break;
		case SCORE: frameScore(); break;
		case OPTIONS: frameOptions(); break;
		case HELP: frameHelp(); break;
	}
}

void
MyMachines::enter(State state)
{
	switch(state)
	{
		case INTRO: enterIntro(); break;
		case MAIN: enterMain(); break;
		case LOAD: enterLoad(); break;
		case PLAY: enterPlay(); break;
		case SCORE: enterScore(); break;
		case OPTIONS: enterOptions(); break;
		case HELP: enterHelp(); break;
	}
	this->state = state;
}

void
MyMachines::leave(void)
{
	switch(state)
	{
		case INTRO: leaveIntro(); break;
		case MAIN: leaveMain(); break;
		case LOAD: leaveLoad(); break;
		case PLAY: leavePlay(); break;
		case SCORE: leaveScore(); break;
		case OPTIONS: leaveOptions(); break;
		case HELP: leaveHelp(); break;
	}
}

void
MyMachines::change(State state)
{
	leave();
	enter(state);
}

void
MyMachines::enterIntro(void)
{
	content.intro.done = true;
}

void
MyMachines::leaveIntro(void)
{
}

void
MyMachines::keyPressIntro(XEvent &event)
{
}

void
MyMachines::keyReleaseIntro(XEvent &event)
{
}

void
MyMachines::frameIntro(void)
{
	if(content.intro.done) change(MAIN);
}

void
MyMachines::enterMain(void)
{
	alSourcePlay(content.menu.music);
}

void
MyMachines::leaveMain(void)
{
	alSourceStop(content.menu.music);
}

void
MyMachines::keyPressMain(XEvent &event)
{
	KeySym keysym;

	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Down: content.main.selection = (content.main.selection + 1) % 3; break;
		case XK_Up: content.main.selection = (content.main.selection + 2) % 3; break;
	}
}

void
MyMachines::keyReleaseMain(XEvent &event)
{
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Escape: loop = false; break;
		case XK_Return:
			switch(content.main.selection)
			{
				case 0: change(LOAD); break;
				case 1: change(OPTIONS); break;
				case 2: change(HELP); break;
			}
			break;
	}
}

void
MyMachines::frameMain(void)
{
	const float aspect = (float) screen.width / screen.height;
	unsigned int metrics[2], i;
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	gluLookAt(0.0f, 0.0f, 100.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f);
	glBindTexture(GL_TEXTURE_2D, content.menu.background);
	glPushMatrix();
		glTranslatef((float) rand() / RAND_MAX - 0.5f, (float) rand() / RAND_MAX - 0.5f, 0.0f);
		glBegin(GL_QUADS);
			glTexCoord2i(0, 0); glVertex2f(-52.0f * aspect, -52.0f);
			glTexCoord2i(1, 0); glVertex2f(52.0f * aspect, -52.0f);
			glTexCoord2i(1, 1); glVertex2f(52.0f * aspect, 52.0f);
			glTexCoord2i(0, 1); glVertex2f(-52.0f * aspect, 52.0f);
		glEnd();
	glPopMatrix();
	glDisable(GL_TEXTURE_2D);
	for(i = 0; i < 3; i++)
	{
		H3D::Helper::FreeType::StringMetrics(content.face, content.main.entries[i], &metrics[0], &metrics[1]);
		glPushMatrix();
			if(i == content.main.selection) glTranslatef((float) rand() / RAND_MAX - 0.5f, (float) rand() / RAND_MAX - 0.5f, 0.0f);
			glScalef(0.2f, 0.2f, 1.0f);
			glTranslatef(- (float) metrics[0] / 2.0f, (1.0f - i) * 100.0f, 1.0f);
			H3D::Helper::OpenGL::FreeType::DrawString(content.face, H3D::Helper::OpenGL::FreeType::FILL, content.main.entries[i], 1, content.tess);
		glPopMatrix();
		if(i == content.main.selection)
		{
			glEnable(GL_TEXTURE_2D);
			glPushMatrix();
				glTranslatef(- (float) metrics[0] / 10.0f - 10.0f, (1.0f - content.main.selection) * 20.0f + 3.75f, 1.0f);
				glRotatef((float) content.menu.selectoranim, 1.0f, 0.0f, 0.0f);
				glScalef(5.0f, 5.0f, 5.0f);
				content.menu.selector->draw();
			glPopMatrix();
			glPushMatrix();
				glTranslatef((float) metrics[0] / 10.0f + 10.0f, (1.0f - content.main.selection) * 20.0f + 3.75f, 1.0f);
				glRotatef((float) content.menu.selectoranim, 1.0f, 0.0f, 0.0f);
				glScalef(5.0f, 5.0f, 5.0f);
				content.menu.selector->draw();
			glPopMatrix();
			glDisable(GL_TEXTURE_2D);
		}
	}
	glEnable(GL_TEXTURE_2D);
	content.menu.selectoranim = (content.menu.selectoranim + 10) % 360;
}
void
MyMachines::enterLoad(void)
{
	content.load.state = START;
}

void
MyMachines::leaveLoad(void)
{
}

void
MyMachines::keyPressLoad(XEvent &event)
{
}

void
MyMachines::keyReleaseLoad(XEvent &event)
{
}

void
MyMachines::frameLoad(void)
{
	std::wstring text;
	const float aspect = (float) screen.width / screen.height;
	unsigned int metrics[2], i;
	float position[3], sphere[4];

	if(content.load.state != END)
	{
		switch(content.load.state)
		{
			case START:
				text = L"Loading track";
				content.load.state = TRACK;
				break;
			case TRACK:
				content.play.track = new MyTrack();
				content.play.track->load(content.options.tracks[content.options.track], content.space);
				content.load.state = ITEMS;
				text = L"Loading items";
				break;
			case ITEMS:
				for(i = 0; i < content.options.nitems; i++)
				{
					content.play.items[i].box = new MyBox();
					unsigned int checkpoint = ((float) rand() / RAND_MAX) * content.play.track->info.ncheckpoints;
					content.play.items[i].type = (MyMachines::ItemType)(((float) rand() / RAND_MAX) * (sizeof(content.widgets) / sizeof(content.widgets[0])));
					H3D::Math::Point::Copy(&content.play.track->content.checkpoints[checkpoint * 3], sphere);
					sphere[3] = 2.0f;
					content.play.items[i].box->load(sphere);
					switch(content.play.items[i].type)
					{
						case CUBE: content.play.items[i].widget = new MyCube(); break;
						case TURBO: content.play.items[i].widget = new MyTurbo(); break;
						case FAKE: content.play.items[i].widget = new MyFake(); break;
					}
					content.play.items[i].widget->load(content.world, content.space);
					content.play.items[i].widget->disable();
					content.play.items[i].player = -1;
					content.play.items[i].frames = 0;
				}
				content.load.state = MACHINES;
				text = L"Loading machines";
				break;
			case MACHINES:
				for(i = 0; i < content.options.nmachines; i++)
				{
					memset(content.play.players[i].keys, 0, 5 * sizeof(bool));
					content.play.players[i].checkpoint = 0;
					content.play.players[i].distance = 0.0f;
					content.play.players[i].points = 0;
					content.play.players[i].item = -1;
					content.play.players[i].machine = new MyMachine();
					H3D::Math::Point::Load(position, ((i % 4) < 2 ? -5.0f : 5.0f) + ((i & 1) ? 2.5f : -2.5f), 5.0f, float(i / 4) * 7.5f);
					content.play.players[i].machine->load(content.options.machines[content.options.machine] + " " + content.machines[i], content.world, content.space, position);
				}
				content.load.state = END;
				text = L"Ready";
				break;
			default:
				break;
		}

		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glLoadIdentity();
		gluLookAt(0.0, 0.0, 100.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
		glBindTexture(GL_TEXTURE_2D, content.menu.background);
		glPushMatrix();
			glBegin(GL_QUADS);
				glTexCoord2i(0, 0); glVertex2f(-52.0f * aspect, -52.0f);
				glTexCoord2i(1, 0); glVertex2f(52.0f * aspect, -52.0f);
				glTexCoord2i(1, 1); glVertex2f(52.0f * aspect, 52.0f);
				glTexCoord2i(0, 1); glVertex2f(-52.0f * aspect, 52.0f);
			glEnd();
		glPopMatrix();
		glDisable(GL_TEXTURE_2D);
		H3D::Helper::FreeType::StringMetrics(content.face, text, &metrics[0], &metrics[1]);
		glPushMatrix();
			glScalef(0.1f, 0.1f, 1.0f);
			glTranslatef(- (float) metrics[0] / 2.0f, 0.0f, 1.0f);
			H3D::Helper::OpenGL::FreeType::DrawString(content.face, H3D::Helper::OpenGL::FreeType::FILL, text, 1, content.tess);
		glPopMatrix();
		glEnable(GL_TEXTURE_2D);
	}
	else change(PLAY);
}

void
MyMachines::enterPlay(void)
{
	float listener[6] = {0.0f, -1.0f, 0.0f, 0.0f, 0.0f, -1.0f};

	H3D::Math::Point::LoadZero(content.play.camera);

	alListener3f(AL_POSITION, content.play.camera[0], 0.0f, content.play.camera[2]);
	alListenerfv(AL_ORIENTATION, listener);
}

void
MyMachines::leavePlay(void)
{
	for(unsigned int i = 0; i < content.options.nitems; i++)
	{
		content.play.items[i].box->unload();
		delete content.play.items[i].box;
		content.play.items[i].widget->unload();
		delete content.play.items[i].widget;
	}

	for(unsigned int i = 0; i < content.options.nmachines; i++)
	{
		content.play.players[i].machine->unload();
		delete content.play.players[i].machine;
	}

	content.play.track->unload();
	delete content.play.track;
}

void
MyMachines::keyPressPlay(XEvent &event)
{
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Up: content.play.players[0].keys[0] = true; break;
		case XK_Down: content.play.players[0].keys[1] = true; break;
		case XK_Left: content.play.players[0].keys[2] = true; break;
		case XK_Right: content.play.players[0].keys[3] = true; break;
		case XK_Return: content.play.players[0].keys[4] = true; break;
		case XK_w: content.play.players[1].keys[0] = true; break;
		case XK_s: content.play.players[1].keys[1] = true; break;
		case XK_a: content.play.players[1].keys[2] = true; break;
		case XK_d: content.play.players[1].keys[3] = true; break;
		case XK_q: content.play.players[1].keys[4] = true; break;
		case XK_i: content.play.players[2].keys[0] = true; break;
		case XK_k: content.play.players[2].keys[1] = true; break;
		case XK_j: content.play.players[2].keys[2] = true; break;
		case XK_l: content.play.players[2].keys[3] = true; break;
		case XK_u: content.play.players[2].keys[4] = true; break;
		case XK_t: content.play.players[3].keys[0] = true; break;
		case XK_g: content.play.players[3].keys[1] = true; break;
		case XK_f: content.play.players[3].keys[2] = true; break;
		case XK_h: content.play.players[3].keys[3] = true; break;
		case XK_r: content.play.players[3].keys[4] = true; break;
	}
}

void
MyMachines::keyReleasePlay(XEvent &event)
{
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Escape: change(MAIN); break;
		case XK_Up: content.play.players[0].keys[0] = false; break;
		case XK_Down: content.play.players[0].keys[1] = false; break;
		case XK_Left: content.play.players[0].keys[2] = false; break;
		case XK_Right: content.play.players[0].keys[3] = false; break;
		case XK_Return: content.play.players[0].keys[4] = false; break;
		case XK_w: content.play.players[1].keys[0] = false; break;
		case XK_s: content.play.players[1].keys[1] = false; break;
		case XK_a: content.play.players[1].keys[2] = false; break;
		case XK_d: content.play.players[1].keys[3] = false; break;
		case XK_q: content.play.players[1].keys[4] = false; break;
		case XK_i: content.play.players[2].keys[0] = false; break;
		case XK_k: content.play.players[2].keys[1] = false; break;
		case XK_j: content.play.players[2].keys[2] = false; break;
		case XK_l: content.play.players[2].keys[3] = false; break;
		case XK_u: content.play.players[2].keys[4] = false; break;
		case XK_t: content.play.players[3].keys[0] = false; break;
		case XK_g: content.play.players[3].keys[1] = false; break;
		case XK_f: content.play.players[3].keys[2] = false; break;
		case XK_h: content.play.players[3].keys[3] = false; break;
		case XK_r: content.play.players[3].keys[4] = false; break;
	}
}

void
MyMachines::framePlay(void)
{
	unsigned int i, first, actives;
	const float* v;

	for(actives = 0, i = 0; i < content.options.nmachines; i++)
		if(dBodyIsEnabled(content.play.players[i].machine->body[0]))
		{
			actives++;
			if(i < content.options.nplayers) human(i);
			else computer(i);
		}

	for(i = 0; i < 16; i++)
	{
		dSpaceCollide(content.space, this, &MyMachines::NearCallback);
		dWorldStep(content.world, 0.01f);
		dJointGroupEmpty(content.joints);
	}

	for(first = 0, i = 0; i < content.options.nmachines; i++)
	{
		if(dBodyIsEnabled(content.play.players[i].machine->body[0]) && (content.play.players[i].checkpoint > content.play.players[first].checkpoint || (content.play.players[i].checkpoint == content.play.players[first].checkpoint && content.play.players[i].distance < content.play.players[first].distance)))
			first = i;
	}

	if(actives > 1)
	{
		const float *p;
		float a[3], b[3];
		p = dBodyGetPosition(content.play.players[first].machine->body[0]);
		H3D::Math::Vector::Sub(p, content.play.camera, a);
		H3D::Math::Vector::ScalarProduct(a, 0.15f, b);
		H3D::Math::Vector::AddInPlace(content.play.camera, b);
	}
	else
	{
		float position[3], rotation[16], target[3], forward[3];
		content.play.players[first].points++;
		H3D::Math::Point::Copy(&content.play.track->content.checkpoints[content.play.players[first].checkpoint % (content.play.track->info.ncheckpoints)* 3], content.play.camera);
		H3D::Math::Point::Copy(&content.play.track->content.checkpoints[(content.play.players[first].checkpoint + 1) % (content.play.track->info.ncheckpoints) * 3], target);
		H3D::Math::Vector::Sub(target, content.play.camera, forward);
		H3D::Math::Vector::NormalInPlace(forward);
		H3D::Math::Matrix::Rotation(forward, H3D::Math::Vector::Y, rotation);
		for(i = 0; i < content.options.nmachines; i++)
		{
			if(!dBodyIsEnabled(content.play.players[i].machine->body[0]))
				content.play.players[i].machine->enable();
			content.play.players[i].checkpoint = content.play.players[first].checkpoint;
			H3D::Math::Point::Load(position, ((i % 4) < 2 ? -5.0f : 5.0f) + ((i & 1) ? 2.5f : -2.5f), 5.0f, float(i / 4) * 7.5f);
			H3D::Math::Vector::MatrixProductInPlace(position, rotation);
			H3D::Math::Point::AddInPlace(position, content.play.camera);
			content.play.players[i].machine->place(position, rotation);
		}
	}

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	gluLookAt(content.play.camera[0], 70.0f, content.play.camera[2], content.play.camera[0], content.play.camera[1], content.play.camera[2], 0.0f, 0.0f, -1.0f);
	H3D::Helper::OpenGL::Frustum::Load(content.play.frustum);
	content.play.track->frame();
	for(i = 0; i < content.options.nmachines; i++)
	{
		if(dBodyIsEnabled(content.play.players[i].machine->body[0]))
		{
			if(H3D::Math::Frustum::Point(content.play.frustum, dBodyGetPosition(content.play.players[i].machine->body[0])))
				content.play.players[i].machine->frame();
			else
			{
				if(content.play.players[i].item != -1)
				{
					unsigned int checkpoint = ((float) rand() / RAND_MAX) * content.play.track->info.ncheckpoints;
					H3D::Math::Point::Copy(&content.play.track->content.checkpoints[checkpoint * 3], content.play.items[content.play.players[i].item].box->sphere);
					content.play.items[content.play.players[i].item].player = -1;
					content.play.players[i].item = -1;
				}
				content.play.players[i].machine->disable();
				if(actives == content.options.nmachines && content.play.players[i].points) content.play.players[i].points--;
			}
		}
	}

	for(i = 0; i < content.options.nitems; i++)
	{
		if(content.play.items[i].frames > 0)
		{
			if(content.play.items[i].player == -1)
				content.play.items[i].widget->frame();
			content.play.items[i].frames--;
			if(content.play.items[i].frames == 0)
			{
				if(content.play.items[i].player != -1)
				{
					content.play.players[content.play.items[i].player].item = -1;
					content.play.items[i].player = -1;
				}
				else
					content.play.items[i].widget->disable();
				unsigned int checkpoint = ((float) rand() / RAND_MAX) * content.play.track->info.ncheckpoints;
				H3D::Math::Point::Copy(&content.play.track->content.checkpoints[checkpoint * 3], content.play.items[i].box->sphere);
			}
		}
		else
			content.play.items[i].box->frame();
	}

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0f, screen.width, 0.0f, screen.height);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	DrawScore(content.face, H3D::Helper::OpenGL::FreeType::FILL, 1, content.tess);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45.0f, (float) screen.width / screen.height, 1.0f, 250.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	alListener3f(AL_POSITION, content.play.camera[0], 0, content.play.camera[2]);
	v = dBodyGetLinearVel(content.play.players[first].machine->body[0]);
	alListener3f(AL_VELOCITY, v[0], v[1], v[2]);

	if(content.play.players[first].points == content.options.npoints)
	{
		content.score.winner = first;
		change(SCORE);
	}
}

void
MyMachines::enterScore(void)
{
	alSourcePlay(content.menu.music);
}

void
MyMachines::leaveScore(void)
{
	alSourceStop(content.menu.music);
}

void
MyMachines::keyPressScore(XEvent &event)
{
}

void
MyMachines::keyReleaseScore(XEvent &event)
{
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Escape: change(MAIN); break;
	}
}

void
MyMachines::frameScore(void)
{
	const float aspect = (float) screen.width / screen.height;
	unsigned int metrics[2];
	std::wstringstream text;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	gluLookAt(0.0f, 0.0f, 100.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f);
	glBindTexture(GL_TEXTURE_2D, content.menu.background);
	glPushMatrix();
		glTranslatef((float) rand() / RAND_MAX - 0.5f, (float) rand() / RAND_MAX - 0.5f, 0.0f);
		glBegin(GL_QUADS);
			glTexCoord2i(0, 0); glVertex2f(-52.0f * aspect, -52.0f);
			glTexCoord2i(1, 0); glVertex2f(52.0f * aspect, -52.0f);
			glTexCoord2i(1, 1); glVertex2f(52.0f * aspect, 52.0f);
			glTexCoord2i(0, 1); glVertex2f(-52.0f * aspect, 52.0f);
		glEnd();
	glPopMatrix();

	text << L"Player " << content.score.winner + 1 << L" wins";

	glDisable(GL_TEXTURE_2D);
	H3D::Helper::FreeType::StringMetrics(content.face, text.str(), &metrics[0], &metrics[1]);
	glPushAttrib(GL_CURRENT_BIT);
	glPushMatrix();
	glScalef(0.1f, 0.1f, 1.0f);
	glTranslatef((float) rand() / RAND_MAX * 20.0f - 10.0f, (float) rand() / RAND_MAX * 20.0f - 10.0f, 0.0f);
	glTranslatef(- (float) metrics[0] / 2.0f, 0.0f, 1.0f);
	switch(content.score.winner)
	{
		case 0: glColor3f(1, 0, 0); break;
		case 1: glColor3f(0, 1, 0); break;
		case 2: glColor3f(0, 0, 1); break;
		case 3: glColor3f(1, 1, 0); break;
		default: break;
	}
	H3D::Helper::OpenGL::FreeType::DrawString(content.face, H3D::Helper::OpenGL::FreeType::FILL, text.str(), 1, content.tess);
	glPopMatrix();
	glPopAttrib();
	glEnable(GL_TEXTURE_2D);
}

void
MyMachines::enterOptions(void)
{
	alSourcePlay(content.menu.music);
}

void
MyMachines::leaveOptions(void)
{
	alSourceStop(content.menu.music);
}

void
MyMachines::keyPressOptions(XEvent &event)
{
	const unsigned int ntracks = sizeof(content.options.tracks) / sizeof(content.options.tracks[0]);
	const unsigned int nmachines = sizeof(content.options.machines) / sizeof(content.options.machines[0]);
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Left:
			switch(content.options.selection)
			{
				case 0:
					if(content.options.nmachines > 2) content.options.nmachines--;
					if(content.options.nplayers > content.options.nmachines) content.options.nplayers = content.options.nmachines;
					break;
				case 1: if(content.options.nplayers > 1) content.options.nplayers--; break;
				case 2: if(content.options.nitems > 0) content.options.nitems--; break;
				case 3: if(content.options.npoints > 1) content.options.npoints--; break;
				case 4: content.options.track = (content.options.track + (ntracks - 1)) % ntracks; break;
				case 5: content.options.machine = (content.options.machine + (nmachines - 1)) % nmachines; break;
			}
			break;
		case XK_Right:
			switch(content.options.selection)
			{
				case 0: if(content.options.nmachines < 4) content.options.nmachines++; break;
				case 1: if(content.options.nplayers < content.options.nmachines) content.options.nplayers++; break;
				case 2: if(content.options.nitems < 9) content.options.nitems++; break;
				case 3: if(content.options.npoints < 9)content.options.npoints++; break;
				case 4: content.options.track = (content.options.track + 1) % ntracks; break;
				case 5: content.options.machine = (content.options.machine + 1) % nmachines; break;
			}
			break;
		case XK_Up: content.options.selection = (content.options.selection + 5) % 6; break;
		case XK_Down: content.options.selection = (content.options.selection + 1) % 6;	break;
	}
}

void
MyMachines::keyReleaseOptions(XEvent &event)
{
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Escape: change(MAIN); break;
	}
}

void
MyMachines::frameOptions(void)
{
	const float aspect = (float) screen.width / screen.height;
	unsigned int metrics[2], i;
	wchar_t text[128];
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	gluLookAt(0.0, 0.0, 100.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
	glBindTexture(GL_TEXTURE_2D, content.menu.background);
	glPushMatrix();
		glTranslatef((float) rand() / RAND_MAX - 0.5f, (float) rand() / RAND_MAX - 0.5f, 0.0f);
		glBegin(GL_QUADS);
			glTexCoord2i(0, 0); glVertex2f(-52.0f * aspect, -52.0f);
			glTexCoord2i(1, 0); glVertex2f(52.0f * aspect, -52.0f);
			glTexCoord2i(1, 1); glVertex2f(52.0f * aspect, 52.0f);
			glTexCoord2i(0, 1); glVertex2f(-52.0f * aspect, 52.0f);
		glEnd();
	glPopMatrix();
	glDisable(GL_TEXTURE_2D);
	for(i = 0; i < 6; i++)
	{
		switch(i)
		{
			case 0: swprintf(text, 128, L"%ls: %i", content.options.entries[i].c_str(), content.options.nmachines); break;
			case 1: swprintf(text, 128, L"%ls: %i", content.options.entries[i].c_str(), content.options.nplayers); break;
			case 2: swprintf(text, 128, L"%ls: %i", content.options.entries[i].c_str(), content.options.nitems); break;
			case 3: swprintf(text, 128, L"%ls: %i", content.options.entries[i].c_str(), content.options.npoints); break;
			case 4: swprintf(text, 128, L"%ls: %hs", content.options.entries[i].c_str(), content.options.tracks[content.options.track].c_str()); break;
			case 5: swprintf(text, 128, L"%ls: %hs", content.options.entries[i].c_str(), content.options.machines[content.options.machine].c_str()); break;
		}
		H3D::Helper::FreeType::StringMetrics(content.face, text, &metrics[0], &metrics[1]);
		glPushMatrix();
			if(i == content.options.selection) glTranslatef((float) rand() / RAND_MAX - 0.5f, (float) rand() / RAND_MAX - 0.5f, 0.0f);
			glScalef(0.1f, 0.1f, 1.0f);
			glTranslatef(- (float) metrics[0] / 2.0f, (2.5f - i) * 80.0f, 1.0f);
			H3D::Helper::OpenGL::FreeType::DrawString(content.face, H3D::Helper::OpenGL::FreeType::FILL, text, 1, content.tess);
		glPopMatrix();
		if(i == content.options.selection)
		{
			glEnable(GL_TEXTURE_2D);
			glPushMatrix();
				glTranslatef(- (float) metrics[0] / 20.0f - 8.0f, (2.5f - content.options.selection) * 8.0f + 2.0f, 1.0f);
				glRotatef((float) content.menu.selectoranim, 1.0f, 0.0f, 0.0f);
				glScalef(5.0f, 5.0f, 5.0f);
				content.menu.selector->draw();
			glPopMatrix();
			glPushMatrix();
				glTranslatef((float) metrics[0] / 20.0f + 8.0f, (2.5f - content.options.selection) * 8.0f + 2.0f, 1.0f);
				glRotatef((float) content.menu.selectoranim, 1.0f, 0.0f, 0.0f);
				glScalef(5.0f, 5.0f, 5.0f);
				content.menu.selector->draw();
			glPopMatrix();
			glDisable(GL_TEXTURE_2D);
		}
	}
	glEnable(GL_TEXTURE_2D);
	content.menu.selectoranim = (content.menu.selectoranim + 10) % 360;
}

void
MyMachines::enterHelp(void)
{
	alSourcePlay(content.menu.music);
}

void
MyMachines::leaveHelp(void)
{
	alSourceStop(content.menu.music);
}

void
MyMachines::keyPressHelp(XEvent &event)
{
}

void
MyMachines::keyReleaseHelp(XEvent &event)
{
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Escape: change(MAIN); break;
	}
}

void
MyMachines::frameHelp(void)
{
	const float aspect = (float) screen.width / screen.height;
	unsigned int metrics[2], i;
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	gluLookAt(0.0f, 0.0f, 100.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f);
	glBindTexture(GL_TEXTURE_2D, content.menu.background);
	glPushMatrix();
		glTranslatef((float) rand() / RAND_MAX - 0.5f, (float) rand() / RAND_MAX - 0.5f, 0.0f);
		glBegin(GL_QUADS);
			glTexCoord2i(0, 0); glVertex2f(-52.0f * aspect, -52.0f);
			glTexCoord2i(1, 0); glVertex2f(52.0f * aspect, -52.0f);
			glTexCoord2i(1, 1); glVertex2f(52.0f * aspect, 52.0f);
			glTexCoord2i(0, 1); glVertex2f(-52.0f * aspect, 52.0f);
		glEnd();
	glPopMatrix();
	glDisable(GL_TEXTURE_2D);
	glPushMatrix();
		glScalef(0.05f, 0.05f, 1.0f);
		for(i = 0; i < 7; i++)
		{
			H3D::Helper::FreeType::StringMetrics(content.face, content.help.entries[i], &metrics[0], &metrics[1]);
			glPushMatrix();
				glTranslatef((float) rand() / RAND_MAX * 20.0f - 10.0f, (float) rand() / RAND_MAX * 20.0f - 10.0f, 0.0f);
				glTranslatef(- (float) metrics[0] / 2.0f, (3.0f - i) * 200.0f, 1.0f);
				H3D::Helper::OpenGL::FreeType::DrawString(content.face, H3D::Helper::OpenGL::FreeType::FILL, content.help.entries[i], 1, content.tess);
			glPopMatrix();
		}	
	glPopMatrix();
	glEnable(GL_TEXTURE_2D);
}

void
MyMachines::computer(unsigned int player)
{
	float position[3], target[3], direction[3], orientation[3], item[3], plane[4], angle, variation, distance;
	const float *r;
	int steer, gas;
	bool action, turbo = false;
	unsigned int i;

	variation = (float) rand() / RAND_MAX;

	H3D::Math::Point::Copy(dBodyGetPosition(content.play.players[player].machine->body[0]), position);
	H3D::Math::Point::Copy(&content.play.track->content.checkpoints[(content.play.players[player].checkpoint + 1) % (content.play.track->info.ncheckpoints) * 3], target);
	H3D::Math::Vector::Sub(target, position, direction);
	H3D::Math::Vector::Norm(direction, &content.play.players[player].distance);
	if(content.play.players[player].distance < 20.0f)
	{
		content.play.players[player].checkpoint++;
		H3D::Math::Point::Copy(&content.play.track->content.checkpoints[(content.play.players[player].checkpoint + 1) % (content.play.track->info.ncheckpoints) * 3], target);
		H3D::Math::Vector::Sub(target, position, direction);
		H3D::Math::Vector::Norm(direction, &content.play.players[player].distance);
	}
	H3D::Math::Vector::NormalInPlace(direction);
	r = dBodyGetRotation(content.play.players[player].machine->body[0]);
	orientation[0] = -r[2]; orientation[1] = -r[6]; orientation[2] = -r[10];
	H3D::Math::Vector::CrossProduct(H3D::Math::Vector::Y, orientation, plane);
	plane[3] = -H3D::Math::Vector::DotProductFunc(plane, position);
	distance = H3D::Math::Vector::DotProductFunc(plane, target) + plane[3];
	H3D::Math::Vector::Angle(orientation, direction, &angle);
	if(distance > 10.0f * (0.25f + variation * 0.75f)) steer = 1;
	else if(distance < -10.0f * (0.25 + variation * 0.75f)) steer = -1;
	else steer = 0;
	if(angle > 75.0f)
	{
		const float *v;
		v = dBodyGetLinearVel(content.play.players[player].machine->body[0]);
		if(content.play.players[player].distance < 20.0f * 2.0f)
		{
			if(H3D::Math::Vector::NormFunc(v) > 2.0f) gas = 0;
			else gas = 1;
		}
		else { gas = -1; steer *= -1; }
	}
	else gas = 1;
	action = variation < 0.05;

	if(content.play.players[player].item == -1)
	{
		for(i = 0; i < content.options.nitems; i++)
		{
			if(content.play.items[i].frames == 0)
			{
				H3D::Math::Point::Sub(position, content.play.items[i].box->sphere, item);
				if(H3D::Math::Vector::NormFunc(item) <= content.play.items[i].box->sphere[3])
				{
					content.play.players[player].item = i;
					content.play.items[i].player = player;
					switch(content.play.items[i].type)
					{
						case CUBE: content.play.items[i].frames = 800; break;
						case TURBO: content.play.items[i].frames = 100; break;
						case FAKE: content.play.items[i].frames = 200; break;
					}
				}
			}
		}
	}
	else
	{
		if(action)
		{
			switch(content.play.items[content.play.players[player].item].type)
			{
				case CUBE:
				{
					H3D::Math::Vector::ScalarProductInPlace(orientation, 3.0f);
					H3D::Math::Point::Sub(position, orientation, item);
					content.play.items[content.play.players[player].item].widget->place(item);
					content.play.items[content.play.players[player].item].widget->enable();
					content.play.items[content.play.players[player].item].player = -1;
					content.play.players[player].item = -1;
				}
					break;
				case TURBO:
					turbo = true;
					break;
				case FAKE:
					break;
			}
		}
	}

	content.play.players[player].machine->drive(gas == 1, gas == -1, steer == 1, steer == -1, turbo);
}

void
MyMachines::human(unsigned int player)
{
	float position[3], target[3], direction[3], item[3];
	unsigned int i;
	bool turbo = false;

	H3D::Math::Point::Copy(dBodyGetPosition(content.play.players[player].machine->body[0]), position);
	H3D::Math::Point::Copy(&content.play.track->content.checkpoints[(content.play.players[player].checkpoint + 1) % (content.play.track->info.ncheckpoints) * 3], target);
	H3D::Math::Vector::Sub(target, position, direction);
	H3D::Math::Vector::Norm(direction, &content.play.players[player].distance);
	if(content.play.players[player].distance < 40.0f)
	{
		content.play.players[player].checkpoint++;
		H3D::Math::Point::Copy(&content.play.track->content.checkpoints[(content.play.players[player].checkpoint + 1) % (content.play.track->info.ncheckpoints) * 3], target);
		H3D::Math::Vector::Sub(target, position, direction);
		H3D::Math::Vector::Norm(direction, &content.play.players[player].distance);
	}

	if(content.play.players[player].item == -1)
	{
		for(i = 0; i < content.options.nitems; i++)
		{
			if(content.play.items[i].frames == 0)
			{
				H3D::Math::Point::Sub(position, content.play.items[i].box->sphere, item);
				if(H3D::Math::Vector::NormFunc(item) <= content.play.items[i].box->sphere[3])
				{
					content.play.players[player].item = i;
					content.play.items[i].player = player;
					switch(content.play.items[i].type)
					{
						case CUBE: content.play.items[i].frames = 800; break;
						case TURBO: content.play.items[i].frames = 100; break;
						case FAKE: content.play.items[i].frames = 200; break;
					}
				}
			}
		}
	}
	else
	{
		if(content.play.players[player].keys[4])
		{
			switch(content.play.items[content.play.players[player].item].type)
			{
				case CUBE:
				{
					const float *r;
					r = dBodyGetRotation(content.play.players[player].machine->body[0]);
					direction[0] = -r[2]; direction[1] = -r[6]; direction[2] = -r[10];
					H3D::Math::Vector::ScalarProductInPlace(direction, 3.0f);
					H3D::Math::Point::Sub(position, direction, item);
					content.play.items[content.play.players[player].item].widget->place(item);
					content.play.items[content.play.players[player].item].widget->enable();
					content.play.items[content.play.players[player].item].player = -1;
					content.play.players[player].item = -1;
				}
					break;
				case TURBO:
					turbo = true;
					break;
				case FAKE:
					break;
			}
		}
	}

	content.play.players[player].machine->drive(content.play.players[player].keys[0], content.play.players[player].keys[1], content.play.players[player].keys[2], content.play.players[player].keys[3], turbo);
}

void
MyMachines::DrawScore(FT_Face face, H3D::Helper::OpenGL::FreeType::DrawMode mode, unsigned int z, GLUtesselator *tess)
{
	FT_Int32 flags;
	unsigned int i;

	flags = mode == H3D::Helper::OpenGL::FreeType::TEXTURE ? FT_LOAD_RENDER : FT_LOAD_DEFAULT;
	glPushMatrix();
		for(i = 0; i < content.options.nmachines; i++)
		{
			std::wstringstream score;
			score << content.play.players[i].points;
			FT_Load_Char(face, score.str()[0], flags);
			glPushAttrib(GL_CURRENT_BIT);
			glDisable(GL_TEXTURE_2D);
				switch(i)
				{
					case 0: glColor3f(1.0f, 0.0f, 0.0f); break;
					case 1: glColor3f(0.0f, 1.0f, 0.0f); break;
					case 2: glColor3f(0.0f, 0.0f, 1.0f); break;
					case 3: glColor3f(1.0f, 1.0f, 0.0f); break;
					default: break;
				}
				H3D::Helper::OpenGL::FreeType::DrawGlyph(face->glyph, mode, z, tess);
			glEnable(GL_TEXTURE_2D);
			glPopAttrib();
			glTranslatef((float) face->glyph->metrics.horiAdvance / 64, 0.0f, 0.0f);
			if(content.play.players[i].item != -1)
			{
				glPushMatrix();
					glTranslatef(-(float) face->glyph->metrics.horiAdvance / 128, (float) face->glyph->metrics.vertAdvance / 64, 0.0f);
					glScalef(10.0f, 10.0f, 1.0f);
					content.play.items[content.play.players[i].item].widget->draw();
				glPopMatrix();
			}
		}
	glPopMatrix();
}

void
MyMachines::NearCallback(void *data, dGeomID o1, dGeomID o2)
{
	MyMachines* game;
	const int ncontacts = 1024;
	unsigned int i;
	int n;
	dBodyID b1, b2;
	dContact contacts[ncontacts];
	dJointID c;
	game = (MyMachines*) data;
	b1 = dGeomGetBody(o1);
	b2 = dGeomGetBody(o2);
	if(b1 && b2 && dAreConnectedExcluding(b1, b2, dJointTypeContact))
		return;
	n = dCollide(o1, o2, ncontacts, &contacts[0].geom, sizeof(dContact));
	if(n > 0)
	{
		for(i = 0; i < (unsigned int) n; i++)
		{
			contacts[i].surface.mode = dContactSlip1 | dContactSlip2 | dContactSoftERP | dContactSoftCFM | dContactApprox1 | dContactMu2;
			contacts[i].surface.mu = 0.75f;
			contacts[i].surface.mu2 = 0.75f;
			contacts[i].surface.slip1 = 0.01f;
			contacts[i].surface.slip2 = 0.01f;
			contacts[i].surface.soft_erp = 0.5f;
			contacts[i].surface.soft_cfm = 0.3f;
			c = dJointCreateContact(game->content.world, game->content.joints, &contacts[i]);
			dJointAttach(c, dGeomGetBody(contacts[i].geom.g1), dGeomGetBody(contacts[i].geom.g2));
		}
	}
}
