#include "HMachines.h"
#include <sstream>
#include <X11/keysym.h>
#include "config.h"

HCheckPoints::HCheckPoints(void)
{
	info.magic = 0;
}

HCheckPoints::~HCheckPoints(void)
{
}

bool
HCheckPoints::load(const std::string& filename, bool compiled)
{
	bool result;
	FILE *stream;
	if(compiled)
	{
		stream = fopen(filename.c_str(), "rb");
		if(stream)
		{
			fread(&info, 1, sizeof(Info), stream);
			content.checkpoints = (float*) malloc(info.ncheckpoints * 3 * sizeof(float));
			fread(content.checkpoints, 1, info.ncheckpoints * 3 * sizeof(float), stream);
			fclose(stream);
			result = true;
		}
		else result = false;
	}
	else
	{
		stream = fopen(filename.c_str(), "r");
		if(stream)
		{
			char buffer[256];
			unsigned int i;
			fgets(buffer, 256, stream);
			//sscanf(buffer, "checkpoints %i", &info.ncheckpoints);
			sscanf(buffer, "length %i", &info.ncheckpoints);
			content.checkpoints = (float*) malloc(info.ncheckpoints * 3 * sizeof(float));
			for(i = 0; i < info.ncheckpoints; i++)
			{
				fgets(buffer, 256, stream);
				//sscanf(buffer, "coord %f %f %f", &content.checkpoints[i * 3], &content.checkpoints[i * 3 + 1], &content.checkpoints[i * 3 + 2]);
				sscanf(buffer, "(%f, %f, %f)", &content.checkpoints[i * 3], &content.checkpoints[i * 3 + 1], &content.checkpoints[i * 3 + 2]);
			}
			fclose(stream);
			result = true;
		}
		else result = false;
	}
	return result;
}

void
HCheckPoints::unload(void)
{
	free(content.checkpoints);
}

bool
HCheckPoints::save(const std::string& filename, bool compiled)
{
	bool result;
	FILE *stream;
	if(compiled)
	{
		stream = fopen(filename.c_str(), "wb");
		if(stream)
		{
			fwrite(&info, 1, sizeof(Info),stream);
			fwrite(content.checkpoints, 1, info.ncheckpoints * 3 * sizeof(float), stream);
			fclose(stream);
			result = true;
		}
		else result = false;
	}
	else
	{
		stream = fopen(filename.c_str(), "w");
		if(stream)
		{
			unsigned int i;
			fprintf(stream, "checkpoints %i\n", info.ncheckpoints);
			for(i = 0; i < info.ncheckpoints; i++)
				fprintf(stream, "coord %f %f %f\n", content.checkpoints[i * 3], content.checkpoints[i * 3 + 1], content.checkpoints[i * 3 + 2]);
			fclose(stream);
			result = true;
		}
		else result = false;
		result = false;
	}
	return result;
}

HTrack::HTrack(void)
{
	mesh = new H3D::Helper::Mesh::H3D();
}

HTrack::~HTrack(void)
{
	delete mesh;
}

bool
HTrack::load(const std::string& track, dSpaceID space)
{

	mesh->load(PACKAGE_DATADIR STRPS + track);
	mesh->glLoad(PACKAGE_DATADIR STRPS "textures");
	mesh->deLoad(space);
	HCheckPoints::load(PACKAGE_DATADIR STRPS "checkpoints" STRPS + track);
	//HCheckPoints::load("data" STRPS "ckp" STRPS + track + ".ckp", false);
	//HCheckPoints::save("data" STRPS "ckp" STRPS + track + ".txt", false);
	//HCheckPoints::save("data" STRPS "ckp" STRPS + track);
	return true;
}

void
HTrack::unload(void)
{
	mesh->deUnload();
	mesh->glUnload();
	mesh->unload();
}

void
HTrack::frame(void)
{
	mesh->draw();
}

HMachine::HMachine(void)
{
	wheel = new H3D::Helper::Mesh::H3D();
	chassis = new H3D::Helper::Mesh::H3D();
}

HMachine::~HMachine(void)
{
	delete chassis;
	delete wheel;
}

bool
HMachine::load(const std::string& machine, dWorldID world, dSpaceID space, const float* position)
{
	unsigned int i;
	dMass mass;
	dQuaternion q;
	const float *anchor;
	ALuint buffer;

	info.speed = 70.0f;
	info.steer = 20.0f;

	body[0] = dBodyCreate(world);
	dBodySetPosition(body[0], position[0], position[1], position[2]);
	dMassSetBox(&mass, 1.0f, 2.5f, 1.0f, 4.0f);
	dMassAdjust(&mass, 1.0f);
	dBodySetMass(body[0], &mass);
	geom[0] = dCreateBox(space, 2.5f, 1.0f, 5.5f);
	dGeomSetBody(geom[0], body[0]);

	dQFromAxisAndAngle(q, 0.0f, 1.0f, 0.0f, M_PI * 0.5f);
	dMassSetSphere(&mass, 0.05f, 0.25f);
	dMassAdjust(&mass, 0.05f);
	for(i = 1; i < 5; i++)
	{
		body[i] = dBodyCreate(world);
		dBodySetQuaternion(body[i], q);
		dBodySetMass(body[i], &mass);
		geom[i] = dCreateSphere(space, 0.25f);
		dGeomSetBody(geom[i], body[i]);
	}
	dBodySetPosition(body[1], position[0] - 1.5f, position[1] - 0.5f, position[2] + 1.25f);
	dBodySetPosition(body[2], position[0] + 1.5f, position[1] - 0.5f, position[2] + 1.25f);
	dBodySetPosition(body[3], position[0] - 1.5f, position[1] - 0.5f, position[2] - 1.5f);
	dBodySetPosition(body[4], position[0] + 1.5f, position[1] - 0.5f, position[2] - 1.5f);

	for(i = 0; i < 4; i++)
	{
		joint[i] = dJointCreateHinge2(world, NULL);
		dJointAttach(joint[i], body[0], body[i + 1]);
		anchor = dBodyGetPosition(body[i + 1]);
		dJointSetHinge2Anchor(joint[i], anchor[0], anchor[1], anchor[2]);
		dJointSetHinge2Axis1(joint[i], 0.0f, 1.0f, 0.0f);
		dJointSetHinge2Axis2(joint[i], 1.0f, 0.0f, 0.0f);
		dJointSetHinge2Param(joint[i], dParamSuspensionERP, 0.4f);
		dJointSetHinge2Param(joint[i], dParamSuspensionCFM, 0.8f);
	}
	for(i = 0; i < 2; i++)
	{
		dJointSetHinge2Param(joint[i], dParamLoStop, 0);
		dJointSetHinge2Param(joint[i], dParamHiStop, 0);
	}

	chassis->load(PACKAGE_DATADIR STRPS + machine);
	chassis->glLoad(PACKAGE_DATADIR STRPS "textures");
	wheel->load(PACKAGE_DATADIR STRPS "wheel");
	wheel->glLoad(PACKAGE_DATADIR STRPS "textures");

	alGenSources(1, &engine);
	alGenBuffers(1, &buffer);
	H3D::Sound::H3D* sound = H3D::Sound::VORBIS::import(PACKAGE_DATADIR STRPS "sounds" STRPS "engine.ogg");
	alBufferData(buffer, H3D::Helper::OpenAL::Sound::Format(sound->info.nchannels), sound->content.samples, sound->info.nframes * sound->info.nchannels * sizeof(short), sound->info.rate);
	sound->unload();
	delete sound;
	alSourcei(engine, AL_BUFFER, buffer);
	alSourcei(engine, AL_LOOPING, AL_TRUE);
	alSource3f(engine, AL_POSITION, position[0], position[1], position[2]);
	alSourcePlay(engine);

	return true;
}

void
HMachine::unload(void)
{
	ALuint buffer;
	unsigned int i;

	alGetSourcei(engine, AL_BUFFER, (int*)&buffer);
	alDeleteSources(1, &engine);
	alDeleteBuffers(1, &buffer);

	wheel->glUnload();
	wheel->unload();
	chassis->glUnload();
	chassis->unload();

	for(i = 0; i < 4; i++) dJointDestroy(joint[i]);
	for(i = 0; i < 5; i++) {dGeomDestroy(geom[i]); dBodyDestroy(body[i]);}
}

void
HMachine::frame(void)
{
	H3D::Helper::Mesh::H3D *mesh;
	const float *p, *r;
	float m[16], v;
	unsigned int i;

	v = 0;
	for(i = 0; i < 4; i++)
		v += H3D::Math::Vector::NormFunc(dBodyGetAngularVel(body[i + 1]));
	alSourcef(engine, AL_PITCH, H3D::Math::Clamp(v / 80.0f, 0.0f, 1.5f) + 0.5f);

	p = dBodyGetPosition(body[0]);
	alSource3f(engine, AL_POSITION, p[0], p[1], p[2]);
	p = dBodyGetLinearVel(body[0]);
	alSource3f(engine, AL_VELOCITY, p[0], p[1], p[2]);
	r = dBodyGetRotation(body[0]);
	alSource3f(engine, AL_DIRECTION, -r[2], -r[6], -r[10]);

	for(i = 0; i < 5; i++)
	{
		mesh = i ? wheel : chassis;
		p = dBodyGetPosition(body[i]);
        r = dBodyGetRotation(body[i]);
		H3D::Helper::OpenGL::OpenDE::Matrix(p, r, m);
		glPushMatrix();
			glMultMatrixf(m);
			mesh->draw();
		glPopMatrix();
	}
}

void
HMachine::enable(void)
{
	alSourcePlay(engine);
	for(unsigned int i = 0; i < 5; i++)
	{
		dBodyEnable(body[i]);
		dGeomEnable(geom[i]);
	}
}

void
HMachine::disable(void)
{
	for(unsigned int i = 0; i < 5; i++)
	{
		dGeomDisable(geom[i]);
		dBodyDisable(body[i]);
	}
	alSourceStop(engine);
}

void
HMachine::drive(bool up, bool down, bool left, bool right)
{
	unsigned int i;
	const float speed = info.speed * (up ? 1.0f : (down ? -1.0f : 0.0f));
	const float steer = info.steer * (left ? -1.0f : (right ? 1.0f : 0.0f));
	if(steer)
	{
		for(i = 2; i < 4; i++)
		{
			dJointSetHinge2Param(joint[i], dParamVel, steer);
			dJointSetHinge2Param(joint[i], dParamFMax, 4.0f);
			dJointSetHinge2Param(joint[i], dParamLoStop, -0.75f);
			dJointSetHinge2Param(joint[i], dParamHiStop, 0.75f);
			dJointSetHinge2Param(joint[i], dParamFudgeFactor, 0.1f);
		}
	}
	else
	{
		for(i = 2; i < 4; i++)
		{
			dJointSetHinge2Param(joint[i], dParamLoStop, 0.0f);
			dJointSetHinge2Param(joint[i], dParamHiStop, 0.0f);
			dJointSetHinge2Param(joint[i], dParamFudgeFactor, 0.1f);
		}
	}
	for(i = 0; i < 4; i++)
	{
		dJointSetHinge2Param(joint[i], dParamVel2, speed);
		dJointSetHinge2Param(joint[i], dParamFMax2, 2.0f);
	}
	for(i = 0; i < 2; i++)
	{
		dJointSetHinge2Param(joint[i], dParamLoStop, 0.0f);
		dJointSetHinge2Param(joint[i], dParamHiStop, 0.0f);
		dJointSetHinge2Param(joint[i], dParamFudgeFactor, 0.01f);
	}
}

void
HMachine::place(const float *position, const float *rotation)
{
	float matrix[12], wposition[3];

	H3D::Helper::OpenDE::Rotation(rotation, matrix);
	dBodySetPosition(body[0], position[0], position[1], position[2]);
	dBodySetRotation(body[0], matrix);
	dBodySetLinearVel(body[0], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(body[0], 0.0f, 0.0f, 0.0f);

	H3D::Math::Vector::Load(wposition, -0.5f, - 0.5f, 1.25f);
	H3D::Math::Vector::MatrixProductInPlace(wposition, rotation);
	H3D::Math::Vector::AddInPlace(wposition, position);
	dBodySetPosition(body[1], wposition[0], wposition[1], wposition[2]);
	dBodySetRotation(body[1], matrix);
	dBodySetLinearVel(body[1], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(body[1], 0.0f, 0.0f, 0.0f);

	H3D::Math::Vector::Load(wposition, 1.5f, -0.5f, 1.25f);
	H3D::Math::Vector::MatrixProductInPlace(wposition, rotation);
	H3D::Math::Vector::AddInPlace(wposition, position);
	dBodySetPosition(body[2], wposition[0], wposition[1], wposition[2]);
	dBodySetRotation(body[2], matrix);
	dBodySetLinearVel(body[2], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(body[2], 0.0f, 0.0f, 0.0f);

	H3D::Math::Vector::Load(wposition, -1.5f, -0.5f, -1.5f);
	H3D::Math::Vector::MatrixProductInPlace(wposition, rotation);
	H3D::Math::Vector::AddInPlace(wposition, position);
	dBodySetPosition(body[3], wposition[0], wposition[1], wposition[2]);
	dBodySetRotation(body[3], matrix);
	dBodySetLinearVel(body[3], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(body[3], 0.0f, 0.0f, 0.0f);

	H3D::Math::Vector::Load(wposition, 1.5f, -0.5f, -1.5f);
	H3D::Math::Vector::MatrixProductInPlace(wposition, rotation);
	H3D::Math::Vector::AddInPlace(wposition, position);
	dBodySetPosition(body[4], wposition[0], wposition[1], wposition[2]);
	dBodySetRotation(body[4], matrix);
	dBodySetLinearVel(body[4], 0.0f, 0.0f, 0.0f);
	dBodySetAngularVel(body[4], 0.0f, 0.0f, 0.0f);
}

HMachines::HMachines(void)
{
	ALuint buffer;

	srand(H3D::System::Ticks());

	glEnable(GL_COLOR_MATERIAL);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glEnable(GL_TEXTURE_2D);
	glClearColor(0, 0, 0.5, 0);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_BLEND);
	FT_Init_FreeType(&content.freetype);
	FT_New_Face(content.freetype, PACKAGE_DATADIR STRPS "VeraBd.ttf", 0, &content.face);
	FT_Set_Pixel_Sizes(content.face, 64, 64);

	content.tess = gluNewTess();
	gluTessCallback(content.tess, GLU_TESS_VERTEX, (void (*)())&glVertex3dv);
	gluTessCallback(content.tess, GLU_TESS_BEGIN, (void (*)())&glBegin);
	gluTessCallback(content.tess, GLU_TESS_END, (void (*)())&glEnd);

	glGenTextures(1, &content.menu.background);
	glBindTexture(GL_TEXTURE_2D, content.menu.background);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	H3D::Image::H3D *image = H3D::Image::JPEG::import(PACKAGE_DATADIR STRPS "textures" STRPS "Analog-Wind-1.jpg");
	if(image)
	{
		glTexImage2D(GL_TEXTURE_2D, 0, H3D::Helper::OpenGL::Image::Format(image->info.bpp), image->info.width, image->info.height, 0, H3D::Helper::OpenGL::Image::Format(image->info.bpp), GL_UNSIGNED_BYTE, image->content.pixels);
		image->unload();
		delete image;
	}

	content.world = dWorldCreate();
	content.space = dHashSpaceCreate(0);
	content.joints = dJointGroupCreate(0);
	dWorldSetGravity(content.world, 0.0f, -9.8f, 0.0f);

	content.menu.selector = new H3D::Helper::Mesh::H3D();
	content.menu.selector->load(PACKAGE_DATADIR STRPS "wheel");
	content.menu.selector->glLoad(PACKAGE_DATADIR STRPS "textures");
	content.menu.selectoranim = 0;

	alDistanceModel(AL_NONE);
	alGenSources(1, &content.menu.music);
	alGenBuffers(1, &buffer);
	H3D::Sound::H3D* sound = H3D::Sound::VORBIS::import(PACKAGE_DATADIR STRPS "sounds" STRPS "engine.ogg");
	alBufferData(buffer, H3D::Helper::OpenAL::Sound::Format(sound->info.nchannels), sound->content.samples, sound->info.nframes * sound->info.nchannels * sizeof(short), sound->info.rate);
	sound->unload();
	delete sound;
	alSourcei(content.menu.music, AL_BUFFER, buffer);
	alSourcei(content.menu.music, AL_LOOPING, AL_TRUE);
	alSourcei(content.menu.music, AL_SOURCE_RELATIVE, AL_TRUE);

	content.machines[0] = "red";
	content.machines[1] = "green";
	content.machines[2] = "blue";
	content.machines[3] = "yellow";
	content.main.entries[0] = L"Play";
	content.main.entries[1] = L"Options";
	content.main.entries[2] = L"Help";
	content.main.selection = 0;
	content.options.entries[0] = L"Machines";
	content.options.entries[1] = L"Players";
	content.options.entries[2] = L"Track";
	content.options.entries[3] = L"Machine";
	content.options.entries[4] = L"Points";
	content.options.tracks[0] = "crayons";
	content.options.tracks[1] = "garden";
	content.options.tracks[2] = "land";
	content.options.tracks[3] = "bathtub";
	content.options.machines[0] = "car";
	content.options.machines[1] = "formula 1";
	content.options.machines[2] = "block";
	content.options.nmachines = 4;
	content.options.nplayers = 1;
	content.options.track = 0;
	content.options.machine = 0;
	content.options.npoints = 3;
	content.options.selection = 0;

	content.help.entries[0] = L"A free machines game";
	content.help.entries[1] = L"Drive the machines in the various tracks";
	content.help.entries[2] = L"Player 1 keys: up, down, left, right";
	content.help.entries[3] = L"Player 2 keys: w, s, a, d";
	content.help.entries[4] = L"Player 3 keys: i, k, j, l";
	content.help.entries[5] = L"Player 4 keys: t, g, f, h";
	content.help.entries[6] = L"Have fun";
}

HMachines::~HMachines(void)
{
	ALuint buffer;

	dJointGroupDestroy(content.joints);
	dSpaceDestroy(content.space);
	dWorldDestroy(content.world);

	content.menu.selector->glUnload();
	content.menu.selector->unload();
	delete content.menu.selector;

	gluDeleteTess(content.tess);
	glDeleteTextures(1, &content.menu.background);

	FT_Done_Face(content.face);
	FT_Done_FreeType(content.freetype);

	alGetSourcei(content.menu.music, AL_BUFFER, (ALint*)&buffer);
	alDeleteSources(1, &content.menu.music);
	alDeleteBuffers(1, &buffer);
}

void
HMachines::resize(unsigned int width, unsigned int height)
{
	screen.width = width; screen.height = height;
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glViewport(0, 0, screen.width, screen.height);
	gluPerspective(45.0f, (float) screen.width / screen.height, 1.0f, 250.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

void
HMachines::keyPress(XEvent& event)
{
	switch(state)
	{
		case INTRO: keyPressIntro(event); break;
		case MAIN: keyPressMain(event); break;
		case LOAD: keyPressLoad(event); break;
		case PLAY: keyPressPlay(event); break;
		case OPTIONS: keyPressOptions(event); break;
		case HELP: keyPressHelp(event); break;
	}
}

void
HMachines::keyRelease(XEvent& event)
{
	switch(state)
	{
		case INTRO: keyReleaseIntro(event); break;
		case MAIN: keyReleaseMain(event); break;
		case LOAD: keyReleaseLoad(event); break;
		case PLAY: keyReleasePlay(event); break;
		case OPTIONS: keyReleaseOptions(event); break;
		case HELP: keyReleaseHelp(event); break;
	}
}

void
HMachines::frame(void)
{
	switch(state)
	{
		case INTRO: frameIntro(); break;
		case MAIN: frameMain(); break;
		case LOAD: frameLoad(); break;
		case PLAY: framePlay(); break;
		case OPTIONS: frameOptions(); break;
		case HELP: frameHelp(); break;
	}
}

void
HMachines::enter(State state)
{
	switch(state)
	{
		case INTRO: enterIntro(); break;
		case MAIN: enterMain(); break;
		case LOAD: enterLoad(); break;
		case PLAY: enterPlay(); break;
		case OPTIONS: enterOptions(); break;
		case HELP: enterHelp(); break;
	}
	this->state = state;
}

void
HMachines::leave(void)
{
	switch(state)
	{
		case INTRO: leaveIntro(); break;
		case MAIN: leaveMain(); break;
		case LOAD: leaveLoad(); break;
		case PLAY: leavePlay(); break;
		case OPTIONS: leaveOptions(); break;
		case HELP: leaveHelp(); break;
	}
}

void
HMachines::change(State state)
{
	leave();
	enter(state);
}

void
HMachines::enterIntro(void)
{
	content.intro.done = true;
}

void
HMachines::leaveIntro(void)
{
}

void
HMachines::keyPressIntro(XEvent& event)
{
}

void
HMachines::keyReleaseIntro(XEvent& event)
{
}

void
HMachines::frameIntro(void)
{
	if(content.intro.done) change(MAIN);
}

void
HMachines::enterMain(void)
{
	alSourcePlay(content.menu.music);
}

void
HMachines::leaveMain(void)
{
	alSourceStop(content.menu.music);
}

void
HMachines::keyPressMain(XEvent& event)
{
	KeySym keysym;

	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Down: content.main.selection = (content.main.selection + 1) % 3; break;
		case XK_Up: content.main.selection = (content.main.selection + 2) % 3; break;
	}
}

void
HMachines::keyReleaseMain(XEvent& event)
{
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Escape: loop = false; break;
		case XK_Return:
			switch(content.main.selection)
			{
				case 0: change(LOAD); break;
				case 1: change(OPTIONS); break;
				case 2: change(HELP); break;
			}
			break;
	}
}

void
HMachines::frameMain(void)
{
	const float aspect = (float) screen.width / screen.height;
	unsigned int metrics[2], i;
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	gluLookAt(0.0f, 0.0f, 100.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f);
	glBindTexture(GL_TEXTURE_2D, content.menu.background);
	glPushMatrix();
		glTranslatef((float) rand() / RAND_MAX - 0.5f, (float) rand() / RAND_MAX - 0.5f, 0.0f);
		glBegin(GL_QUADS);
			glTexCoord2i(0, 0); glVertex2f(-52.0f * aspect, -52.0f);
			glTexCoord2i(1, 0); glVertex2f(52.0f * aspect, -52.0f);
			glTexCoord2i(1, 1); glVertex2f(52.0f * aspect, 52.0f);
			glTexCoord2i(0, 1); glVertex2f(-52.0f * aspect, 52.0f);
		glEnd();
	glPopMatrix();
	glDisable(GL_TEXTURE_2D);
	for(i = 0; i < 3; i++)
	{
		H3D::Helper::FreeType::StringMetrics(content.face, content.main.entries[i], &metrics[0], &metrics[1]);
		glPushMatrix();
			if(i == content.main.selection) glTranslatef((float) rand() / RAND_MAX - 0.5f, (float) rand() / RAND_MAX - 0.5f, 0.0f);
			glScalef(0.2f, 0.2f, 1.0f);
			glTranslatef(- (float) metrics[0] / 2.0f, (1.0f - i) * 100.0f, 1.0f);
			H3D::Helper::OpenGL::FreeType::DrawString(content.face, H3D::Helper::OpenGL::FreeType::FILL, content.main.entries[i], 1, content.tess);
		glPopMatrix();
		if(i == content.main.selection)
		{
			glEnable(GL_TEXTURE_2D);
			glPushMatrix();
				glTranslatef(- (float) metrics[0] / 10.0f - 10.0f, (1.0f - content.main.selection) * 20.0f + 3.75f, 1.0f);
				glRotatef((float) content.menu.selectoranim, 1.0f, 0.0f, 0.0f);
				glScalef(5.0f, 5.0f, 5.0f);
				content.menu.selector->draw();
			glPopMatrix();
			glPushMatrix();
				glTranslatef((float) metrics[0] / 10.0f + 10.0f, (1.0f - content.main.selection) * 20.0f + 3.75f, 1.0f);
				glRotatef((float) content.menu.selectoranim, 1.0f, 0.0f, 0.0f);
				glScalef(5.0f, 5.0f, 5.0f);
				content.menu.selector->draw();
			glPopMatrix();
			glDisable(GL_TEXTURE_2D);
		}
	}
	glEnable(GL_TEXTURE_2D);
	content.menu.selectoranim = (content.menu.selectoranim + 10) % 360;
}
void
HMachines::enterLoad(void)
{
	content.load.state = START;
}

void
HMachines::leaveLoad(void)
{
}

void
HMachines::keyPressLoad(XEvent& event)
{
}

void
HMachines::keyReleaseLoad(XEvent& event)
{
}

void
HMachines::frameLoad(void)
{
	std::wstring text;
	const float aspect = (float) screen.width / screen.height;
	unsigned int metrics[2];
	float position[3];

	if(content.load.state != END)
	{
		switch(content.load.state)
		{
			case START:
				text = L"Loading track";
				content.load.state = TRACK;
				break;
			case TRACK:
				content.play.track = new HTrack();
				content.play.track->load(content.options.tracks[content.options.track], content.space);
				content.load.state = MACHINES;
				text = L"Loading machines";
				break;
			case MACHINES:
				for(unsigned int i = 0; i < content.options.nmachines; i++)
				{
					memset(content.play.players[i].keys, 0, 4 * sizeof(bool));
					content.play.players[i].checkpoint = 0;
					content.play.players[i].distance = 0.0f;
					content.play.players[i].points = 0;
					content.play.players[i].machine = new HMachine();
					H3D::Math::Point::Load(position, ((i % 4) < 2 ? -5.0f : 5.0f) + ((i & 1) ? 2.5f : -2.5f), 5.0f, float(i / 4) * 7.5f);
					content.play.players[i].machine->load(content.options.machines[content.options.machine] + " " + content.machines[i], content.world, content.space, position);
				}
				content.load.state = END;
				text = L"Ready";
				break;
			default:
				break;
		}

		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glLoadIdentity();
		gluLookAt(0.0, 0.0, 100.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
		glBindTexture(GL_TEXTURE_2D, content.menu.background);
		glPushMatrix();
			glBegin(GL_QUADS);
				glTexCoord2i(0, 0); glVertex2f(-52.0f * aspect, -52.0f);
				glTexCoord2i(1, 0); glVertex2f(52.0f * aspect, -52.0f);
				glTexCoord2i(1, 1); glVertex2f(52.0f * aspect, 52.0f);
				glTexCoord2i(0, 1); glVertex2f(-52.0f * aspect, 52.0f);
			glEnd();
		glPopMatrix();
		glDisable(GL_TEXTURE_2D);
		H3D::Helper::FreeType::StringMetrics(content.face, text, &metrics[0], &metrics[1]);
		glPushMatrix();
			glScalef(0.1f, 0.1f, 1.0f);
			glTranslatef(- (float) metrics[0] / 2.0f, 0.0f, 1.0f);
			H3D::Helper::OpenGL::FreeType::DrawString(content.face, H3D::Helper::OpenGL::FreeType::FILL, text, 1, content.tess);
		glPopMatrix();
		glEnable(GL_TEXTURE_2D);
	}
	else change(PLAY);
}

void
HMachines::enterPlay(void)
{
	float listener[6] = {0.0f, -1.0f, 0.0f, 0.0f, 0.0f, -1.0f};

	H3D::Math::Point::LoadZero(content.play.camera);

	alListener3f(AL_POSITION, content.play.camera[0], 0.0f, content.play.camera[2]);
	alListenerfv(AL_ORIENTATION, listener);
}

void
HMachines::leavePlay(void)
{
	for(unsigned int i = 0; i < content.options.nmachines; i++)
	{
		content.play.players[i].machine->unload();
		delete content.play.players[i].machine;
	}

	content.play.track->unload();
	delete content.play.track;
}

void
HMachines::keyPressPlay(XEvent& event)
{
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Up: content.play.players[0].keys[0] = true; break;
		case XK_Down: content.play.players[0].keys[1] = true; break;
		case XK_Left: content.play.players[0].keys[2] = true; break;
		case XK_Right: content.play.players[0].keys[3] = true; break;
		case XK_w: content.play.players[1].keys[0] = true; break;
		case XK_s: content.play.players[1].keys[1] = true; break;
		case XK_a: content.play.players[1].keys[2] = true; break;
		case XK_d: content.play.players[1].keys[3] = true; break;
		case XK_i: content.play.players[2].keys[0] = true; break;
		case XK_k: content.play.players[2].keys[1] = true; break;
		case XK_j: content.play.players[2].keys[2] = true; break;
		case XK_l: content.play.players[2].keys[3] = true; break;
		case XK_t: content.play.players[3].keys[0] = true; break;
		case XK_g: content.play.players[3].keys[1] = true; break;
		case XK_f: content.play.players[3].keys[2] = true; break;
		case XK_h: content.play.players[3].keys[3] = true; break;
	}
}

void
HMachines::keyReleasePlay(XEvent& event)
{
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Escape: change(MAIN); break;
		case XK_Up: content.play.players[0].keys[0] = false; break;
		case XK_Down: content.play.players[0].keys[1] = false; break;
		case XK_Left: content.play.players[0].keys[2] = false; break;
		case XK_Right: content.play.players[0].keys[3] = false; break;
		case XK_w: content.play.players[1].keys[0] = false; break;
		case XK_s: content.play.players[1].keys[1] = false; break;
		case XK_a: content.play.players[1].keys[2] = false; break;
		case XK_d: content.play.players[1].keys[3] = false; break;
		case XK_i: content.play.players[2].keys[0] = false; break;
		case XK_k: content.play.players[2].keys[1] = false; break;
		case XK_j: content.play.players[2].keys[2] = false; break;
		case XK_l: content.play.players[2].keys[3] = false; break;
		case XK_t: content.play.players[3].keys[0] = false; break;
		case XK_g: content.play.players[3].keys[1] = false; break;
		case XK_f: content.play.players[3].keys[2] = false; break;
		case XK_h: content.play.players[3].keys[3] = false; break;
	}
}

void
HMachines::framePlay(void)
{
	unsigned int i, first, actives;
	const float* v;
	std::wstringstream text;
	unsigned int metrics[2];

	for(actives = 0, i = 0; i < content.options.nmachines; i++)
		if(dBodyIsEnabled(content.play.players[i].machine->body[0]))
		{
			actives++;
			if(i < content.options.nplayers) human(i);
			else computer(i);
		}

	for(i = 0; i < 16; i++)
	{
		dSpaceCollide(content.space, this, &HMachines::NearCallback);
		dWorldStep(content.world, 0.01f);
		dJointGroupEmpty(content.joints);
	}

	for(first = 0, i = 0; i < content.options.nmachines; i++)
	{
		if(dBodyIsEnabled(content.play.players[i].machine->body[0]) && (content.play.players[i].checkpoint > content.play.players[first].checkpoint || (content.play.players[i].checkpoint == content.play.players[first].checkpoint && content.play.players[i].distance < content.play.players[first].distance)))
				first = i;
	}

	if(actives > 1)
	{
		const float *p;
		float a[3], b[3];
		p = dBodyGetPosition(content.play.players[first].machine->body[0]);
		H3D::Math::Vector::Sub(p, content.play.camera, a);
		H3D::Math::Vector::ScalarProduct(a, 0.15f, b);
		H3D::Math::Vector::AddInPlace(content.play.camera, b);
	}
	else
	{
		float position[3], rotation[16], target[3], forward[3];
		content.play.players[first].points++;
		H3D::Math::Point::Copy(&content.play.track->content.checkpoints[content.play.players[first].checkpoint % (content.play.track->info.ncheckpoints)* 3], content.play.camera);
		H3D::Math::Point::Copy(&content.play.track->content.checkpoints[(content.play.players[first].checkpoint + 1) % (content.play.track->info.ncheckpoints) * 3], target);
		H3D::Math::Vector::Sub(target, content.play.camera, forward);
		H3D::Math::Vector::NormalInPlace(forward);
		H3D::Math::Matrix::Rotation(forward, H3D::Math::Vector::Y, rotation);
		for(i = 0; i < content.options.nmachines; i++)
		{
			if(!dBodyIsEnabled(content.play.players[i].machine->body[0]))
				content.play.players[i].machine->enable();
			content.play.players[i].checkpoint = content.play.players[first].checkpoint;
			H3D::Math::Point::Load(position, ((i % 4) < 2 ? -5.0f : 5.0f) + ((i & 1) ? 2.5f : -2.5f), 5.0f, float(i / 4) * 7.5f);
			H3D::Math::Vector::MatrixProductInPlace(position, rotation);
			H3D::Math::Point::AddInPlace(position, content.play.camera);
			content.play.players[i].machine->place(position, rotation);
		}
	}

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	gluLookAt(content.play.camera[0], 70.0f, content.play.camera[2], content.play.camera[0], content.play.camera[1], content.play.camera[2], 0.0f, 0.0f, -1.0f);
	H3D::Helper::OpenGL::Frustum::Load(content.play.frustum);
	content.play.track->frame();
	for(i = 0; i < content.options.nmachines; i++)
	{
		if(dBodyIsEnabled(content.play.players[i].machine->body[0]))
		{
			if(H3D::Math::Frustum::Point(content.play.frustum, dBodyGetPosition(content.play.players[i].machine->body[0])))
				content.play.players[i].machine->frame();
			else
			{
				content.play.players[i].machine->disable();
				if(actives == content.options.nmachines && content.play.players[i].points) content.play.players[i].points--;
			}
		}
		switch(i)
		{
			case 0: text << L"R"; break;
			case 1: text << L"G"; break;
			case 2: text << L"B"; break;
			case 3: text << L"Y"; break;
		}
		text << content.play.players[i].points << L" ";
	}

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0f, screen.width, 0.0f, screen.height);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glDisable(GL_TEXTURE_2D);
	H3D::Helper::FreeType::StringMetrics(content.face, text.str(), &metrics[0], &metrics[1]);
	glPushMatrix();
		H3D::Helper::OpenGL::FreeType::DrawString(content.face, H3D::Helper::OpenGL::FreeType::OUTLINE, text.str(), 1, content.tess);
	glPopMatrix();
	glEnable(GL_TEXTURE_2D);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(45.0f, (float) screen.width / screen.height, 1.0f, 250.0f);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	alListener3f(AL_POSITION, content.play.camera[0], 0, content.play.camera[2]);
	v = dBodyGetLinearVel(content.play.players[first].machine->body[0]);
	alListener3f(AL_VELOCITY, v[0], v[1], v[2]);

	if(content.play.players[first].points == content.options.npoints) change(MAIN);
}

void
HMachines::enterOptions(void)
{
	alSourcePlay(content.menu.music);
}

void
HMachines::leaveOptions(void)
{
	alSourceStop(content.menu.music);
}

void
HMachines::keyPressOptions(XEvent& event)
{
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Left:
			switch(content.options.selection)
			{
				case 0:
					if(content.options.nmachines > 2) content.options.nmachines--;
					if(content.options.nplayers > content.options.nmachines) content.options.nplayers = content.options.nmachines;
					break;
				case 1: if(content.options.nplayers > 1) content.options.nplayers--; break;
				case 2: content.options.track = (content.options.track + 3) % 4; break;
				case 3: content.options.machine = (content.options.machine + 2) % 3; break;
				case 4: if(content.options.npoints > 1) content.options.npoints--; break;
			}
			break;
		case XK_Right:
			switch(content.options.selection)
			{
				case 0: if(content.options.nmachines < 4) content.options.nmachines++; break;
				case 1: if(content.options.nplayers < content.options.nmachines) content.options.nplayers++; break;
				case 2: content.options.track = (content.options.track + 1) % 4; break;
				case 3: content.options.machine = (content.options.machine + 1) % 3; break;
				case 4: if(content.options.npoints < 9)content.options.npoints++; break;
			}
			break;
		case XK_Up: content.options.selection = (content.options.selection + 4) % 5; break;
		case XK_Down: content.options.selection = (content.options.selection + 1) % 5;	break;
	}
}

void
HMachines::keyReleaseOptions(XEvent& event)
{
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Escape: change(MAIN); break;
	}
}

void
HMachines::frameOptions(void)
{
	const float aspect = (float) screen.width / screen.height;
	unsigned int metrics[2], i;
	wchar_t text[128];
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	gluLookAt(0.0, 0.0, 100.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
	glBindTexture(GL_TEXTURE_2D, content.menu.background);
	glPushMatrix();
		glTranslatef((float) rand() / RAND_MAX - 0.5f, (float) rand() / RAND_MAX - 0.5f, 0.0f);
		glBegin(GL_QUADS);
			glTexCoord2i(0, 0); glVertex2f(-52.0f * aspect, -52.0f);
			glTexCoord2i(1, 0); glVertex2f(52.0f * aspect, -52.0f);
			glTexCoord2i(1, 1); glVertex2f(52.0f * aspect, 52.0f);
			glTexCoord2i(0, 1); glVertex2f(-52.0f * aspect, 52.0f);
		glEnd();
	glPopMatrix();
	glDisable(GL_TEXTURE_2D);
	for(i = 0; i < 5; i++)
	{
		switch(i)
		{
			case 0: swprintf(text, 128, L"%ls: %i", content.options.entries[i].c_str(), content.options.nmachines); break;
			case 1: swprintf(text, 128, L"%ls: %i", content.options.entries[i].c_str(), content.options.nplayers); break;
			case 2: swprintf(text, 128, L"%ls: %hs", content.options.entries[i].c_str(), content.options.tracks[content.options.track].c_str()); break;
			case 3: swprintf(text, 128, L"%ls: %hs", content.options.entries[i].c_str(), content.options.machines[content.options.machine].c_str()); break;
			case 4: swprintf(text, 128, L"%ls: %i", content.options.entries[i].c_str(), content.options.npoints); break;
		}
		H3D::Helper::FreeType::StringMetrics(content.face, text, &metrics[0], &metrics[1]);
		glPushMatrix();
			if(i == content.options.selection) glTranslatef((float) rand() / RAND_MAX - 0.5f, (float) rand() / RAND_MAX - 0.5f, 0.0f);
			glScalef(0.1f, 0.1f, 1.0f);
			glTranslatef(- (float) metrics[0] / 2.0f, (2.0f - i) * 80.0f, 1.0f);
			H3D::Helper::OpenGL::FreeType::DrawString(content.face, H3D::Helper::OpenGL::FreeType::FILL, text, 1, content.tess);
		glPopMatrix();
		if(i == content.options.selection)
		{
			glEnable(GL_TEXTURE_2D);
			glPushMatrix();
				glTranslatef(- (float) metrics[0] / 20.0f - 8.0f, (2.0f - content.options.selection) * 8.0f + 2.0f, 1.0f);
				glRotatef((float) content.menu.selectoranim, 1.0f, 0.0f, 0.0f);
				glScalef(5.0f, 5.0f, 5.0f);
				content.menu.selector->draw();
			glPopMatrix();
			glPushMatrix();
				glTranslatef((float) metrics[0] / 20.0f + 8.0f, (2.0f - content.options.selection) * 8.0f + 2.0f, 1.0f);
				glRotatef((float) content.menu.selectoranim, 1.0f, 0.0f, 0.0f);
				glScalef(5.0f, 5.0f, 5.0f);
				content.menu.selector->draw();
			glPopMatrix();
			glDisable(GL_TEXTURE_2D);
		}
	}
	glEnable(GL_TEXTURE_2D);
	content.menu.selectoranim = (content.menu.selectoranim + 10) % 360;
}

void
HMachines::enterHelp(void)
{
	alSourcePlay(content.menu.music);
}

void
HMachines::leaveHelp(void)
{
	alSourceStop(content.menu.music);
}

void
HMachines::keyPressHelp(XEvent& event)
{
}

void
HMachines::keyReleaseHelp(XEvent& event)
{
	KeySym keysym;
	keysym = XLookupKeysym(&event.xkey, 0);
	switch(keysym)
	{
		case XK_Escape: change(MAIN); break;
	}
}

void
HMachines::frameHelp(void)
{
	const float aspect = (float) screen.width / screen.height;
	unsigned int metrics[2], i;
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	gluLookAt(0.0f, 0.0f, 100.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f);
	glBindTexture(GL_TEXTURE_2D, content.menu.background);
	glPushMatrix();
		glTranslatef((float) rand() / RAND_MAX - 0.5f, (float) rand() / RAND_MAX - 0.5f, 0.0f);
		glBegin(GL_QUADS);
			glTexCoord2i(0, 0); glVertex2f(-52.0f * aspect, -52.0f);
			glTexCoord2i(1, 0); glVertex2f(52.0f * aspect, -52.0f);
			glTexCoord2i(1, 1); glVertex2f(52.0f * aspect, 52.0f);
			glTexCoord2i(0, 1); glVertex2f(-52.0f * aspect, 52.0f);
		glEnd();
	glPopMatrix();
	glDisable(GL_TEXTURE_2D);
	glPushMatrix();
	glScalef(0.05f, 0.05f, 1.0f);
	for(i = 0; i < 7; i++)
	{
		H3D::Helper::FreeType::StringMetrics(content.face, content.help.entries[i], &metrics[0], &metrics[1]);
		glPushMatrix();
			glTranslatef((float) rand() / RAND_MAX * 20.0f - 10.0f, (float) rand() / RAND_MAX * 20.0f - 10.0f, 0.0f);
			glTranslatef(- (float) metrics[0] / 2.0f, (3.0f - i) * 200.0f, 1.0f);
			H3D::Helper::OpenGL::FreeType::DrawString(content.face, H3D::Helper::OpenGL::FreeType::FILL, content.help.entries[i], 1, content.tess);
		glPopMatrix();
	}	
	glPopMatrix();
	glEnable(GL_TEXTURE_2D);
}

void
HMachines::computer(unsigned int player)
{
	float position[3], target[3], direction[3], orientation[3], plane[4], angle, variation, distance;
	const float *r;
	int steer, gas;
	variation = (float) rand() / RAND_MAX;
	H3D::Math::Point::Copy(dBodyGetPosition(content.play.players[player].machine->body[0]), position);
	H3D::Math::Point::Copy(&content.play.track->content.checkpoints[(content.play.players[player].checkpoint + 1) % (content.play.track->info.ncheckpoints) * 3], target);
	H3D::Math::Vector::Sub(target, position, direction);
	H3D::Math::Vector::Norm(direction, &content.play.players[player].distance);
	if(content.play.players[player].distance < 20.0f)
	{
		content.play.players[player].checkpoint++;
		H3D::Math::Point::Copy(&content.play.track->content.checkpoints[(content.play.players[player].checkpoint + 1) % (content.play.track->info.ncheckpoints) * 3], target);
		H3D::Math::Vector::Sub(target, position, direction);
		H3D::Math::Vector::Norm(direction, &content.play.players[player].distance);
	}
	H3D::Math::Vector::NormalInPlace(direction);
	r = dBodyGetRotation(content.play.players[player].machine->body[0]);
	orientation[0] = -r[2]; orientation[1] = -r[6]; orientation[2] = -r[10];
	H3D::Math::Vector::CrossProduct(H3D::Math::Vector::Y, orientation, plane);
	plane[3] = -H3D::Math::Vector::DotProductFunc(plane, position);
	distance = H3D::Math::Vector::DotProductFunc(plane, target) + plane[3];
	H3D::Math::Vector::Angle(orientation, direction, &angle);
	if(distance > 10.0f * (0.25f + variation * 0.75f)) steer = 1;
	else if(distance < -10.0f * (0.25 + variation * 0.75f)) steer = -1;
	else steer = 0;
	if(angle > 75.0f)
	{
		const float *v;
		v = dBodyGetLinearVel(content.play.players[player].machine->body[0]);
		if(content.play.players[player].distance < 20.0f * 2.0f)
		{
			if(H3D::Math::Vector::NormFunc(v) > 2.0f) gas = 0;
			else gas = 1;
		}
		else { gas = -1; steer *= -1; }
	}
	else gas = 1;
	content.play.players[player].machine->drive(gas == 1, gas == -1, steer == 1, steer == -1);
}

void
HMachines::human(unsigned int player)
{
	float position[3], target[3], direction[3];

	H3D::Math::Point::Copy(dBodyGetPosition(content.play.players[player].machine->body[0]), position);
	H3D::Math::Point::Copy(&content.play.track->content.checkpoints[(content.play.players[player].checkpoint + 1) % (content.play.track->info.ncheckpoints) * 3], target);
	H3D::Math::Vector::Sub(target, position, direction);
	H3D::Math::Vector::Norm(direction, &content.play.players[player].distance);
	if(content.play.players[player].distance < 40.0f)
	{
		content.play.players[player].checkpoint++;
		H3D::Math::Point::Copy(&content.play.track->content.checkpoints[(content.play.players[player].checkpoint + 1) % (content.play.track->info.ncheckpoints) * 3], target);
		H3D::Math::Vector::Sub(target, position, direction);
		H3D::Math::Vector::Norm(direction, &content.play.players[player].distance);
	}
	content.play.players[player].machine->drive(content.play.players[player].keys[0], content.play.players[player].keys[1], content.play.players[player].keys[2], content.play.players[player].keys[3]);
}

void
HMachines::NearCallback(void *data, dGeomID o1, dGeomID o2)
{
	HMachines* game;
	const int ncontacts = 1024;
	unsigned int i;
	int n;
	dBodyID b1, b2;
	dContact contacts[ncontacts];
	dJointID c;
	game = (HMachines*) data;
	b1 = dGeomGetBody(o1);
	b2 = dGeomGetBody(o2);
	if(b1 && b2 && dAreConnectedExcluding(b1, b2, dJointTypeContact))
		return;
	n = dCollide(o1, o2, ncontacts, &contacts[0].geom, sizeof(dContact));
	if(n > 0)
	{
		for(i = 0; i < (unsigned int) n; i++)
		{
			contacts[i].surface.mode = dContactSlip1 | dContactSlip2 | dContactSoftERP | dContactSoftCFM | dContactApprox1 | dContactMu2;
			contacts[i].surface.mu = 0.75f;
			contacts[i].surface.mu2 = 0.75f;
			contacts[i].surface.slip1 = 0.01f;
			contacts[i].surface.slip2 = 0.01f;
			contacts[i].surface.soft_erp = 0.5f;
			contacts[i].surface.soft_cfm = 0.3f;
			c = dJointCreateContact(game->content.world, game->content.joints, &contacts[i]);
			dJointAttach(c, dGeomGetBody(contacts[i].geom.g1), dGeomGetBody(contacts[i].geom.g2));
		}
	}
}
