#pragma once

#include <H3D/Macros.h>
#include <H3D/Math.h>
#include <string>
#include <GL/glew.h>
#include <GL/gl.h>

namespace H3D
{
	namespace Mesh
	{
		class H3D_API H3D
		{
			public:

				typedef struct
				{
					int magic;
					unsigned int nvertices;
					unsigned int ntextures;
					unsigned int nmaterials;
					unsigned int nfaces;
				} Info;

				typedef struct
				{
					float coord[3];
					float normal[3];
				} Vertex;

				typedef struct
				{
					char name[64];
				} Texture;
				typedef struct
				{
					unsigned int texture;
					float alpha;
					float emit;
					float ambient[3];
					float diffuse[3];
					float specular[3];
				} Material;

				typedef struct
				{
					unsigned int vertices[3];
					float texcoords[3][2];
					float normal[3];
					unsigned int material;
				} Face;

				typedef struct
				{
					Vertex* vertices;
					Texture* textures;
					Material* materials;
					Face* faces;
				} Content;

				Info info;
				Content content;

				H3D(void);

				virtual
				~H3D(void);

				bool
				load(const std::string& filename, bool compiled = true);

				virtual
				void
				unload(void);

				bool
				save(const std::string& filename, bool compiled = true);
		};

		namespace Q3
		{
			class H3D_API BSP
			{
				public:

					typedef enum { NONE, GL } Renderer;

					typedef struct
					{
						int offset;
						int length;
					} DirEntry;

					typedef struct
					{
						char magic[4];
						int version;
						DirEntry direntries[17];
					} Header;

					typedef struct
					{
						char *ents;
					} Entity;

					typedef struct
					{
						char name[64];
						int flags;
						int contents;
					} Texture;

					typedef struct
					{
						float normal[3];
						float dist;
					} Plane;

					typedef struct
					{
						int plane;
						int children[2];
						int mins[3];
						int maxs[3];
					} Node;

					typedef struct
					{
						int cluster;
						int area;
						int mins[3];
						int maxs[3];
						int leafface;
						int nleaffaces;
						int leafbrush;
						int nleafbrushes;
					} Leaf;

					typedef struct
					{
						int face;
					} LeafFace;

					typedef struct
					{
						int brush;
					} LeafBrush;

					typedef struct
					{
						float mins[3];
						float maxs[3];
						int face;
						int nfaces;
						int brush;
						int nbrushes;
					} Model;

					typedef struct
					{
						int brushside;
						int nbrushsides;
						int texture;
					} Brush;

					typedef struct
					{
						int plane;
						int texture;
					} BrushSide;

					typedef struct
					{
						float position[3];
						float texcoord[2][2];
						float normal[3];
						unsigned char color[4];
					} Vertex;

					typedef struct
					{
						int offset;
					} MeshVert;

					typedef struct
					{
						char name[64];
						int brush;
						int unknown;
					} Effect;

					typedef struct
					{
						int texture;
						int effect;
						int type;
						int vertex;
						int nvertices;
						int meshvert;
						int nmeshverts;
						int lightmap;
						int lmstart[2];
						int lmsize[2];
						float lmorigin[3];
						float lmvecs[2][3];
						float normal[3];
						int size[2];
					} Face;

					typedef struct
					{
						unsigned char map[128][128][3];
					} LightMap;

					typedef struct
					{
						unsigned char ambient[3];
						unsigned char directional[3];
						unsigned char dir[2];
					} LightVol;

					typedef struct
					{
						int nvecs;
						int szvecs;
						unsigned char *vecs;
					} VisData;

					typedef struct
					{
						GLuint *textures;
						GLuint *lightmaps;
					} GraphicData;
/*
					typedef struct
					{
						int level;
						Vertex *vertices;
						int indices;
						int nrowtriangles;
						int rowindices;
						Vertex controls[9];
					} Patch;
*/
					Header header;

					Entity entity;
					Texture *textures;
					Plane *planes;
					Node *nodes;
					Leaf *leaves;
					LeafFace *leaffaces;
					LeafBrush *leafbrushes;
					Model *models;
					Brush *brushes;
					BrushSide *brushsides;
					Vertex *vertices;
					MeshVert *meshverts;
					Effect *effects;
					Face *faces;
					LightMap *lightmaps;
					LightVol *lightvols;
					VisData visdata;

					GraphicData graphicdata;

					BSP(void);

					virtual
					~BSP(void);

					int
					load(const std::string& filename);

					void
					loadGraphicData(const std::string& path);

					void
					render(void);

			protected:

					void
					loadEntity(FILE *stream);

					void
					loadTexture(FILE *stream);

					void
					loadPlane(FILE *stream);

					void
					loadNode(FILE *stream);

					void
					loadLeaf(FILE *stream);

					void
					loadLeafFace(FILE *stream);

					void
					loadLeafBrush(FILE *stream);

					void
					loadModel(FILE *stream);

					void
					loadBrush(FILE *stream);

					void
					loadBrushSide(FILE *stream);

					void
					loadVertex(FILE *stream);

					void
					loadMeshVert(FILE *stream);

					void
					loadEffect(FILE *stream);

					void
					loadFace(FILE *stream);

					void
					loadLightMap(FILE *stream);

					void
					loadLightVol(FILE *stream);

					void
					loadVisData(FILE *stream);

					void
					renderFace(unsigned int i);
			};

			H3D_API
			H3D *
			load(const std::string& filename);
		}
	}
}
