#!BPY

# Copyright (C) 2008 Juan Manuel Borges Caño

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Name: 'BTM (.btm)...'
Blender: 243
Group: 'Export'
Tooltip: 'Save a BTM file'
"""

"""
Mesh:
	name
	Materials
		alpha
		amb
		diffuse (r g b)
		emit
		specular (r g b)
		texture
	Verts
		co (x y z)
		no (x, y, z)
	Faces
		mat
		verts
		col
		no
		uv
"""

import Blender, sys

def export(filename):
	if not filename.lower().endswith('.btm'):
		filename += '.btm'
	stream = open(filename, 'w')

	scene = Blender.Scene.GetCurrent()
	world = scene.world
	stream.write('begin objects\n')
	stream.write('	length %s\n' % len(scene.objects.selected))
	for object in scene.objects.selected:

		mesh = object.getData(mesh=1)

		stream.write('	{\n')
		stream.write('		name %s\n' % object.name)
		stream.write('		begin mesh\n')

		stream.write('			name %s\n' % mesh.name)

		stream.write('			begin materials\n')
		stream.write('				length %s\n' % len(mesh.materials))
		for material in mesh.materials:
			stream.write('				{\n')
			stream.write('					name %s\n' % material.name)
			textures = material.getTextures()
			if textures[0]:
				texture = textures[0]
				stream.write('					texture %s\n' % Blender.sys.basename(texture.tex.image.filename))
			stream.write('					alpha %s\n' % material.alpha)
			stream.write('					emit %s\n' % material.emit)
			amb0 = world.amb[0] * material.amb
			amb1 = world.amb[1] * material.amb
			amb2 = world.amb[2] * material.amb
			stream.write('					ambient %s %s %s\n' % (amb0, amb1, amb2))
			stream.write('					diffuse %s %s %s\n' % (material.R, material.G, material.B))
			stream.write('					specular %s %s %s\n' %(material.specR, material.specG, material.specG))
			stream.write('				}\n')

		stream.write('			end\n')
				
		stream.write('			begin vertices\n')
		stream.write('				length %s\n' % len(mesh.verts))
		for vert in mesh.verts:
			co = vert.co * object.getMatrix()
			no = vert.no * object.getMatrix()
			stream.write('				{\n')
			stream.write('					co %s %s %s\n' % (co.x, co.y, co.z))
			stream.write('					no %s %s %s\n' % (no.x, no.y, no.z))
			stream.write('				}\n')
		stream.write('			end\n')

		stream.write('			begin faces\n')
		stream.write('				length %s\n' % len(mesh.faces))
		for face in mesh.faces:
			stream.write('				{\n')
			stream.write('					mat %s\n' % face.mat)
			stream.write('					verts')
			for vert in face.verts:
				stream.write(' %s' % vert.index)
			stream.write('\n')
			if mesh.faceUV:
				stream.write('					{\n')
				for i in range(len(face.verts)):
					if mesh.faceUV:
						stream.write('						uv %s %s\n' % (face.uv[i][0], face.uv[i][1]))
				stream.write('					}\n')
			if mesh.faceUV:
				stream.write('					no %s %s %s\n' % (face.no.x, face.no.y, face.no.z))
			stream.write('				}\n')
		stream.write('			end\n')
		stream.write('		end\n')
		stream.write('	}\n')
	stream.write('end\n')
	stream.close()

Blender.Window.FileSelector(export, 'Export btm', Blender.sys.makename(ext='.btm'))
