// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "ml.h"
#include <stdio.h>
#include <string.h>

// Point

float mlPointZero[3] = {0, 0, 0};

// Vector

float mlVectorX[3] = {1, 0, 0};
float mlVectorY[3] = {0, 1, 0};
float mlVectorZ[3] = {0, 0, 1};
float mlVectorNegX[3] = {-1, 0, 0};
float mlVectorNegY[3] = {0, -1, 0};
float mlVectorNegZ[3] = {0, 0, -1};

void
mlVectorPrint(const float *a)
{
	printf("(%f, %f, %f)\n", a[0], a[1], a[2]);
}

void
mlVectorCopy(const float *a, float *result)
{
	memcpy(result, a, 3 * sizeof(float));
}

void
mlVectorLoad(float *a, float x, float y, float z)
{
	a[0] = x;
	a[1] = y;
	a[2] = z;
}

void
mlVectorAdd(const float *a, const float *b, float *result)
{
	unsigned int i;
	for(i = 0; i < 3; i++)
		result[i] = a[i] + b[i];
}

void
mlVectorAddInPlace(float *a, const float *b)
{
	unsigned int i;
	for(i = 0; i < 3; i++)
		a[i] += b[i];
}

void
mlVectorSub(const float *a, const float *b, float *result)
{
	unsigned int i;
	for(i = 0; i < 3; i++)
		result[i] = a[i] - b[i];
}

void
mlVectorSubInPlace(float *a, const float *b)
{
	unsigned int i;
	for(i = 0; i < 3; i++)
		a[i] -= b[i];
}

void
mlVectorScalarProduct(const float *a, float b, float *result)
{
	unsigned int i;
	for(i = 0; i < 3; i++)
		result[i] = a[i] * b;
}

void
mlVectorScalarProductInPlace(float *a, float b)
{
	unsigned int i;
	for(i = 0; i < 3; i++)
		a[i] *= b;
}

void
mlVectorNorm(const float *a, float *result)
{
	*result = sqrt(a[0] * a[0] + a[1] * a[1] + a[2] * a[2]);
}

float
mlVectorNormFunc(const float *a)
{
	return sqrt(a[0] * a[0] + a[1] * a[1] + a[2] * a[2]);
}

void
mlVectorDotProduct(const float *a, const float *b, float *result)
{
	*result = a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
}

float
mlVectorDotProductFunc(const float *a, const float *b)
{
	return a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
}

void
mlVectorCrossProduct(const float *a, const float *b, float *result)
{
	result[0] = a[1] * b[2] - a[2] * b[1];
	result[1] = a[2] * b[0] - a[0] * b[2];
	result[2] = a[0] * b[1] - a[1] * b[0];
}

void
mlVectorCrossProductInPlace(float *a, const float *b)
{
	float result[3];
	mlVectorCrossProduct(a, b, result);
	mlVectorCopy(result, a);
}

void
mlVectorNormal(const float *a, float *result)
{
	float norm;
	mlVectorNorm(a, &norm);
	mlVectorScalarProduct(a, 1.0 / norm, result);
}

void
mlVectorNormalInPlace(float *a)
{
	mlVectorScalarProductInPlace(a, 1.0 / mlVectorNormFunc(a));
}

void
mlVectorMatrixProduct(const float *a, const float *b, float *result)
{
	result[0] = a[0] * b[0] + a[1] * b[1] + a[2] * b[2] + b[3];
	result[1] = a[0] * b[4] + a[1] * b[5] + a[2] * b[6] + b[7];
	result[2] = a[0] * b[8] + a[1] * b[9] + a[2] * b[10] + b[11];
}

void
mlVectorMatrixProductInPlace(float *a, const float *b)
{
	float result[3];
	mlVectorMatrixProduct(a, b, result);
	mlVectorCopy(result, a);
}

void
mlVectorRotate(const float *a, const float *axis, float angle, float *result)
{
	float matrix[9];
	float c = cosf(mlDegToRad(angle));
	float s = sinf(mlDegToRad(angle));
	matrix[0] = axis[0] * axis[0] * (1 - c) + c;
	matrix[1] = axis[0] * axis[1] * (1 - c) - axis[2] * s;
	matrix[2] = axis[0] * axis[2] * (1 - c) + axis[1] * s;
	matrix[3] = axis[1] * axis[0] * (1 - c) + axis[2] * s;
	matrix[4] = axis[1] * axis[1] * (1 - c) + c;
	matrix[5] = axis[1] * axis[2] * (1 - c) - axis[0] * s;
	matrix[6] = axis[0] * axis[2] * (1 - c) - axis[1] * s;
	matrix[7] = axis[1] * axis[2] * (1 - c) + axis[0] * s;
	matrix[8] = axis[2] * axis[2] * (1 - c) + c;
	mlVectorMatrixProduct(a, matrix, result);
}

void
mlVectorRotateInPlace(float *a, const float *axis, float angle)
{
	float result[3];
	mlVectorRotate(a, axis, angle, result);
	mlVectorCopy(result, a);
}

void
mlVectorAngle(const float *a, const float *b, float *angle)
{
	float dp;
	float na, nb;
	mlVectorDotProduct(a, b, &dp);
	mlVectorNorm(a, &na);
	mlVectorNorm(b, &nb);
	*angle = mlRadToDeg(acosf(dp/(na * nb)));
}

float
mlVectorAngleFunc(const float *a, const float *b)
{
	float dp;
	float na, nb;
	mlVectorDotProduct(a, b, &dp);
	mlVectorNorm(a, &na);
	mlVectorNorm(b, &nb);
	return mlRadToDeg(acosf(dp/(na * nb)));
}

// Matrix

void
mlMatrixPrint(const float *a)
{
	printf("[(%f, %f, %f, %f), (%f, %f, %f, %f), (%f, %f, %f, %f), (%f, %f, %f, %f)]\n", a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9], a[10], a[11], a[12], a[13], a[14], a[15]);
}

void
mlMatrixCopy(const float *a, float *result)
{
	memcpy(result, a, 16 * sizeof(float));
}

void
mlMatrixLoad(float *x, float a, float b, float c, float d, float e, float f, float g, float h, float i, float j, float k, float l, float m, float n, float o, float p)
{
	x[0] = a;  x[1] = b;  x[2] = c;  x[3] = d;
	x[4] = e;  x[5] = f;  x[6] = g;  x[7] = h;
	x[8] = i;  x[9] = j; x[10] = k; x[11] = l;
	x[12] = m; x[13] = n; x[14] = o; x[15] = p;
}

void
mlMatrixLoadIdentity(float *a)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		a[i] = (i % 5) ? 0 : 1;
}

void
mlMatrixAdd(const float *a, const float *b, float *result)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		result[i] = a[i] + b[i];
}

void
mlMatrixAddInPlace(float *a, const float *b)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		a[i] += b[i];
}

void
mlMatrixSub(const float *a, const float *b, float *result)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		result[i] = a[i] - b[i];
}

void
mlMatrixSubInPlace(float *a, const float *b)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		a[i] -= b[i];
}

void
mlMatrixScalarProduct(const float *a, float b, float *result)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		result[i] = a[i] * b;
}

void
mlMatrixScalarProductInPlace(float *a, float b)
{
	unsigned int i;
	for(i = 0; i < 16; i++)
		a[i] *= b;
}

void
mlMatrixProduct(const float *a, const float *b, float *result)
{
	unsigned int i, j;
	for(j = 0; j < 4; j++)
		for(i = 0; i < 4; i++)
			result[j * 4 + i] = a[j * 4 + 0] * b[0 * 4 + i] + a[j * 4 + 1] * b[1 * 4 + i] + a[j * 4 + 2] * b[2 * 4 + i] + a[j * 4 + 3] * b[3 * 4 + i];
}

void
mlMatrixProductInPlace(float *a, const float *b)
{
	float result[16];
	mlMatrixProduct(a, b, result);
	mlMatrixCopy(result, a);
}

void
mlMatrixTranspose(const float *a, float *result)
{
	unsigned int i, j;
	for(j = 0; j < 4; j++)
		for(i = 0; i < 4; i++)
			result[j * 4 + i] = a[i * 4 + j];
}

void
mlMatrixTransposeInPlace(float *a)
{
	float result[16];
	mlMatrixTranspose(a, result);
	mlMatrixCopy(result, a);
}

void
mlMatrixScale(const float *a, const float *b, float *result)
{
	float c[16] = {b[0],    0,     0, 0,
		          0, b[1],     0, 0,
		          0,    0,  b[2], 0,
		          0,    0,     0, 1};
	mlMatrixProduct(a, c, result);
}

void
mlMatrixScaleInPlace(float *a, const float *b)
{
	float result[16];
	mlMatrixScale(a, b, result);
	mlMatrixCopy(result, a);
}

void
mlMatrixRotate(const float *a, const float *axis, float angle, float *result)
{
	float b[16];
	float c = cosf(mlDegToRad(angle));
	float s = sinf(mlDegToRad(angle));

	b[0] = axis[0] * axis[0] * (1 - c) + c;
	b[1] = axis[0] * axis[1] * (1 - c) - axis[2] * s;
	b[2] = axis[0] * axis[2] * (1 - c) + axis[1] * s;

	b[4] = axis[1] * axis[0] * (1 - c) + axis[2] * s;
	b[5] = axis[1] * axis[1] * (1 - c) + c;
	b[6] = axis[1] * axis[2] * (1 - c) - axis[0] * s;

	b[8] = axis[0] * axis[2] * (1 - c) - axis[1] * s;
	b[9] = axis[1] * axis[2] * (1 - c) + axis[0] * s;
	b[10] = axis[2] * axis[2] * (1 - c) + c;

	b[3] = b[7] = b[11] = b[12] = b[13] = b[14] = 0;
	b[15] = 1;

	mlMatrixProduct(a, b, result);
}

void
mlMatrixRotateInPlace(float *a, const float *axis, float angle)
{
	float result[16];
	mlMatrixRotate(a, axis, angle, result);
	mlMatrixCopy(result, a);
}

void
mlMatrixTranslate(const float *a, const float *b, float *result)
{
	float c[16] = {1, 0, 0, b[0],
		       0, 1, 0, b[1],
		       0, 0, 1, b[2],
		       0, 0, 0, 1};
	mlMatrixProduct(a, c, result);
}

void
mlMatrixTranslateInPlace(float *a, const float *b)
{
	float result[16];
	mlMatrixTranslate(a, b, result);
	mlMatrixCopy(result, a);
}

void
mlMatrixLookAt(float *eye, float *center, float *up, float *result)
{
	float tmp[3], forward[3];

	mlVectorSub(center, eye, tmp);
	mlVectorNormal(tmp, forward);

	mlMatrixLook(eye, forward, up, result);
}

void
mlMatrixLook(float *position, float *forward, float *up, float *result)
{
	unsigned int i;
	float right[3], rotation[16];

	mlVectorCrossProduct(forward, up, right);
	for(i = 0; i < 3; i++)
	{
		rotation[i] = right[i];
		rotation[1 * 4 + i] = up[i];
		rotation[2 * 4 + i] = -forward[i];
		rotation[3 * 4 + i] = rotation[i * 4 + 3] = 0;
	}
	rotation[15] = 1;

	// tmp use of right
	mlVectorScalarProduct(position, -1, right);
	mlMatrixTranslate(rotation, right, result);
}

void
mlMatrixGetPosition(const float *a, float *result)
{
	float translation[16];
	mlMatrixGetTranslation(a, translation);
	result[0] = translation[3];
	result[1] = translation[7];
	result[2] = translation[11];
}

void
mlMatrixGetRight(const float *a, float *result)
{
	float tmp[3];
	tmp[0] = a[0];
	tmp[1] = a[1];
	tmp[2] = a[2];
	mlVectorNormal(tmp, result);
}

void
mlMatrixGetUp(const float *a, float *result)
{
	float tmp[3];
	tmp[0] = a[4];
	tmp[1] = a[5];
	tmp[2] = a[6];
	mlVectorNormal(tmp, result);
}

void
mlMatrixGetForward(const float *a, float *result)
{
	float tmp[3];
	tmp[0] = -a[8];
	tmp[1] = -a[9];
	tmp[2] = -a[10];
	mlVectorNormal(tmp, result);
}

void
mlMatrixGetEscalation(const float *a, float *result)
{
	float tmp[3];

	             result[1] =  result[2] = result[3] = 
	result[4] =               result[6] = result[7] =
	result[8] =  result[9] =              result[11] =
	result[12] = result[13] = result[14] =           0;

	tmp[0] = a[0]; tmp[1] = a[1]; tmp[2] = a[2];
	mlVectorNorm(tmp, &result[0]);

	tmp[0] = a[4]; tmp[1] = a[5]; tmp[2] = a[6];
	mlVectorNorm(tmp, &result[5]);

	tmp[0] = -a[8]; tmp[1] = -a[9]; tmp[2] = -a[10];
	mlVectorNorm(tmp, &result[10]);

	result[15] = 1;
}

void
mlMatrixGetRotation(const float *a, float *result)
{
	float tmp[3];

	mlMatrixGetRight(a, tmp);
	result[0] = tmp[0]; result[1] = tmp[1]; result[2] = tmp[2];

	mlMatrixGetUp(a, tmp);
	result[4] = tmp[0]; result[5] = tmp[1]; result[6] = tmp[2];

	mlMatrixGetForward(a, tmp);
	result[8] = -tmp[0]; result[9] = -tmp[1]; result[10] = -tmp[2];

	result[3] =
	result[7] =
	result[11] =
	result[12] = result[13] = result[14] = 0;

	result[15] = 1;
}

void
mlMatrixGetTranslation(const float *a, float *result)
{
	float b[16], c[16];
	mlMatrixGetRotation(a, b);
	mlMatrixTranspose(b, c);
	b[0] = b[1] = b[2] = 
	b[4] = b[5] = b[6] = 
	b[8] = b[9] = b[10] = 
	b[12] = b[13] = b[14] = 0;
       	b[15] = 1;
	b[3] = a[3];
	b[7] = a[7];
	b[11] = a[11];
	mlMatrixProduct(c, b, result);
}

// Triangle

void
mlTrianglePrint(const float *a)
{
	printf("[(%f, %f, %f), (%f, %f, %f), (%f, %f, %f)]", a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
}

void
mlTriangleCopy(const float *a, float *result)
{
	memcpy(result, a, 3 * 3 * sizeof(float));
}

void
mlTriangleGetBoundSphere(const float *a, float *result)
{
	result[0] = (a[0 * 3 + 0] + a[1 * 3 + 0] + a[2 * 3 + 0]) / 3.0;
	result[1] = (a[0 * 3 + 1] + a[1 * 3 + 1] + a[2 * 3 + 1]) / 3.0;
	result[2] = (a[0 * 3 + 2] + a[1 * 3 + 2] + a[2 * 3 + 2]) / 3.0;
  	result[3] = sqrt(powf(result[0] - a[0 * 3 + 0], 2) + powf(result[1] - a[0 * 3 + 1], 2) + powf(result[2] - a[0 * 3 + 2], 2));
}

// Sphere

void
mlSpherePrint(const float *a)
{
	printf("[(%f, %f, %f), %f]\n", a[0], a[1], a[2], a[3]);
}

void
mlSphereCopy(const float *a, float *result)
{
	memcpy(result, a, 4 * sizeof(float));
}

void
mlSphereLoad(float *a, float x, float y, float z, float r)
{
	a[0] = x;
	a[1] = y;
	a[2] = z;
	a[3] = r;
}

void
mlSphereMerge(const float *a, const float *b, float *result)
{
	float distancer[3];
	float distance;
	distancer[0] = b[0] - a[0];
	distancer[1] = b[1] - a[1];
	distancer[2] = b[2] - a[2];
	distance = sqrt(powf(distancer[0], 2) + powf(distancer[1], 2) + powf(distancer[2], 2));
	if(distance + b[3] > a[3])
	{
		if(distance + a[3] > b[3])
		{
			distancer[0] /= distance;
			distancer[1] /= distance;
			distancer[2] /= distance;
			result[3] = (distance + a[3] + b[3]) / 2.0;
			result[0] = a[0] + distancer[0] * (result[3] - a[3]);
			result[1] = a[1] + distancer[1] * (result[3] - a[3]);
			result[2] = a[2] + distancer[2] * (result[3] - a[3]);
		}
		else
			mlSphereCopy(b, result);
	}
	else
		mlSphereCopy(a, result);
}

void
mlSphereMergeInPlace(float *a, const float *b)
{
	float result[4];
	mlSphereMerge(a, b, result);
	mlSphereCopy(result, a);
}
