// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "menuplay.h"
#include "menumain.h"
#include "main.h"
#include "camera.h"
#include "machine.h"
#include "track.h"
#include "chain.h"
#include "cl.h"
#include "ml.h"
#include <X11/keysym.h>
#include <GL/glu.h>

static FM_Camera camera;

static chain *machines;
static chain *activemachines;
static bool keys[4][4];

#define CONTACTS 1024

static
void
FM_NearCallback(void *data, dGeomID o1, dGeomID o2)
{
	FM_Game *game;
	dBodyID b1, b2;
	dContact contact[CONTACTS];
	dJointID c;
	int i, n;
	void *d1, *d2;

	game = data;

	b1 = dGeomGetBody(o1);
	b2 = dGeomGetBody(o2);
	if(b1 && b2 && dAreConnectedExcluding(b1, b2, dJointTypeContact))
		return;

	d1 = dGeomGetData(o1);
	d2 = dGeomGetData(o2);

	if(chain_contains(machines, d1) && chain_contains(machines, d2) && (!chain_contains(activemachines, d1) || !chain_contains(activemachines, d2)))
		return;

	n = dCollide(o1, o2, CONTACTS, &contact[0].geom, sizeof(dContact));
	if(n > 0)
	{
		for(i = 0; i < n; i++)
		{
			contact[i].surface.mode = dContactSlip1 | dContactSlip2 | dContactSoftERP | dContactSoftCFM | dContactApprox1 | dContactMu2;
			contact[i].surface.mu = 0.99;
			contact[i].surface.mu2 = 0.99;
			contact[i].surface.slip1 = 0.01;
			contact[i].surface.slip2 = 0.01;
			contact[i].surface.soft_erp = 0.5;
			contact[i].surface.soft_cfm = 0.3;
			c = dJointCreateContact(game->play.world, game->play.contactgroup, &contact[i]);
			dJointAttach(c, dGeomGetBody(contact[i].geom.g1), dGeomGetBody(contact[i].geom.g2));
		}
	}
}

static
void
FM_ActivateMachinesScanner(FM_Machine *machine)
{
	activemachines = chain_append(activemachines, machine);
}

static
void
FM_ActivateMachines()
{
	chain_scan(machines, (scanner)FM_ActivateMachinesScanner);
}

void
FM_EnterMenuPlay(FM_Game *game)
{
	float pos[3];
	unsigned int i;

	alSourceStop(game->menu.audiosources[FM_GAME_MENU_MUSIC]);

	memset(keys, 0, sizeof(bool) * 16);
	machines = activemachines = NULL;

	mlPointLoad(camera.object, 0, 0, 0);
	alListener3f(AL_POSITION, camera.object[0], 75, camera.object[2]);

	if(!game->play.track)
	{
		game->play.track = FM_LoadTrack(game->play.space, game->tracknames[game->options.track]);
		game->play.cache.track = game->options.track;
	}
	else
	{
		if(game->options.track != game->play.cache.track)
		{
			FM_FreeTrack(game->play.track);
			game->play.track = FM_LoadTrack(game->play.space, game->tracknames[game->options.track]);
			game->play.cache.track = game->options.track;
		}
	}

	for(i = 0; i < game->options.nmachines; i++)
	{
		mlPointLoad(pos, (i - game->options.nmachines / 2.0) * 5.0, 5, 0);
		machines = chain_append(machines, FM_LoadMachine(game->play.world, game->play.space, 75, 20, pos, -90, game->machinenames[game->options.machine]));
	}

	FM_ActivateMachines();

	FM_SetKeyPressEventHandler(FM_KeyPressEventMenuPlay);
	FM_SetKeyReleaseEventHandler(FM_KeyReleaseEventMenuPlay);
	FM_SetUpdateHandler(FM_UpdateMenuPlay);
	FM_SetExitHandler(FM_ExitMenuPlay);
}

void
FM_ExitMenuPlay(FM_Game *game)
{
	chain_scan(machines, (scanner)FM_FreeMachine);
	chain_free(machines);
	chain_free(activemachines);
}

static
void
FM_UpdateActiveMachines()
{
	chain *c, *n;
	FM_Machine *machine;

	c = activemachines;
	while(c)
	{
		machine = c->data;
		n = c->next;
		if(!FM_UpdateMachine(machine, &camera))
		{
			activemachines = chain_remove_chain(activemachines, c);
			alSourceStop(machine->engine);
		}
		c = n;
	}
}

static
FM_Machine *
FM_FirstMachine()
{
	FM_Machine *machine, *firstmachine;
	chain *l;

	firstmachine = NULL;
	if(activemachines)
	{
		firstmachine = activemachines->data;
		for(l = activemachines->next; l; l = l->next)
		{
			machine = l->data;
			if(machine->checkpoint > firstmachine->checkpoint || (machine->checkpoint == firstmachine->checkpoint && machine->checkpointdistance < firstmachine->checkpointdistance))
				firstmachine = machine;
		}
	}
	return firstmachine;
}

static
void
FM_RenderMenuPlay(FM_Game *game)
{
	FM_Machine *firstmachine;
	float a[3], b[3];
	const dReal *p, *v;
	wchar_t text[2];

	firstmachine = FM_FirstMachine();

	if(!firstmachine)
		return;

       	p = dBodyGetPosition(firstmachine->body[0]);
	v = dBodyGetLinearVel(firstmachine->body[0]);

	mlVectorSub(p, camera.object, a);
	mlVectorScalarProduct(a, 0.15, b);
	mlVectorAddInPlace(camera.object, b);

	alListener3f(AL_POSITION, camera.object[0], 70, camera.object[2]);
	alListenerfv(AL_ORIENTATION, mlVectorNegY);
	alListenerfv(AL_VELOCITY, v);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glLoadIdentity();

	gluLookAt(camera.object[0], 70, camera.object[2], camera.object[0], camera.object[1], camera.object[2], 0, 0, -1);
	FM_CameraGLFrustumLoad(&camera);

	FM_UpdateActiveMachines();
	FM_RenderTrack(game->play.track, &camera);
	chain_scan(activemachines, (scanner)FM_RenderMachine);

	swprintf(text, 2, L"%i", chain_length(activemachines));
	glDisable(GL_TEXTURE_2D);
	glPushMatrix();
	glTranslatef(camera.object[0] - 4.0 * (float) game->screen.w / game->screen.h, 60, camera.object[2] + 4.0);
	glRotatef(-90, 1, 0, 0);
	glScalef(0.02, 0.02, 1);
	clRenderString(game->face, CL_OUTLINE, text, 1);
	glPopMatrix();
	glEnable(GL_TEXTURE_2D);
}

static
void
FM_ComputerControlMachine(FM_Game *game, FM_Machine *machine)
{
	float position[3], target[3], direction[3], orientation[3], plane[4], angle, variation, distance;
	const dReal *r;
	int steer, gas;

	variation = (float)rand() / RAND_MAX;

	mlPointCopy(dBodyGetPosition(machine->body[0]), position);
	mlPointCopy(&game->play.track->checkpoints[(machine->checkpoint + 1) % (game->play.track->ncheckpoints) * 3], target);

	mlVectorSub(target, position, direction);

	mlVectorNorm(direction, &machine->checkpointdistance);

	if(machine->checkpointdistance < 20)
	{
		machine->checkpoint++;

		mlPointCopy(&game->play.track->checkpoints[(machine->checkpoint + 1) % (game->play.track->ncheckpoints) * 3], target);

		mlVectorSub(target, position, direction);

		mlVectorNorm(direction, &machine->checkpointdistance);
	}

	mlVectorNormalInPlace(direction);

	r = dBodyGetRotation(machine->body[0]);
	orientation[0] = -r[2];
	orientation[1] = -r[6];
	orientation[2] = -r[10];

	mlVectorCrossProduct(mlVectorY, orientation, plane);
	plane[3] = -mlVectorDotProductFunc(plane, position);

	distance = mlVectorDotProductFunc(plane, target) + plane[3];

	mlVectorAngle(orientation, direction, &angle);

	if(distance > 10.0 * (0.5 + variation * 0.5)) steer = 1;
	else if(distance < -10.0 * (0.5 + variation * 0.5)) steer = -1;
	else steer = 0;

	if(angle > 75)
	{
		const dReal *v;

		v = dBodyGetLinearVel(machine->body[0]);

		if(machine->checkpointdistance < 20 * 2)
		{
		       if(mlVectorNormFunc(v) > 2) gas = 0;
		       else gas = 1;
		}
		else { gas = -1; steer *= -1; }
	}
	else gas = 1;

	FM_ControlMachine(machine, gas == 1, gas == -1, steer == 1, steer == -1);
}

static
void
FM_HumanControlMachine(FM_Game *game, FM_Machine *machine, const bool *keys)
{
	float position[3], target[3], direction[3];

	mlPointCopy(dBodyGetPosition(machine->body[0]), position);
	mlPointCopy(&game->play.track->checkpoints[(machine->checkpoint + 1) % (game->play.track->ncheckpoints) * 3], target);

	mlVectorSub(target, position, direction);

	mlVectorNorm(direction, &machine->checkpointdistance);

	if(machine->checkpointdistance < 40)
	{
		machine->checkpoint++;

		mlPointCopy(&game->play.track->checkpoints[(machine->checkpoint + 1) % (game->play.track->ncheckpoints) * 3], target);

		mlVectorSub(target, position, direction);

		mlVectorNorm(direction, &machine->checkpointdistance);
	}

	FM_ControlMachine(machine, keys[0], keys[1], keys[2], keys[3]);
}

static
void
FM_ControlActiveMachines(FM_Game *game)
{
	chain *l;
	unsigned int i;
	FM_Machine *machine;
	for(i = 0, l = machines; l; l = l->next, i++)
	{
		machine = l->data;
		if(chain_contains(activemachines, machine))
		{
			i < game->options.nplayers ? FM_HumanControlMachine(game, machine, keys[i]) : FM_ComputerControlMachine(game, machine);
		}
	}
}

bool
FM_KeyPressEventMenuPlay(FM_Game *game, XEvent *event)
{
	KeySym keysym;

	keysym = XLookupKeysym(&event->xkey, 0);
	switch(keysym)
	{
		case XK_Up: keys[0][0] = true; break;
		case XK_Down: keys[0][1] = true; break;
		case XK_Left: keys[0][2] = true; break;
		case XK_Right: keys[0][3] = true; break;
		case XK_w: keys[1][0] = true; break;
		case XK_s: keys[1][1] = true; break;
		case XK_a: keys[1][2] = true; break;
		case XK_d: keys[1][3] = true; break;
		case XK_i: keys[2][0] = true; break;
		case XK_k: keys[2][1] = true; break;
		case XK_j: keys[2][2] = true; break;
		case XK_l: keys[2][3] = true; break;
		case XK_t: keys[3][0] = true; break;
		case XK_g: keys[3][1] = true; break;
		case XK_f: keys[3][2] = true; break;
		case XK_h: keys[3][3] = true; break;
	}
	return true;
	return true;
}

bool
FM_KeyReleaseEventMenuPlay(FM_Game *game, XEvent *event)
{
	KeySym keysym;

	keysym = XLookupKeysym(&event->xkey, 0);
	switch(keysym)
	{
		case XK_Up: keys[0][0] = false; break;
		case XK_Down: keys[0][1] = false; break;
		case XK_Left: keys[0][2] = false; break;
		case XK_Right: keys[0][3] = false; break;
		case XK_w: keys[1][0] = false; break;
		case XK_s: keys[1][1] = false; break;
		case XK_a: keys[1][2] = false; break;
		case XK_d: keys[1][3] = false; break;
		case XK_i: keys[2][0] = false; break;
		case XK_k: keys[2][1] = false; break;
		case XK_j: keys[2][2] = false; break;
		case XK_l: keys[2][3] = false; break;
		case XK_t: keys[3][0] = false; break;
		case XK_g: keys[3][1] = false; break;
		case XK_f: keys[3][2] = false; break;
		case XK_h: keys[3][3] = false; break;
		case XK_Escape: FM_ExitMenuPlay(game); FM_EnterMenuMain(game); break;
	}
	return true;
}

bool
FM_UpdateMenuPlay(FM_Game *game)
{
	unsigned int i;
	
	FM_ControlActiveMachines(game);

	for(i = 0; i < 16; i++)
	{
		dSpaceCollide(game->play.space, game, &FM_NearCallback);
		dWorldStep(game->play.world, 0.01);
		dJointGroupEmpty(game->play.contactgroup);
	}

	FM_RenderMenuPlay(game);

	return true;
}

