// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "menuedit.h"
#include "menumain.h"
#include "main.h"
#include "camera.h"
#include "machine.h"
#include "track.h"
#include "chain.h"
#include "ml.h"
#include "cl.h"
#include <X11/keysym.h>
#include <GL/glu.h>

static FM_Camera camera;
static bool keys[4];

static chain *machines;
static chain *activemachines;
static chain *checkpoints;

#define CONTACTS 1024

static
void
FM_NearCallback(void *data, dGeomID o1, dGeomID o2)
{
	dBodyID b1, b2;
	dContact contact[CONTACTS];
	dJointID c;
	int i, n;
	void *d1, *d2;
	FM_Game *game;

	game = data;

	b1 = dGeomGetBody(o1);
	b2 = dGeomGetBody(o2);
	if(b1 && b2 && dAreConnectedExcluding(b1, b2, dJointTypeContact))
		return;

	d1 = dGeomGetData(o1);
	d2 = dGeomGetData(o2);

	if(chain_contains(machines, d1) && chain_contains(machines, d2) && (!chain_contains(activemachines, d1) || !chain_contains(activemachines, d2)))
		return;

	n = dCollide(o1, o2, CONTACTS, &contact[0].geom, sizeof(dContact));
	if(n > 0)
	{
		for(i = 0; i < n; i++)
		{
			contact[i].surface.mode = dContactSlip1 | dContactSlip2 | dContactSoftERP | dContactSoftCFM | dContactApprox1 | dContactMu2;
			contact[i].surface.mu = 0.99;
			contact[i].surface.mu2 = 0.99;
			contact[i].surface.slip1 = 0.01;
			contact[i].surface.slip2 = 0.01;
			contact[i].surface.soft_erp = 0.5;
			contact[i].surface.soft_cfm = 0.3;
			c = dJointCreateContact(game->play.world, game->play.contactgroup, &contact[i]);
			dJointAttach(c, dGeomGetBody(contact[i].geom.g1), dGeomGetBody(contact[i].geom.g2));
		}
	}
}

static
void
FM_ActivateMachinesScanner(FM_Machine *machine)
{
	activemachines = chain_append(activemachines, machine);
}

static
void
FM_ActivateMachines()
{
	chain_scan(machines, (scanner)FM_ActivateMachinesScanner);
}

static
float *
FM_GenCheckPoint(const float *point)
{
	float *checkpoint;
	checkpoint = malloc(sizeof(float) * 3);
	mlPointCopy(point, checkpoint);
	return checkpoint;
}

void
FM_EnterMenuEdit(FM_Game *game)
{
	float pos[3];
	unsigned int i;

	alSourceStop(game->menu.audiosources[FM_GAME_MENU_MUSIC]);

	memset(keys, 0, sizeof(bool) * 16);

	mlPointLoad(camera.object, 0, 0, 0);

	if(!game->play.track)
	{
		game->play.track = FM_LoadTrack(game->play.space, game->tracknames[game->options.track]);
		game->play.cache.track = game->options.track;
	}
	else
	{
		if(game->options.track != game->play.cache.track)
		{
			FM_FreeTrack(game->play.track);
			game->play.track = FM_LoadTrack(game->play.space, game->tracknames[game->options.track]);
			game->play.cache.track = game->options.track;
		}
	}

	machines = activemachines = checkpoints = NULL;

	for(i = 0; i < game->play.track->ncheckpoints; i++)
		checkpoints = chain_append(checkpoints, FM_GenCheckPoint(&game->play.track->checkpoints[i * 3]));

	mlPointLoad(pos, 0, 5, 0);
	machines = chain_append(machines, FM_LoadMachine(game->play.world, game->play.space, 37.5, 20, pos, -90, game->machinenames[game->options.machine]));

	FM_ActivateMachines();

	FM_SetKeyPressEventHandler(FM_KeyPressEventMenuEdit);
	FM_SetKeyReleaseEventHandler(FM_KeyReleaseEventMenuEdit);
	FM_SetUpdateHandler(FM_UpdateMenuEdit);
	FM_SetExitHandler(FM_ExitMenuEdit);
}

void
FM_ExitMenuEdit(FM_Game *game)
{

	chain_scan(machines, (scanner)FM_FreeMachine);
	chain_free(machines);

	chain_free(activemachines);

	chain_scan(checkpoints, (scanner)free);
	chain_free(checkpoints);
}

static
void
FM_UpdateActiveMachines()
{
	chain *c, *n;
	FM_Machine *machine;

	c = activemachines;
	while(c)
	{
		machine = c->data;
		n = c->next;
		if(!FM_UpdateMachine(machine, &camera))
		{
			activemachines = chain_remove_chain(activemachines, c);
			alSourceStop(machine->engine);
		}
		c = n;
	}
}

static
void
FM_RenderMenuEdit(FM_Game *game)
{
	FM_Machine *machine;
	float a[3], b[3];
	const dReal *p, *v;

	machine = activemachines->data;

       	p = dBodyGetPosition(machine->body[0]);
	v = dBodyGetLinearVel(machine->body[0]);

	mlVectorSub(p, camera.object, a);
	mlVectorScalarProduct(a, 0.15, b);
	mlVectorAddInPlace(camera.object, b);

	alListener3f(AL_POSITION, camera.object[0], 70, camera.object[2]);
	alListenerfv(AL_ORIENTATION, mlVectorNegY);
	alListener3f(AL_VELOCITY, v[0], v[1], v[2]);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glLoadIdentity();
	gluLookAt(camera.object[0], 70, camera.object[2], camera.object[0], camera.object[1], camera.object[2], 0, 0, -1);
	FM_CameraGLFrustumLoad(&camera);

	FM_UpdateActiveMachines();
	FM_RenderTrack(game->play.track, &camera);
	chain_scan(activemachines, (scanner)FM_RenderMachine);

	if(chain_length(checkpoints) >= 2)
	{
		chain *l;
		glBegin(GL_LINE_STRIP);
		for(l = checkpoints; l; l = l->next)
			glVertex3fv(l->data);
		glEnd();
	}
}

static
void
FM_ControlActiveMachines(FM_Game *game)
{
	FM_Machine *machine;

	machine = activemachines->data;
	FM_ControlMachine(machine, keys[0], keys[1], keys[2], keys[3]);
}

bool
FM_KeyPressEventMenuEdit(FM_Game *game, XEvent *event)
{
	KeySym keysym;

	keysym = XLookupKeysym(&event->xkey, 0);
	switch(keysym)
	{
		case XK_Up: keys[0] = true; break;
		case XK_Down: keys[1] = true; break;
		case XK_Left: keys[2] = true; break;
		case XK_Right: keys[3] = true; break;
	}
	return true;
}

bool
FM_KeyReleaseEventMenuEdit(FM_Game *game, XEvent *event)
{
	KeySym keysym;

	keysym = XLookupKeysym(&event->xkey, 0);
	switch(keysym)
	{
		case XK_Up: keys[0] = false; break;
		case XK_Down: keys[1] = false; break;
		case XK_Left: keys[2] = false; break;
		case XK_Right: keys[3] = false; break;
		case XK_c: checkpoints = chain_append(checkpoints, FM_GenCheckPoint(dBodyGetPosition(((FM_Machine *)machines->data)->body[0]))); break; case XK_v: checkpoints = chain_remove(checkpoints, chain_length(checkpoints) - 1); break;
		case XK_p: chain_scan(checkpoints, (scanner)mlPointPrint); break;
		case XK_Escape: FM_ExitMenuEdit(game); FM_EnterMenuMain(game); break;
	}
	return true;
}

bool
FM_UpdateMenuEdit(FM_Game *game)
{
	unsigned int i;
	
	FM_ControlActiveMachines(game);

	//FM_UpdateTrack(track, machines[0]);

	for(i = 0; i < 16; i++)
	{
		dSpaceCollide(game->play.space, game, &FM_NearCallback);
		dWorldStep(game->play.world, 0.01);
		dJointGroupEmpty(game->play.contactgroup);
	}

	FM_RenderMenuEdit(game);

	return true;
}

