// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "main.h"
#include "menumain.h"
#include "track.h"
#include "tl.h"
#include "il.h"
#include "sl.h"
#include <stdlib.h>
#include <stdio.h>
#include <GL/glx.h>
#include <GL/glu.h>
#include <AL/alc.h>
#include <AL/al.h>
#include "config.h"

static bool (*FM_Update)(FM_Game *game);
static bool (*FM_KeyPressEvent)(FM_Game *game, XEvent *event);
static bool (*FM_KeyReleaseEvent)(FM_Game *game, XEvent *event);
static void (*FM_Exit)(FM_Game *game);

static const char *tracknames[] = 
{
	"square",
	"circle",
	"triangle",
	"crayons",
	"garden",
	"abyss",
	"bathtub",
	"land"
};

static const char *machinenames[] = 
{
	"car",
	"formula1",
	"block"
};

void
FM_Init(FM_Game *game, int argc, char **argv)
{
	GLubyte *pixels;
	unsigned int width, height;
	GLenum gformat;
	ALvoid *data;
	ALsizei size, freq;
	ALenum aformat;
	ALuint music;

	srand(tlTicks());

	FT_Init_FreeType(&game->freetype);
	FT_New_Face(game->freetype, FM_FONT, 0, &game->face);
	FT_Set_Pixel_Sizes(game->face, 64, 64);

	glEnable(GL_COLOR_MATERIAL);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glEnable(GL_TEXTURE_2D);
	glClearColor(0, 0, 0.5, 0);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	FM_GL_3D(game);

	glGenTextures(1, &game->menu.background);
	glBindTexture(GL_TEXTURE_2D, game->menu.background);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	ilTexLoadMIME(FM_MENU_BACKGROUND, &pixels, &width, &height, &gformat);
	glTexImage2D(GL_TEXTURE_2D, 0, gformat, width, height, 0, gformat, GL_UNSIGNED_BYTE, pixels);
	free(pixels);

	btmLoad(&game->menu.selector, FM_MENU_SELECTOR);

	alGenSources(1, &game->menu.audiosources[FM_GAME_MENU_MUSIC]);
	alGenBuffers(1, &music);
	slBufferLoadMIME(FM_MENU_MUSIC, &size, &freq, &data, &aformat);
	alBufferData(music, aformat, data, size, freq);
	free(data);
	alSourcei(game->menu.audiosources[FM_GAME_MENU_MUSIC], AL_BUFFER, music);
	alSourcei(game->menu.audiosources[FM_GAME_MENU_MUSIC], AL_LOOPING, AL_TRUE);

	game->options.nmachines = 4;
	game->options.nplayers = 1;

	game->ntracknames = sizeof(tracknames) / sizeof(&tracknames[0]);
	game->tracknames = tracknames;

	game->nmachinenames = sizeof(machinenames) / sizeof(&machinenames[0]);
	game->machinenames = machinenames;

	game->options.track = rand() % game->ntracknames;
	game->options.machine = rand() % game->nmachinenames;

	game->play.world = dWorldCreate();
	game->play.space = dHashSpaceCreate(0);
	game->play.contactgroup = dJointGroupCreate(0);
	dWorldSetGravity(game->play.world, 0, -9.8, 0);
	game->play.track = NULL;

}

void
FM_Term(FM_Game *game)
{
	ALuint music;

	if(game->play.track) FM_FreeTrack(game->play.track);
	dJointGroupDestroy(game->play.contactgroup);
	dSpaceDestroy(game->play.space);
	dWorldDestroy(game->play.world);

	alSourceStop(game->menu.audiosources[FM_GAME_MENU_MUSIC]);

	alGetSourcei(game->menu.audiosources[FM_GAME_MENU_MUSIC], AL_BUFFER, (int*)&music);
	alDeleteSources(1, &game->menu.audiosources[FM_GAME_MENU_MUSIC]);
	alDeleteBuffers(1, &music);

	btmUnload(&game->menu.selector);

	glDeleteTextures(1, &game->menu.background);

	FT_Done_Face(game->face);
	FT_Done_FreeType(game->freetype);
}

void
FM_SetUpdateHandler(bool (*handler)(FM_Game *))
{
	FM_Update = handler;
}

void
FM_SetKeyPressEventHandler(bool (*handler)(FM_Game *, XEvent *))
{
	FM_KeyPressEvent = handler;
}

void
FM_SetKeyReleaseEventHandler(bool (*handler)(FM_Game *, XEvent *))
{
	FM_KeyReleaseEvent = handler;
}

void
FM_SetExitHandler(void (*handler)(FM_Game *))
{
	FM_Exit = handler;
}
void
FM_GL_2D(FM_Game *game)
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glViewport(0, 0, game->screen.w, game->screen.h);
	gluOrtho2D(0, game->screen.w, 0, game->screen.h);
	game->projection = FM_PROJECTION_2D;

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

void
FM_GL_3D(FM_Game *game)
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glViewport(0, 0, game->screen.w, game->screen.h);
	gluPerspective(45, (float)game->screen.w / game->screen.h, 1, 250);
	game->projection = FM_PROJECTION_3D;

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

static
void
FM_Resize(FM_Game *game, unsigned int width, unsigned int height)
{
	game->screen.w = width;
	game->screen.h = height;
	switch(game->projection)
	{
		case FM_PROJECTION_2D: FM_GL_2D(game); break;
		case FM_PROJECTION_3D: FM_GL_3D(game); break;
	}
}


int
main(int argc, char **argv)
{
	FM_Game game;

	Display *display;
	Window root;
	XVisualInfo *info;
	Window window;
	XSetWindowAttributes swa;
	Atom wm_protocols, wm_delete_window;
	GLXContext gcontext;
	GLint attributes[] = {GLX_RGBA, GLX_DEPTH_SIZE, 24, GLX_DOUBLEBUFFER, None};
	ALCdevice *device;
	ALCcontext *acontext;
	XEvent event;


	bool loop;
	unsigned int startticks, endticks, ticks;

	display = XOpenDisplay(NULL);
	if(display)
	{
		root = DefaultRootWindow(display);
		info = glXChooseVisual(display, 0, attributes);
		if(info)
		{
			swa.colormap = XCreateColormap(display, root, info->visual, AllocNone);
			swa.event_mask = ExposureMask | StructureNotifyMask | KeyPressMask | KeyReleaseMask;

			window = XCreateWindow(display, root, 0, 0, 640, 480, 0, info->depth, InputOutput, info->visual, CWColormap | CWEventMask, &swa);
			wm_delete_window = XInternAtom(display, "WM_DELETE_WINDOW", 0);
			XSetWMProtocols(display, window, &wm_delete_window, 1);
			wm_protocols = XInternAtom(display, "WM_PROTOCOLS", 0);
			XMapWindow(display, window);
			XStoreName(display, window, PACKAGE_STRING);
			gcontext = glXCreateContext(display, info, NULL, GL_TRUE);
			if(gcontext)
			{
				glXMakeCurrent(display, window, gcontext);
				device = alcOpenDevice(NULL);
				if(device)
				{
					acontext = alcCreateContext(device, NULL);
					if(acontext)
					{
						alcMakeContextCurrent(acontext);
						dInitODE();
						FM_Init(&game, argc, argv);
						FM_EnterMenuMain(&game);
						loop = true;
						while(loop)
						{
							startticks = tlTicks();
							loop = FM_Update(&game);
							glXSwapBuffers(display, window);
							while(loop && XPending(display))
							{
								XNextEvent(display, &event);
								switch(event.type)
								{
									//case Expose: glXSwapBuffers(display, window); break;
									case ConfigureNotify: FM_Resize(&game, event.xconfigure.width, event.xconfigure.height); break;
									case KeyPress: loop = FM_KeyPressEvent(&game, &event); break;
									case KeyRelease: loop = FM_KeyReleaseEvent(&game, &event); break;
									case ClientMessage: loop = !(event.xclient.message_type == wm_protocols && event.xclient.data.l[0] == wm_delete_window); break;
								}
							}
							endticks = tlTicks();
							ticks = endticks - startticks;
							if(ticks < 40) tlSleep(40 - ticks);
						}
						FM_Exit(&game);
						FM_Term(&game);
						dCloseODE();
						alcMakeContextCurrent(NULL);
						alcDestroyContext(acontext);
					}
					alcCloseDevice(device);
				}
				glXMakeCurrent(display, None, NULL);
			}
			glXDestroyContext(display, gcontext);
			XDestroyWindow(display, window);
		}
		XCloseDisplay(display);
	}
	return EXIT_SUCCESS;
}
