// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "iol.h"
#include <stdlib.h>
#include <string.h>
#include <bzlib.h>

static
BZSTREAM
*bzopenread(FILE *file)
{
	BZSTREAM *stream = malloc(sizeof(BZSTREAM));
	stream->file = BZ2_bzReadOpen(NULL, file, 0, 0, NULL, 0);
	stream->pos = 0;
	stream->read = 0;
	stream->eof = false;
	return stream;
}

static
void
bzclose(BZSTREAM *stream)
{
	BZ2_bzReadClose(NULL, stream->file);
	free(stream);
}

static
int
bzgetc(BZSTREAM *stream)
{
	int c, bzerror;
	if(stream->pos < stream->read) c = stream->buffer[stream->pos++];
	else
	{
		if(!stream->eof)
		{
			stream->read = BZ2_bzRead(&bzerror, stream->file, stream->buffer, sizeof(char)*BZSTREAMSIZE);
			stream->eof = bzerror == BZ_STREAM_END;
			stream->pos = 0;
			c = stream->buffer[stream->pos++];
		}
		else c = EOF;
	}
	return c;
}

static
void
bzungetc(int c, BZSTREAM *stream)
{
	stream->buffer[--stream->pos] = c;
}


IOLFILE *
iolfopenread(const char *filename)
{
	unsigned int len;

	IOLFILE *file = malloc(sizeof(IOLFILE));
	len = strlen(filename);
	file->raw = fopen(filename, "rb");
	if(!file->raw)
	{
		free(file);
		return NULL;
	}
	if(len > 4 && !strcmp(filename + len - 4, ".bz2")) file->compressed = bzopenread(file->raw);
	else file->compressed = NULL;
	return file;
}

void
iolfclose(IOLFILE *file)
{
	if(file->compressed) bzclose(file->compressed);
	fclose(file->raw);
	free(file);
}

int
iolfgetc(IOLFILE *file)
{
	int c;

	if(file->compressed) c = bzgetc(file->compressed);
	else c = getc(file->raw);
	return c;
}

void
iolungetc(int c, IOLFILE *file)
{
	if(file->compressed) bzungetc(c, file->compressed);
	else ungetc(c, file->raw);
}

unsigned int
iolgetline(char **line, unsigned int *n, IOLFILE *file)
{
	char buffer[BZSTREAMSIZE];
	int c;

	*n = 0;
	c = iolfgetc(file);

	if(c == EOF) return -1;

	while(c != '\n' && c != EOF)
	{
		buffer[(*n)++] = c;
		c = iolfgetc(file);
	}
	buffer[(*n)++] = '\0';

	*line = malloc(*n * sizeof(char));
	memcpy(*line, buffer, *n * sizeof(char));
	return *n - 1;
}
