// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "il.h"
#include "ml.h"
#include <stdio.h>
#include <stdlib.h>
#include <png.h>
#include <jpeglib.h>
#include <magic.h>
#include <GL/glu.h>

// Loader

int
ilTexLoad(ILenum loader, const char *filename, GLubyte **pixels, GLuint *width, GLuint *height, GLenum *format)
{
	int r;

	switch(loader)
	{
		case IL_PNG: r = ilTexLoadPNG(filename, pixels, width, height, format); break;
		case IL_JPEG: r = ilTexLoadJPEG(filename, pixels, width, height, format); break;
		case IL_MIME: r = ilTexLoadMIME(filename, pixels, width, height, format); break;
		default: r = -1; break;
	}
	return r;
}

// MIME

int
ilTexLoadMIME(const char *filename, GLubyte **pixels, GLuint *width, GLuint *height, GLenum *format)
{
	const char *mime;
	ILenum loader;
	magic_t cookie;

	loader = IL_MIME;
	cookie = magic_open(MAGIC_MIME_TYPE | MAGIC_NO_CHECK_COMPRESS | MAGIC_NO_CHECK_TAR);
	magic_load(cookie, NULL);
	mime = magic_file(cookie, filename);
	if(!strcmp(mime, "image/png")) loader = IL_PNG;
	else if(!strcmp(mime, "image/jpeg")) loader = IL_JPEG;
	magic_close(cookie);
	return loader != IL_MIME ? ilTexLoad(loader, filename, pixels, width, height, format) : 1;
}

// PNG

int
ilTexLoadPNG(const char *filename, GLubyte **pixels, GLuint *width, GLuint *height, GLenum *format)
{
	FILE *stream; 
	png_structp png; 
	png_infop info;
	unsigned char signature[8]; 
	unsigned long w, h;
	int depth, type, i;
	png_bytepp rows;

	*pixels = NULL;
	rows = NULL;

	stream = fopen(filename, "rb");
	if(!stream)
		return -1;

	fread(signature, 1, 8, stream);
	if(!png_check_sig(signature, 8))
	{
		fclose(stream);
		return -1;
	}
	png = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if(!png)
	{
		fclose(stream);
		return -2;
	}
	info = png_create_info_struct(png);
	if (!info)
	{
		png_destroy_read_struct(&png, (png_infopp) NULL, (png_infopp) NULL);
		fclose(stream);
		return -2;
	}
	if(setjmp(png_jmpbuf(png)))
	{
		png_destroy_read_struct(&png, &info, NULL);
		fclose(stream);
		return -1;
	}

	png_init_io(png, stream);
	png_set_sig_bytes(png, 8);
	png_read_info(png, info);

	png_get_IHDR(png, info, &w, &h, &depth, &type, NULL, NULL, NULL);
	*width = mlNextTwoPower(w);
	*height = mlNextTwoPower(h);

	*format = (type & PNG_COLOR_MASK_ALPHA) ? GL_RGBA : GL_RGB;

	if(depth > 8)
		png_set_strip_16(png);
	if(type == PNG_COLOR_TYPE_GRAY || type == PNG_COLOR_TYPE_GRAY_ALPHA)
		png_set_gray_to_rgb(png);
	if(type == PNG_COLOR_TYPE_PALETTE)
		png_set_palette_to_rgb(png);
	png_read_update_info(png, info);

	if(!(*pixels = malloc((*width)  * (*height) * ilSizeOfFormat(*format))))
	{
		png_destroy_read_struct(&png, &info, NULL);
		return -2;
	}

	if(!(rows = malloc(h * sizeof(png_bytep))))
	{
		png_destroy_read_struct(&png, &info, NULL);
		free(*pixels);
		*pixels = NULL;
		return -2;
	}
	for(i = 0; i < h; i++)
		rows[h - 1 - i] = *pixels + i * w * ilSizeOfFormat(*format);
	png_read_image(png, rows);
	free(rows);

	png_destroy_read_struct(&png, &info, NULL);
	fclose(stream);

	gluScaleImage(*format, w, h, GL_UNSIGNED_BYTE, *pixels, *width, *height, GL_UNSIGNED_BYTE, *pixels);

	return 0;
}

// JPEG

int
ilTexLoadJPEG(const char *filename, GLubyte **pixels, GLuint *width, GLuint *height, GLenum *format)
{
	
	FILE *stream; 
	struct jpeg_decompress_struct info;
	unsigned int w, h, i, row;
	unsigned char **rows;
	struct jpeg_error_mgr error;

	*pixels = NULL;
	rows = NULL;

	stream = fopen(filename, "rb");
	if(!stream)
		return -1;

	jpeg_create_decompress(&info);

	info.err = jpeg_std_error(&error);

	jpeg_stdio_src(&info, stream);

	jpeg_read_header(&info, TRUE);

	jpeg_start_decompress(&info);

	w = info.image_width;
	h = info.image_height;
	*width = mlNextTwoPower(w);
	*height = mlNextTwoPower(h);

	*format = GL_RGB;

	*pixels = malloc((*width) * (*height) * ilSizeOfFormat(*format));

	rows = malloc(h * sizeof(unsigned char *));
	for(i = 0; i < h; i++)
		rows[i] = *pixels + i * w * ilSizeOfFormat(*format);
	row = 0;
	while(info.output_scanline < info.output_height)
		row += jpeg_read_scanlines(&info, &rows[row], info.output_height - row);
	free(rows);

	jpeg_finish_decompress(&info);

	jpeg_destroy_decompress(&info);

	fclose(stream);

	gluScaleImage(*format, w, h, GL_UNSIGNED_BYTE, *pixels, *width, *height, GL_UNSIGNED_BYTE, *pixels);

	return 0;
}
