// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "cl.h"
#include <GL/glu.h>
#include <math.h>

#define mlNextTwoPower(i) (pow(2, ceil(log2(i))))
#define mlMax(a, b) ((a) > (b) ? (a) : (b))

void
clStringMetrics(FT_Face face, const wchar_t *string, unsigned int *w, unsigned int *h)
{
	const wchar_t *c;
	*w = *h = 0;
	for(c = string; *c; c++)
	{
		FT_Load_Char(face, *c, FT_LOAD_DEFAULT);
		*w += face->glyph->metrics.horiAdvance / 64;
		*h = mlMax(*h, face->glyph->metrics.height / 64);
	}
}

void
clTexGlyph2D(unsigned int w, unsigned int h, FT_GlyphSlot glyph)
{
	unsigned int i, j;
	GLubyte pixels[2 * w * h];

	memset(pixels, 0, w * h * sizeof(GLubyte));
	for(j = 0; j < glyph->bitmap.rows; j++)
        	for(i = 0; i < glyph->bitmap.width; i++)
		        pixels[2 * (j * w + i)] = pixels[2 * (j * w + i) + 1] = glyph->bitmap.buffer[(glyph->bitmap.rows - j - 1) * glyph->bitmap.pitch + i];
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, w, h, 0, GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, pixels);
}

void
clRenderTextureGlyph(FT_GlyphSlot glyph)
{
	unsigned int w, h;
	float tw, th;

	w = mlNextTwoPower(glyph->bitmap.width);
	h = mlNextTwoPower(glyph->bitmap.rows);

	clTexGlyph2D(w, h, glyph);

	tw = (float) glyph->bitmap.width / w;
	th = (float) glyph->bitmap.rows / h;

	w = glyph->metrics.width / 64;
	h = glyph->metrics.height / 64;

	glPushMatrix();
	glTranslatef((float) glyph->metrics.horiBearingX / 64, - (float) (glyph->metrics.height - glyph->metrics.horiBearingY) / 64, 0);
	glBegin(GL_QUADS);
	glTexCoord2f(0, 0); glVertex2f(0, 0);
	glTexCoord2f(0, th); glVertex2f(0, h);
	glTexCoord2f(tw, th); glVertex2f(w, h);
	glTexCoord2f(tw, 0); glVertex2f(w, 0);
	glEnd();
	glPopMatrix();
}

void
clRenderOutlineGlyph(FT_GlyphSlot glyph)
{
	unsigned int i, j;

	if(glyph->outline.n_contours)
	{
		glBegin(GL_LINE_LOOP);
		for(i = 0, j = 0; i < glyph->outline.n_points; i++)
		{
			glVertex2f((float) glyph->outline.points[i].x / 64, (float) glyph->outline.points[i].y / 64);
			if(i == glyph->outline.contours[j])
			{
				glEnd();
				if(j < glyph->outline.n_contours - 1)
				{
					j++;
					glBegin(GL_LINE_LOOP);
				}
			}
		}
	}
}

void
clRenderWireGlyph(FT_GlyphSlot glyph, unsigned int z)
{
	unsigned int i, j;

	glPushMatrix();
	glTranslatef(0, 0, - (float) z);
	clRenderOutlineGlyph(glyph);
	glPopMatrix();

	if(glyph->outline.n_contours)
	{
		glBegin(GL_LINES);
		for(i = 0, j = 0; i < glyph->outline.n_points; i++)
		{
			glVertex3f((float) glyph->outline.points[i].x / 64, (float) glyph->outline.points[i].y / 64, - (float) z);
			glVertex3f((float) glyph->outline.points[i].x / 64, (float) glyph->outline.points[i].y / 64, 0);
		}
		glEnd();
	}

	clRenderOutlineGlyph(glyph);
}

void
clRenderFillGlyph(FT_GlyphSlot glyph)
{
	unsigned int i, j;
	GLUtesselator *tess;
	GLdouble vertices[glyph->outline.n_points][3];

	tess = gluNewTess();
	gluTessCallback(tess, GLU_TESS_VERTEX, (GLvoid (*) ( )) &glVertex3dv);
	gluTessCallback(tess, GLU_TESS_BEGIN, (GLvoid (*) ( )) &glBegin);
	gluTessCallback(tess, GLU_TESS_END, (GLvoid (*) ( )) &glEnd);

	gluTessBeginPolygon(tess, NULL);
	if(glyph->outline.n_contours)
	{
		gluTessBeginContour(tess);
		for(i = 0, j = 0; i < glyph->outline.n_points; i++)
		{
			vertices[i][0] = (double) glyph->outline.points[i].x / 64;
			vertices[i][1] = (double) glyph->outline.points[i].y / 64;
			vertices[i][2] = 0;
			gluTessVertex(tess, vertices[i], vertices[i]);
			if(i == glyph->outline.contours[j])
			{
				gluTessEndContour(tess);
				if(j < glyph->outline.n_contours - 1)
				{
					j++;
					gluTessBeginContour(tess);
				}
			}
		}
	}
	gluTessEndPolygon(tess);

	gluDeleteTess(tess);
}

void
clRenderSolidGlyph(FT_GlyphSlot glyph, unsigned int z)
{
	unsigned int i, j;

	glPushMatrix();
	glTranslatef(0, 0, - (float) z);
	clRenderFillGlyph(glyph);
	glPopMatrix();

	if(glyph->outline.n_contours)
	{
		glBegin(GL_QUAD_STRIP);
		for(i = 0, j = 0; i < glyph->outline.n_points; i++)
		{
			glVertex3f((float) glyph->outline.points[i].x / 64, (float) glyph->outline.points[i].y / 64, - (float) z);
			glVertex3f((float) glyph->outline.points[i].x / 64, (float) glyph->outline.points[i].y / 64, 0);
			if(i == glyph->outline.contours[j])
			{
				glEnd();
				if(j < glyph->outline.n_contours - 1)
				{
					j++;
					glBegin(GL_QUAD_STRIP);
				}
			}
		}
		glEnd();
	}

	clRenderFillGlyph(glyph);
}

void
clRenderGlyph(FT_GlyphSlot glyph, CLenum mode, unsigned int z)
{
	switch(mode)
	{
		case CL_TEXTURE: clRenderTextureGlyph(glyph); break;
		case CL_OUTLINE: clRenderOutlineGlyph(glyph); break;
		case CL_WIRE: clRenderWireGlyph(glyph, z); break;
		case CL_FILL: clRenderFillGlyph(glyph); break;
		case CL_SOLID: clRenderSolidGlyph(glyph, z); break;
	}
}

void
clRenderString(FT_Face face, CLenum mode, const wchar_t *string, unsigned int z)
{
	const wchar_t *c;
	FT_Int32 flags;

	flags = mode == CL_TEXTURE ? FT_LOAD_RENDER : FT_LOAD_DEFAULT;

	glPushMatrix();
	for(c = string; *c; c++)
	{
		FT_Load_Char(face, *c, flags);
		clRenderGlyph(face->glyph, mode, z);
		glTranslatef((float) face->glyph->metrics.horiAdvance / 64, 0, 0);
	}
	glPopMatrix();
}
