// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

// Based on the tutorial "Frustum Culling in OpenGL" by Mark Morley, December 200.

#include "camera.h"
#include <math.h>
#include <GL/gl.h>

void
FM_CameraGLFrustumLoad(FM_Camera *camera)
{
	float proj[16];
	float modl[16];
	float clip[16];
	float t;

	/* Get the current PROJECTION matrix from OpenGL */
	glGetFloatv( GL_PROJECTION_MATRIX, proj );

	/* Get the current MODELVIEW matrix from OpenGL */
	glGetFloatv( GL_MODELVIEW_MATRIX, modl );

	/* Combine the two matrices (multiply projection by modelview) */
	clip[ 0] = modl[ 0] * proj[ 0] + modl[ 1] * proj[ 4] + modl[ 2] * proj[ 8] + modl[ 3] * proj[12];
	clip[ 1] = modl[ 0] * proj[ 1] + modl[ 1] * proj[ 5] + modl[ 2] * proj[ 9] + modl[ 3] * proj[13];
	clip[ 2] = modl[ 0] * proj[ 2] + modl[ 1] * proj[ 6] + modl[ 2] * proj[10] + modl[ 3] * proj[14];
	clip[ 3] = modl[ 0] * proj[ 3] + modl[ 1] * proj[ 7] + modl[ 2] * proj[11] + modl[ 3] * proj[15];

	clip[ 4] = modl[ 4] * proj[ 0] + modl[ 5] * proj[ 4] + modl[ 6] * proj[ 8] + modl[ 7] * proj[12];
	clip[ 5] = modl[ 4] * proj[ 1] + modl[ 5] * proj[ 5] + modl[ 6] * proj[ 9] + modl[ 7] * proj[13];
	clip[ 6] = modl[ 4] * proj[ 2] + modl[ 5] * proj[ 6] + modl[ 6] * proj[10] + modl[ 7] * proj[14];
	clip[ 7] = modl[ 4] * proj[ 3] + modl[ 5] * proj[ 7] + modl[ 6] * proj[11] + modl[ 7] * proj[15];

	clip[ 8] = modl[ 8] * proj[ 0] + modl[ 9] * proj[ 4] + modl[10] * proj[ 8] + modl[11] * proj[12];
	clip[ 9] = modl[ 8] * proj[ 1] + modl[ 9] * proj[ 5] + modl[10] * proj[ 9] + modl[11] * proj[13];
	clip[10] = modl[ 8] * proj[ 2] + modl[ 9] * proj[ 6] + modl[10] * proj[10] + modl[11] * proj[14];
	clip[11] = modl[ 8] * proj[ 3] + modl[ 9] * proj[ 7] + modl[10] * proj[11] + modl[11] * proj[15];

	clip[12] = modl[12] * proj[ 0] + modl[13] * proj[ 4] + modl[14] * proj[ 8] + modl[15] * proj[12];
	clip[13] = modl[12] * proj[ 1] + modl[13] * proj[ 5] + modl[14] * proj[ 9] + modl[15] * proj[13];
	clip[14] = modl[12] * proj[ 2] + modl[13] * proj[ 6] + modl[14] * proj[10] + modl[15] * proj[14];
	clip[15] = modl[12] * proj[ 3] + modl[13] * proj[ 7] + modl[14] * proj[11] + modl[15] * proj[15];

	/* Extract the numbers for the RIGHT plane */
	camera->frustum[0][0] = clip[ 3] - clip[ 0];
	camera->frustum[0][1] = clip[ 7] - clip[ 4];
	camera->frustum[0][2] = clip[11] - clip[ 8];
	camera->frustum[0][3] = clip[15] - clip[12];

	/* Normalize the result */
	t = sqrt( camera->frustum[0][0] * camera->frustum[0][0] + camera->frustum[0][1] * camera->frustum[0][1] + camera->frustum[0][2] * camera->frustum[0][2] );
	camera->frustum[0][0] /= t;
	camera->frustum[0][1] /= t;
	camera->frustum[0][2] /= t;
	camera->frustum[0][3] /= t;

	/* Extract the numbers for the LEFT plane */
	camera->frustum[1][0] = clip[ 3] + clip[ 0];
	camera->frustum[1][1] = clip[ 7] + clip[ 4];
	camera->frustum[1][2] = clip[11] + clip[ 8];
	camera->frustum[1][3] = clip[15] + clip[12];

	/* Normalize the result */
	t = sqrt( camera->frustum[1][0] * camera->frustum[1][0] + camera->frustum[1][1] * camera->frustum[1][1] + camera->frustum[1][2] * camera->frustum[1][2] );
	camera->frustum[1][0] /= t;
	camera->frustum[1][1] /= t;
	camera->frustum[1][2] /= t;
	camera->frustum[1][3] /= t;

	/* Extract the BOTTOM plane */
	camera->frustum[2][0] = clip[ 3] + clip[ 1];
	camera->frustum[2][1] = clip[ 7] + clip[ 5];
	camera->frustum[2][2] = clip[11] + clip[ 9];
	camera->frustum[2][3] = clip[15] + clip[13];

	/* Normalize the result */
	t = sqrt( camera->frustum[2][0] * camera->frustum[2][0] + camera->frustum[2][1] * camera->frustum[2][1] + camera->frustum[2][2] * camera->frustum[2][2] );
	camera->frustum[2][0] /= t;
	camera->frustum[2][1] /= t;
	camera->frustum[2][2] /= t;
	camera->frustum[2][3] /= t;

	/* Extract the TOP plane */
	camera->frustum[3][0] = clip[ 3] - clip[ 1];
	camera->frustum[3][1] = clip[ 7] - clip[ 5];
	camera->frustum[3][2] = clip[11] - clip[ 9];
	camera->frustum[3][3] = clip[15] - clip[13];

	/* Normalize the result */
	t = sqrt( camera->frustum[3][0] * camera->frustum[3][0] + camera->frustum[3][1] * camera->frustum[3][1] + camera->frustum[3][2] * camera->frustum[3][2] );
	camera->frustum[3][0] /= t;
	camera->frustum[3][1] /= t;
	camera->frustum[3][2] /= t;
	camera->frustum[3][3] /= t;

	/* Extract the FAR plane */
	camera->frustum[4][0] = clip[ 3] - clip[ 2];
	camera->frustum[4][1] = clip[ 7] - clip[ 6];
	camera->frustum[4][2] = clip[11] - clip[10];
	camera->frustum[4][3] = clip[15] - clip[14];

	/* Normalize the result */
	t = sqrt( camera->frustum[4][0] * camera->frustum[4][0] + camera->frustum[4][1] * camera->frustum[4][1] + camera->frustum[4][2] * camera->frustum[4][2] );
	camera->frustum[4][0] /= t;
	camera->frustum[4][1] /= t;
	camera->frustum[4][2] /= t;
	camera->frustum[4][3] /= t;

	/* Extract the NEAR plane */
	camera->frustum[5][0] = clip[ 3] + clip[ 2];
	camera->frustum[5][1] = clip[ 7] + clip[ 6];
	camera->frustum[5][2] = clip[11] + clip[10];
	camera->frustum[5][3] = clip[15] + clip[14];

	/* Normalize the result */
	t = sqrt( camera->frustum[5][0] * camera->frustum[5][0] + camera->frustum[5][1] * camera->frustum[5][1] + camera->frustum[5][2] * camera->frustum[5][2] );
	camera->frustum[5][0] /= t;
	camera->frustum[5][1] /= t;
	camera->frustum[5][2] /= t;
	camera->frustum[5][3] /= t;
}

bool
FM_CameraFrustumPoint(const FM_Camera *camera, const float *point)
{
	unsigned int f;

	for(f = 0; f < 6; f++)
		if(camera->frustum[f][0] * point[0] + camera->frustum[f][1] * point[1] + camera->frustum[f][2] * point[2] + camera->frustum[f][3] <= 0)
			return false;
	return true;
}

bool
FM_CameraFrustumSphere(const FM_Camera *camera, float *sphere)
{
	unsigned int p;

	for(p = 0; p < 6; p++)
		if(camera->frustum[p][0] * sphere[0] + camera->frustum[p][1] * sphere[1] + camera->frustum[p][2] * sphere[2] + camera->frustum[p][3] < -sphere[3])
			return true;
	return false;
}

bool
FM_CameraFrustumTriangle(const FM_Camera *camera, float *triangle)
{
	unsigned int f, p;

	for(f = 0; f < 6; f++)
	{
		for(p = 0; p < 3; p++)
		{
			if(camera->frustum[f][0] * triangle[p * 3 + 0] + camera->frustum[f][1] * triangle[p * 3 + 1] + camera->frustum[f][2] * triangle[p * 3 + 2] + camera->frustum[f][3] > 0)
				break;
		}
		if(p == 3) return false;
	}
	return true;
}

bool
FM_CameraFrustumPolygon(const FM_Camera *camera, int npoints, float *points)
{
	unsigned int f, p;

	for(f = 0; f < 6; f++)
	{
		for(p = 0; p < npoints; p++)
		{
			if(camera->frustum[f][0] * points[p * 3] + camera->frustum[f][1] * points[p * 3 + 1] + camera->frustum[f][2] * points[p * 3 + 2] + camera->frustum[f][3] > 0)
			break;
		}
		if(p == npoints) return false;
	}
	return true;
}

bool
FM_CameraFrustumCube(const FM_Camera *camera, float *cube)
{
	unsigned int p;

	for(p = 0; p < 6; p++)
	{
		if(camera->frustum[p][0] * (cube[0] - cube[3]) + camera->frustum[p][1] * (cube[1] - cube[3]) + camera->frustum[p][2] * (cube[2] - cube[3]) + camera->frustum[p][3] > 0)
			continue;
		if(camera->frustum[p][0] * (cube[0] + cube[3]) + camera->frustum[p][1] * (cube[1] - cube[3]) + camera->frustum[p][2] * (cube[2] - cube[3]) + camera->frustum[p][3] > 0)
			continue;
		if(camera->frustum[p][0] * (cube[0] - cube[3]) + camera->frustum[p][1] * (cube[1] + cube[3]) + camera->frustum[p][2] * (cube[2] - cube[3]) + camera->frustum[p][3] > 0)
			continue;
		if(camera->frustum[p][0] * (cube[0] + cube[3]) + camera->frustum[p][1] * (cube[1] + cube[3]) + camera->frustum[p][2] * (cube[2] - cube[3]) + camera->frustum[p][3] > 0)
			continue;
		if(camera->frustum[p][0] * (cube[0] - cube[3]) + camera->frustum[p][1] * (cube[1] - cube[3]) + camera->frustum[p][2] * (cube[2] + cube[3]) + camera->frustum[p][3] > 0)
			continue;
		if(camera->frustum[p][0] * (cube[0] + cube[3]) + camera->frustum[p][1] * (cube[1] - cube[3]) + camera->frustum[p][2] * (cube[2] + cube[3]) + camera->frustum[p][3] > 0)
			continue;
		if(camera->frustum[p][0] * (cube[0] - cube[3]) + camera->frustum[p][1] * (cube[1] + cube[3]) + camera->frustum[p][2] * (cube[2] + cube[3]) + camera->frustum[p][3] > 0)
			continue;
		if(camera->frustum[p][0] * (cube[0] + cube[3]) + camera->frustum[p][1] * (cube[1] + cube[3]) + camera->frustum[p][2] * (cube[2] + cube[3]) + camera->frustum[p][3] > 0)
			continue;
		return false;
	}
	return true;
}
