// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "track.h"
#include "main.h"
#include "btm.h"
#include "ml.h"

static
void
FM_LoadTrackCheckpoints(FM_Track *track, const char *filename)
{
	FILE *stream;
	char buffer[128];
	float *checkpoint;

	stream = fopen(filename, "r");

	fscanf(stream, "length %i", &track->ncheckpoints);

	track->checkpoints = malloc(sizeof(float) * 3 * track->ncheckpoints);

	checkpoint = track->checkpoints;
	while(fgets(buffer, 128, stream))
	{
		fscanf(stream, "(%f, %f, %f)", &checkpoint[0], &checkpoint[1], &checkpoint[2]);
		checkpoint += 3;
	}

	fclose(stream);
}

static
void
FM_FreeTrackCheckpoints(FM_Track *track)
{
	free(track->checkpoints);
}

static
void
FM_ODE_LoadTrack(FM_Track *track, dSpaceID space)
{
	unsigned int i, j, k, ntriangles, accum;

	ntriangles = 0;
	for(i = 0; i < track->model.nobjects; i++)
		ntriangles += track->model.objects[i].mesh.nfaces;

	track->vertices = malloc(ntriangles * 3 * sizeof(dVector3));
	track->indices = malloc(ntriangles * 3 * sizeof(int));

	accum = 0;
	for(k = 0; k < track->model.nobjects; k++)
	{
		for(j = 0; j < track->model.objects[k].mesh.nfaces; j++)
		{
			for(i = 0; i < 3; i++)
			{
				track->indices[(accum + j) * 3 + i] = (accum + j) * 3 + i;
				mlPointCopy(track->model.objects[k].mesh.vertices[track->model.objects[k].mesh.faces[j].verts[i]].co, track->vertices[((accum + j) * 3) + i]);
			}
		}
		accum += j;
	}

	track->trimeshdata = dGeomTriMeshDataCreate();
	dGeomTriMeshDataBuildSimple(track->trimeshdata, (dReal *)track->vertices, ntriangles * 3, track->indices, ntriangles * 3);
	track->geom = dCreateTriMesh(space, track->trimeshdata, NULL, NULL, NULL);

}

static
void
FM_ODE_FreeTrack(FM_Track *track)
{
	dGeomTriMeshDataDestroy(track->trimeshdata);
	dGeomDestroy(track->geom);

	free(track->vertices);
	free(track->indices);
}

FM_Track *
FM_LoadTrack(dSpaceID space, const char *name)
{
	FM_Track *track;
	char buffer[128];

	track = malloc(sizeof(FM_Track));

	sprintf(buffer, PACKAGE_DATADIR "/tracks/%s.btm.bz2", name);
	btmLoad(&track->model, buffer);

	FM_ODE_LoadTrack(track, space);

	sprintf(buffer, PACKAGE_DATADIR "/tracks/%s.ckp", name);
	FM_LoadTrackCheckpoints(track, buffer);

	return track;
}

void
FM_FreeTrack(FM_Track *track)
{
	FM_FreeTrackCheckpoints(track);

	FM_ODE_FreeTrack(track);

	btmUnload(&track->model);

	free(track);
}

void
FM_UpdateTrack(FM_Track *track, FM_Machine *machine)
{
}

void
FM_RenderTrack(const FM_Track *track, const RILcamera *camera)
{
	btmRender(&track->model);
//	btmRenderWithCamera(camera, &track->model);
}

