// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "menuplay.h"
#include <ode/ode.h>
#include <GL/glu.h>
#include "main.h"
#include "menumain.h"
#include "machine.h"
#include "track.h"
#include "rilcamera.h"
#include "rilfont.h"
#include "list.h"
#include "set.h"
#include "ml.h"

dWorldID world;
dSpaceID space;
dJointGroupID contactgroup;

static RILfont *font;
static RILcamera camera;
static FM_Track *track;

static list *machines;
static set *activemachines;

static const char *fontfilename = PACKAGE_DATADIR "/VeraBd.ttf";

#define MAX_CONTACTS 1000

static
void
FM_NearCallback(void *data, dGeomID o1, dGeomID o2)
{
	dBodyID b1, b2;
	dContact contact[MAX_CONTACTS];
	dJointID c;
	unsigned int i;
	int n;
	void *d1, *d2;

	b1 = dGeomGetBody(o1);
	b2 = dGeomGetBody(o2);
	if(b1 && b2 && dAreConnectedExcluding(b1, b2, dJointTypeContact))
		return;

	d1 = dGeomGetData(o1);
	d2 = dGeomGetData(o2);

	if(set_contains(machines, d1) && set_contains(machines, d2) && (!set_contains(activemachines, d1) || !set_contains(activemachines, d2)))
		return;

	n = dCollide(o1, o2, MAX_CONTACTS, &contact[0].geom, sizeof(dContact));
	if(n > 0)
	{
		for(i = 0; i < n; i++)
		{
			contact[i].surface.mode = dContactSlip1 | dContactSlip2 | dContactSoftERP | dContactSoftCFM | dContactApprox1 | dContactMu2;
			contact[i].surface.mu = 0.99;
			contact[i].surface.mu2 = 0.99;
			contact[i].surface.slip1 = 0.01;
			contact[i].surface.slip2 = 0.01;
			contact[i].surface.soft_erp = 0.5;
			contact[i].surface.soft_cfm = 0.3;
			c = dJointCreateContact(world,contactgroup,&contact[i]);
			dJointAttach(c, dGeomGetBody(contact[i].geom.g1), dGeomGetBody(contact[i].geom.g2));
		}
	}
}

static
void
FM_ActivateMachinesScanner(FM_Machine *machine)
{
	activemachines = set_add(activemachines, machine);
}

static
void
FM_ActivateMachines()
{
	list_scan(machines, (list_scanner)FM_ActivateMachinesScanner);
}

void
FM_EnterMenuPlay(FM_Game *game)
{
	float pos[3];
	unsigned int i;

	alSourceStop(game->menuaudiosources[FM_GAME_MENU_MUSIC]);

	font = rilFontLoad(fontfilename, RIL_FONT_BITMAP);

	world = dWorldCreate();
	space = dHashSpaceCreate(0);
	contactgroup = dJointGroupCreate(0);
	dWorldSetGravity(world, 0, -9.8, 0);

	mlPointLoad(camera.object, 0, 0, 0);
	alListener3f(AL_POSITION, camera.object[0], 75, camera.object[2]);

	track = FM_LoadTrack(space, game->tracknames[game->track]);

	machines = activemachines = NULL;

	for(i = 0; i < game->nmachines; i++)
	{
		mlPointLoad(pos, (i - game->nmachines / 2.0) * 5.0, 5, 0);
		machines = list_append(machines, FM_LoadMachine(world, space, 75, 20, pos, -90, game->machinenames[game->machine]));
	}

	FM_ActivateMachines();

	FM_SetEventHandler(FM_EventMenuPlay);
	FM_SetUpdateHandler(FM_UpdateMenuPlay);
}

void
FM_ExitMenuPlay(FM_Game *game)
{
	rilFontUnload(font);

	list_scan(machines, (list_scanner)FM_FreeMachine);
	list_free(machines);

	FM_FreeTrack(track);

	dJointGroupDestroy(contactgroup);
	dSpaceDestroy(space);
	dWorldDestroy(world);
}

static
void
FM_UpdateMachineScanner(FM_Machine *machine, const RILcamera *camera)
{
	if(!FM_UpdateMachine(machine, camera))
	{
		activemachines = set_remove(activemachines, machine);
		alSourceStop(machine->source);
	}
}

static
FM_Machine *
FM_FirstMachine()
{
	FM_Machine *machine, *firstmachine;
	list *l;

	firstmachine = NULL;
	if(activemachines)
	{
		firstmachine = activemachines->data;
		for(l = activemachines->next; l; l = l->next)
		{
			machine = l->data;
			if(machine->checkpoint > firstmachine->checkpoint || (machine->checkpoint == firstmachine->checkpoint && machine->checkpointdistance < firstmachine->checkpointdistance))
				firstmachine = machine;
		}
	}
	return firstmachine;
}

static
void
FM_RenderMenuPlay(FM_Game *game)
{
	FM_Machine *firstmachine;
	float a[3], b[3];
	const dReal *p, *v;
	char text[2];

	firstmachine = FM_FirstMachine();

	if(!firstmachine)
		return;

       	p = dBodyGetPosition(firstmachine->body[0]);
	v = dBodyGetLinearVel(firstmachine->body[0]);

	mlVectorSub(p, camera.object, a);
	mlVectorScalarProduct(a, 0.15, b);
	mlVectorAddInPlace(camera.object, b);

	alListener3f(AL_POSITION, camera.object[0], 70, camera.object[2]);
	alListenerfv(AL_ORIENTATION, mlVectorNegY);
	alListenerfv(AL_VELOCITY, v);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glLoadIdentity();

	gluLookAt(camera.object[0], 70, camera.object[2], camera.object[0], camera.object[1], camera.object[2], 0, 0, -1);
	rilCameraGLFrustumLoad(&camera);

	list_user_scan(activemachines, (list_user_scanner)FM_UpdateMachineScanner, &camera);
	FM_RenderTrack(track, &camera);
	list_scan(activemachines, (list_scanner)FM_RenderMachine);

	FM_GL_2D(game);
	rilFontFaceSize(font, 43);
	sprintf(text, "%i", set_size(activemachines));
	glRasterPos2i(0, 0);
	rilFontRender(font, text);
	FM_GL_3D(game);

	SDL_GL_SwapBuffers();
}

static
void
FM_ComputerControlMachine(FM_Machine *machine)
{
	float position[3], target[3], direction[3], orientation[3], plane[4], angle, variation, distance;
	const dReal *r;
	int steer, gas;

	variation = (float)rand() / RAND_MAX;

	mlPointCopy(dBodyGetPosition(machine->body[0]), position);
	mlPointCopy(&track->checkpoints[(machine->checkpoint + 1) % (track->ncheckpoints) * 3], target);

	mlVectorSub(target, position, direction);

	mlVectorNorm(direction, &machine->checkpointdistance);

	if(machine->checkpointdistance < 20)
	{
		machine->checkpoint++;

		mlPointCopy(&track->checkpoints[(machine->checkpoint + 1) % (track->ncheckpoints) * 3], target);

		mlVectorSub(target, position, direction);

		mlVectorNorm(direction, &machine->checkpointdistance);
	}

	mlVectorNormalInPlace(direction);

	r = dBodyGetRotation(machine->body[0]);
	orientation[0] = -r[2];
	orientation[1] = -r[6];
	orientation[2] = -r[10];

	mlVectorCrossProduct(mlVectorY, orientation, plane);
	plane[3] = -mlVectorDotProductFunc(plane, position);

	distance = mlVectorDotProductFunc(plane, target) + plane[3];

	mlVectorAngle(orientation, direction, &angle);

	if(distance > 10.0 * (0.5 + variation * 0.5)) steer = 1;
	else if(distance < -10.0 * (0.5 + variation * 0.5)) steer = -1;
	else steer = 0;

	if(angle > 75)
	{
		const dReal *v;

		v = dBodyGetLinearVel(machine->body[0]);

		if(machine->checkpointdistance < 20 * 2)
		{
		       if(mlVectorNormFunc(v) > 2) gas = 0;
		       else gas = 1;
		}
		else { gas = -1; steer *= -1; }
	}
	else gas = 1;

	FM_ControlMachine(machine, gas == 1, gas == -1, steer == 1, steer == -1);
}

static
void
FM_HumanControlMachine(FM_Machine *machine, bool up, bool down, bool left, bool right)
{
	float position[3], target[3], direction[3];

	mlPointCopy(dBodyGetPosition(machine->body[0]), position);
	mlPointCopy(&track->checkpoints[(machine->checkpoint + 1) % (track->ncheckpoints) * 3], target);

	mlVectorSub(target, position, direction);

	mlVectorNorm(direction, &machine->checkpointdistance);

	if(machine->checkpointdistance < 40)
	{
		machine->checkpoint++;

		mlPointCopy(&track->checkpoints[(machine->checkpoint + 1) % (track->ncheckpoints) * 3], target);

		mlVectorSub(target, position, direction);

		mlVectorNorm(direction, &machine->checkpointdistance);
	}

	FM_ControlMachine(machine, up, down, left, right);
}

static
void
FM_ControlActiveMachines(FM_Game *game)
{
	list *l;
	unsigned int i;
	FM_Machine *machine;

	for(i = 0, l = machines; l; l = l->next, i++)
	{
		machine = l->data;
		if(set_contains(activemachines, machine))
		{
			if(i < game->nplayers)
				switch(i)
				{
					case 0:
						FM_HumanControlMachine(machine, game->keys[SDLK_UP], game->keys[SDLK_DOWN], game->keys[SDLK_LEFT], game->keys[SDLK_RIGHT]);
						break;
					case 1:
						FM_HumanControlMachine(machine, game->keys[SDLK_w], game->keys[SDLK_s], game->keys[SDLK_a], game->keys[SDLK_d]);
						break;
					case 2:
						FM_HumanControlMachine(machine, game->keys[SDLK_i], game->keys[SDLK_k], game->keys[SDLK_j], game->keys[SDLK_l]);
						break;
					case 3:
						FM_HumanControlMachine(machine, game->keys[SDLK_t], game->keys[SDLK_g], game->keys[SDLK_f], game->keys[SDLK_h]);
						break;
				}
			else
				FM_ComputerControlMachine(machine);
		}
	}
}

bool
FM_UpdateMenuPlay(FM_Game *game)
{
	unsigned int i;
	
	FM_ControlActiveMachines(game);

	for(i = 0; i < 16; i++)
	{
		dSpaceCollide(space, NULL, &FM_NearCallback);
		dWorldStep(world, 0.01);
		dJointGroupEmpty(contactgroup);
	}

	FM_RenderMenuPlay(game);

	return false;
}

bool
FM_EventMenuPlay(FM_Game *game, SDL_Event *event)
{
	bool exitrequest = false;
	switch(event->type)
	{
		case SDL_KEYUP:
			switch(event->key.keysym.sym)
			{
				case SDLK_ESCAPE:
					FM_ExitMenuPlay(game);
					FM_EnterMenuMain(game);
					break;
				default:
					break;
			}
			break;
		case SDL_QUIT:
			FM_ExitMenuPlay(game);
			exitrequest = true;
			break;
	}
	return exitrequest;
}
