// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

// Based on splayer.c from OggTheora.

#include "menuintro.h"
#include <theora/theora.h>
#include <SDL.h>
#include <GL/glu.h>
#include "ml.h"
#include "rilmedia.h"
#include "menumain.h"

static const char *videofilename = PACKAGE_DATADIR "/intro.ogv";
static const char *audiofilename = PACKAGE_DATADIR "/intro.oga";
static GLuint video;
static ALuint audiobuffer, audiosource;

static ogg_sync_state oy; 
static ogg_page og;
static ogg_stream_state to;
static theora_info ti;
static theora_comment tc;
static theora_state td;

static int theora_p = 0;
static int stateflag = 0;

FILE * infile = NULL;

static bool videobuf_ready = false;
static ogg_int64_t  videobuf_granulepos = -1;
static double videobuf_time = 0;
static GLubyte *rgb;

static bool hasdatatobuffer = true;
static bool playbackdone = false;
static double now, delay, last_frame_time = 0;

static void
FM_VideoWrite(FM_Game *game)
{
	yuv_buffer yuv;
	int i, j;
	float Y, U, V;
	int chroma;

	theora_decode_YUVout(&td, &yuv);

	chroma = yuv.y_width / yuv.uv_width;

	for(i = 0; i < ti.frame_width; i++)
	{
		for(j = 0; j < ti.frame_height; j++)
		{
			Y = *(yuv.y + yuv.y_stride * j + i);
			U = *(yuv.u + (yuv.uv_stride * (j / chroma) + (i / chroma)));
			V = *(yuv.v + (yuv.uv_stride * (j / chroma) + (i / chroma)));

			rgb[((ti.frame_height - j - 1) * 512 + i) * 3 + 0] = mlClampf(1.164 * (Y - 16) + 1.596 * (V - 128), 0, 255);
			rgb[((ti.frame_height - j - 1) * 512 + i) * 3 + 1] = mlClampf(1.164 * (Y - 16) - 0.813 * (V - 128) - 0.391 * (U - 128), 0, 255);
			rgb[((ti.frame_height - j - 1) * 512 + i) * 3 + 2] = mlClampf(1.164 * (Y - 16) + 2.018 * (U - 128), 0, 255);
		}
	}

	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, 512, 512, 0, GL_RGB, GL_UNSIGNED_BYTE, rgb);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glBegin(GL_QUADS);
		glTexCoord2f(0, 0); glVertex2f(0, 0);
		glTexCoord2f((float)ti.frame_width / 512, 0); glVertex2f(game->screen->w, 0);
		glTexCoord2f((float)ti.frame_width / 512, (float)ti.frame_height / 512); glVertex2f(game->screen->w, game->screen->h);
		glTexCoord2f(0, (float)ti.frame_height / 512); glVertex2f(0, game->screen->h);
	glEnd();

	SDL_GL_SwapBuffers();
}

static
int
FM_BufferData(ogg_sync_state *oy)
{
	char *buffer = ogg_sync_buffer(oy, 4096);
	int bytes = fread(buffer, 1, 4096, infile);
	ogg_sync_wrote(oy, bytes);
	return(bytes);
}

static
int
FM_QueuePage(ogg_page *page)
{
	if(theora_p)
		ogg_stream_pagein(&to,page);
	return 0;
}                                   

static
void
FM_ParseHeaders()
{
	ogg_packet op;
	while(!stateflag)
	{
		int ret = FM_BufferData(&oy);
		if(!ret) break;
		while(ogg_sync_pageout(&oy,&og)>0)
		{
			ogg_stream_state test;
      
			if(!ogg_page_bos(&og))
			{
				FM_QueuePage(&og);
				stateflag=1;
				break;
			}
      
			ogg_stream_init(&test,ogg_page_serialno(&og));
			ogg_stream_pagein(&test,&og);
			ogg_stream_packetout(&test,&op);
      
			if(!theora_p && theora_decode_header(&ti, &tc, &op) >= 0)
			{
				memcpy(&to, &test, sizeof(test));
				theora_p = 1;
			}
			else
			{
				ogg_stream_clear(&test);
			}
		}
	}

	while((theora_p && theora_p < 3))
	{
		int ret;

		while(theora_p && (theora_p < 3) && (ret = ogg_stream_packetout(&to, &op)))
		{
			if(ret < 0)
			{
				fprintf(stderr, "Theora: error parsing Theora stream headers; corrupt stream?\n");
				exit(1);
			}
			if(theora_decode_header(&ti, &tc, &op))
			{
				fprintf(stderr, "Theora: error parsing Theora stream headers; corrupt stream?\n");
				exit(1);
			}
			theora_p++;
			if(theora_p == 3) break;
		}

		if(ogg_sync_pageout(&oy,&og)>0)
			FM_QueuePage(&og);
		else
		{
			int ret = FM_BufferData(&oy);
			if(!ret)
			{
				fprintf(stderr, "Theora: end of file while searching for codec headers.\n");
				exit(1);
			}
		}
	}
}

static
double
FM_GetTime()
{
	static Uint32 startticks = 0;
	double curtime;
	if(!startticks)
		startticks = SDL_GetTicks();
	curtime = 1.0e-3 * (double)(SDL_GetTicks() - startticks);
	return curtime;
}

void
FM_EnterMenuIntro(FM_Game *game)
{
	audiobuffer = rilMediaALCreateBufferFromOgg(audiofilename);
	alGenSources(1, &audiosource);
	alSourcei(audiosource, AL_BUFFER, audiobuffer);

	alSourcePlay(audiosource);

	glGenTextures(1, &video);
	glBindTexture(GL_TEXTURE_2D, video);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

	rgb = malloc(512 * 512 * 3 * sizeof(GLubyte));

	infile = fopen(videofilename, "rb" );

	ogg_sync_init(&oy);

	theora_comment_init(&tc);
	theora_info_init(&ti);

	FM_ParseHeaders();

	theora_decode_init(&td, &ti);

	FM_GL_2D(game);

	FM_SetEventHandler(FM_EventMenuIntro);
	FM_SetUpdateHandler(FM_UpdateMenuIntro);
}

void
FM_ExitMenuIntro(FM_Game *game)
{
	FM_GL_3D(game);

	fclose(infile);

	if(theora_p)
	{
		ogg_stream_clear(&to);
		theora_clear(&td);
		theora_comment_clear(&tc);
		theora_info_clear(&ti);
	}
	ogg_sync_clear(&oy);

	free(rgb);
	glDeleteTextures(1, &video);

	alSourceStop(audiosource);

	alDeleteSources(1, &audiosource);
	alDeleteBuffers(1, &audiobuffer);
}

bool
FM_EventMenuIntro(FM_Game *game, SDL_Event *event)
{
	bool exitrequest = false;
	switch(event->type)
	{
		case SDL_KEYUP:
			switch(event->key.keysym.sym)
			{
				case SDLK_ESCAPE:
					FM_ExitMenuIntro(game);
					FM_EnterMenuMain(game);
					break;
				default:
					break;
			}
			break;
		case SDL_QUIT:
			FM_ExitMenuIntro(game);
			exitrequest = true;
			break;
	}
	return exitrequest;
}

bool
FM_UpdateMenuIntro(FM_Game *game)
{
	ogg_packet op;

	while(theora_p && !videobuf_ready)
	{
		if(ogg_stream_packetout(&to,&op) > 0)
		{
			theora_decode_packetin(&td,&op);

			videobuf_granulepos=td.granulepos;
			videobuf_time = theora_granule_time(&td, videobuf_granulepos);

			now = FM_GetTime();
			delay = videobuf_time - now;
			if(delay >= 0.0)
				videobuf_ready = true;
			else if(now - last_frame_time >= 1.0)
				videobuf_ready = true;
		}
		else
			break;
	}

	if(!hasdatatobuffer && !videobuf_ready)
		playbackdone = true;

	if(!theora_p || videobuf_ready)
	{
		int ticks = 1.0e3 * (videobuf_time - FM_GetTime());
		if(ticks > 0) SDL_Delay(ticks);
	}
	
	if(videobuf_ready)
	{
		FM_VideoWrite(game);
		videobuf_ready = false;
		last_frame_time = FM_GetTime();
	}

	if(hasdatatobuffer)
		hasdatatobuffer = FM_BufferData(&oy);
    
	if(ogg_sync_pageout(&oy,&og) > 0)
		FM_QueuePage(&og);

	if(playbackdone)
	{
		FM_ExitMenuIntro(game);
		FM_EnterMenuMain(game);
	}

	return false;
}
