// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "menuedit.h"
#include <ode/ode.h>
#include <GL/glu.h>
#include "main.h"
#include "menumain.h"
#include "machine.h"
#include "track.h"
#include "rilcamera.h"
#include "rilfont.h"
#include "list.h"
#include "set.h"
#include "ml.h"

dWorldID world;
dSpaceID space;
dJointGroupID contactgroup;

static RILfont *font;
static RILcamera camera;
static FM_Track *track;

static list *machines;
static set *activemachines;
static list *checkpoints; // Used for generator.

static const char *fontfilename = PACKAGE_DATADIR "/VeraBd.ttf";

#define MAX_CONTACTS 1000

static
void
FM_NearCallback(void *data, dGeomID o1, dGeomID o2)
{
	dBodyID b1, b2;
	dContact contact[MAX_CONTACTS];
	dJointID c;
	unsigned int i;
	int n;
	void *d1, *d2;

	b1 = dGeomGetBody(o1);
	b2 = dGeomGetBody(o2);
	if(b1 && b2 && dAreConnectedExcluding(b1, b2, dJointTypeContact))
		return;

	d1 = dGeomGetData(o1);
	d2 = dGeomGetData(o2);

	if(set_contains(machines, d1) && set_contains(machines, d2) && (!set_contains(activemachines, d1) || !set_contains(activemachines, d2)))
		return;

	n = dCollide(o1, o2, MAX_CONTACTS, &contact[0].geom, sizeof(dContact));
	if(n > 0)
	{
		for(i = 0; i < n; i++)
		{
			contact[i].surface.mode = dContactSlip1 | dContactSlip2 | dContactSoftERP | dContactSoftCFM | dContactApprox1 | dContactMu2;
			contact[i].surface.mu = 0.99;
			contact[i].surface.mu2 = 0.99;
			contact[i].surface.slip1 = 0.01;
			contact[i].surface.slip2 = 0.01;
			contact[i].surface.soft_erp = 0.5;
			contact[i].surface.soft_cfm = 0.3;
			c = dJointCreateContact(world,contactgroup,&contact[i]);
			dJointAttach(c, dGeomGetBody(contact[i].geom.g1), dGeomGetBody(contact[i].geom.g2));
		}
	}
}

static
void
FM_ActivateMachinesScanner(FM_Machine *machine)
{
	activemachines = set_add(activemachines, machine);
}

static
void
FM_ActivateMachines()
{
	list_scan(machines, (list_scanner)FM_ActivateMachinesScanner);
}

static
float *
FM_GenCheckPoint(const float *point)
{
	float *checkpoint;
	checkpoint = malloc(sizeof(float) * 3);
	mlPointCopy(point, checkpoint);
	return checkpoint;
}

void
FM_EnterMenuEdit(FM_Game *game)
{
	float pos[3];
	unsigned int i;

	alSourceStop(game->menuaudiosources[FM_GAME_MENU_MUSIC]);

	font = rilFontLoad(fontfilename, RIL_FONT_BITMAP);

	world = dWorldCreate();
	space = dHashSpaceCreate(0);
	contactgroup = dJointGroupCreate(0);
	dWorldSetGravity(world, 0, -9.8, 0);

	mlPointLoad(camera.object, 0, 0, 0);

	track = FM_LoadTrack(space, game->tracknames[game->track]);

	machines = activemachines = checkpoints = NULL;

	for(i = 0; i < track->ncheckpoints; i++)
		checkpoints = list_append(checkpoints, FM_GenCheckPoint(&track->checkpoints[i * 3]));

	mlPointLoad(pos, 0, 5, 0);
	machines = list_append(machines, FM_LoadMachine(world, space, 37.5, 20, pos, -90, game->machinenames[game->machine]));

	FM_ActivateMachines();

	FM_SetEventHandler(FM_EventMenuEdit);
	FM_SetUpdateHandler(FM_UpdateMenuEdit);
}

void
FM_ExitMenuEdit(FM_Game *game)
{
	rilFontUnload(font);

	list_scan(machines, (list_scanner)FM_FreeMachine);
	list_free(machines);

	list_scan(checkpoints, (list_scanner)free);
	list_free(checkpoints);

	FM_FreeTrack(track);

	dJointGroupDestroy(contactgroup);
	dSpaceDestroy(space);
	dWorldDestroy(world);
}

static
void
FM_UpdateMachineScanner(FM_Machine *machine, const RILcamera *camera)
{
	if(!FM_UpdateMachine(machine, camera))
	{
		activemachines = set_remove(activemachines, machine);
		alSourceStop(machine->source);
	}
}

static
void
FM_RenderMenuEdit(FM_Game *game)
{
	FM_Machine *machine;
	float a[3], b[3];
	const dReal *p, *v;
	char text[2];

	machine = activemachines->data;

       	p = dBodyGetPosition(machine->body[0]);
	v = dBodyGetLinearVel(machine->body[0]);

	mlVectorSub(p, camera.object, a);
	mlVectorScalarProduct(a, 0.15, b);
	mlVectorAddInPlace(camera.object, b);

	alListener3f(AL_POSITION, camera.object[0], 70, camera.object[2]);
	alListenerfv(AL_ORIENTATION, mlVectorNegY);
	alListener3f(AL_VELOCITY, v[0], v[1], v[2]);

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glLoadIdentity();
	gluLookAt(camera.object[0], 70, camera.object[2], camera.object[0], camera.object[1], camera.object[2], 0, 0, -1);
	rilCameraGLFrustumLoad(&camera);

	list_user_scan(activemachines, (list_user_scanner)FM_UpdateMachineScanner, &camera);
	FM_RenderTrack(track, &camera);
	list_scan(activemachines, (list_scanner)FM_RenderMachine);

	if(list_length(checkpoints) >= 2)
	{
		list *l;
		glBegin(GL_LINE_STRIP);
		for(l = checkpoints; l; l = l->next)
			glVertex3fv(l->data);
		glEnd();
	}

	FM_GL_2D(game);
	rilFontFaceSize(font, 43);
	sprintf(text, "%i", set_size(activemachines));
	glRasterPos2i(0, 0);
	rilFontRender(font, text);
	FM_GL_3D(game);

	SDL_GL_SwapBuffers();
}

static
void
FM_ControlActiveMachines(FM_Game *game)
{
	FM_Machine *machine;

	machine = activemachines->data;
	FM_ControlMachine(machine, game->keys[SDLK_UP], game->keys[SDLK_DOWN], game->keys[SDLK_LEFT], game->keys[SDLK_RIGHT]);
}

bool
FM_UpdateMenuEdit(FM_Game *game)
{
	unsigned int i;
	
	FM_ControlActiveMachines(game);

	//FM_UpdateTrack(track, machines[0]);

	for(i = 0; i < 16; i++)
	{
		dSpaceCollide(space, NULL, &FM_NearCallback);
		dWorldStep(world, 0.01);
		dJointGroupEmpty(contactgroup);
	}

	FM_RenderMenuEdit(game);

	return false;
}

bool
FM_EventMenuEdit(FM_Game *game, SDL_Event *event)
{
	bool exitrequest = false;
	switch(event->type)
	{
		case SDL_KEYUP:
			switch(event->key.keysym.sym)
			{
				case SDLK_c:
					checkpoints = list_append(checkpoints, FM_GenCheckPoint(dBodyGetPosition(((FM_Machine *)machines->data)->body[0])));
					break;
				case SDLK_v:
					checkpoints = list_remove(checkpoints, list_length(checkpoints) - 1);
					break;
				case SDLK_p:
					list_scan(checkpoints, (list_scanner)mlPointPrint);
					break;
				case SDLK_ESCAPE:
					FM_ExitMenuEdit(game);
					FM_EnterMenuMain(game);
					break;
				default:
					break;
			}
			break;
		case SDL_QUIT:
			FM_ExitMenuEdit(game);
			exitrequest = true;
			break;
	}
	return exitrequest;
}
