// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "main.h"
#include <stdlib.h>
#include <stdio.h>
#include <SDL/SDL.h>
#include <GL/glu.h>
#include <IL/ilut.h>
#include <AL/alut.h>
#include "menuintro.h"
#include "rilimage.h"
#include "rilmedia.h"
#include "config.h"

#define FM_VIDEO_FLAGS SDL_OPENGL | SDL_RESIZABLE | SDL_ANYFORMAT
#define FM_AUDIO_FLAGS SDL_OPENAL

enum 
{
	SDL_OPENAL
};

static bool (*FM_UpdateHandler)(FM_Game *game);
static bool (*FM_EventHandler)(FM_Game *game, SDL_Event *event);

static bool exitrequest;

static const char *tracknames[] = 
{
	"square",
	"circle",
	"triangle",
	"crayons",
	"garden",
	"abyss",
	"bathtub",
	"land"
};

static const char *machinenames[] = 
{
	"car",
	"formula1",
	"block"
};

static
void
SDL_SetAudioMode(Uint32 flags)
{
	switch(flags)
	{
		case SDL_OPENAL:
			alutInit(0, NULL);
			break;
	}
}

static
void
SDL_Quit2()
{
	alutExit();

	SDL_Quit();
}

void
FM_EnterMain(FM_Game *game)
{
	srand(time(NULL));

	SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO);

	SDL_EnableKeyRepeat(100, 100);

	SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 8);
	SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 8);
	SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 8);
	SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
	game->screen = SDL_SetVideoMode(FM_SCREEN_WIDTH, FM_SCREEN_HEIGHT, FM_SCREEN_BPP, FM_VIDEO_FLAGS);

	SDL_SetAudioMode(FM_AUDIO_FLAGS);

	SDL_WM_SetCaption(PACKAGE_STRING, NULL);

	game->keys = SDL_GetKeyState(NULL);

	glEnable(GL_COLOR_MATERIAL);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glEnable(GL_TEXTURE_2D);
	glClearColor(0, 0, 0.5, 0);
	//glEnable(GL_BLEND);
	//glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	//glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

	ilInit();
	iluInit();
	ilEnable(IL_CONV_PAL);
	ilutRenderer(ILUT_OPENGL);
	ilutEnable(ILUT_OPENGL_CONV);

	FM_GL_3D(game);

	game->menubackground = rilImageGLGenTexture(FM_MENU_BACKGROUND);
	game->menufont = rilFontLoad(FM_FONT, RIL_FONT_EXTRUDE);
	btmLoad(&game->menuselector, FM_MENU_SELECTOR);

	game->menuaudiobuffers[FM_GAME_MENU_MUSIC] = rilMediaALCreateBufferFromOgg(FM_MENU_MUSIC);
	alGenSources(1, &game->menuaudiosources[FM_GAME_MENU_MUSIC]);
	alSourcei(game->menuaudiosources[FM_GAME_MENU_MUSIC], AL_BUFFER, game->menuaudiobuffers[FM_GAME_MENU_MUSIC]);
	alSourcei(game->menuaudiosources[FM_GAME_MENU_MUSIC], AL_LOOPING, AL_TRUE);

	game->nmachines = 4;

	game->nplayers = 1;

	game->ntracknames = sizeof(tracknames) / sizeof(&tracknames[0]);
	game->tracknames = tracknames;

	game->nmachinenames = sizeof(machinenames) / sizeof(&machinenames[0]);
	game->machinenames = machinenames;

	game->track = rand() % game->ntracknames;
	game->machine = rand() % game->nmachinenames;
}

void
FM_ExitMain(FM_Game *game)
{
	alSourceStop(game->menuaudiosources[FM_GAME_MENU_MUSIC]);

	alDeleteSources(1, &game->menuaudiosources[FM_GAME_MENU_MUSIC]);
	alDeleteBuffers(1, &game->menuaudiobuffers[FM_GAME_MENU_MUSIC]);

	btmUnload(&game->menuselector);

	rilFontUnload(game->menufont);
	glDeleteTextures(1, &game->menubackground);

	ilShutDown();

	SDL_Quit2();
}

void
FM_SetUpdateHandler(bool (*handler)(FM_Game *))
{
	FM_UpdateHandler = handler;
}

void
FM_SetEventHandler(bool (*handler)(FM_Game *, SDL_Event *))
{
	FM_EventHandler = handler;
}

void
FM_GL_2D(FM_Game *game)
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glViewport(0, 0, game->screen->w, game->screen->h);
	gluOrtho2D(0, game->screen->w, 0, game->screen->h);
	game->projection = FM_PROJECTION_2D;

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

void
FM_GL_3D(FM_Game *game)
{
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glViewport(0, 0, game->screen->w, game->screen->h);
	gluPerspective(45, (float)game->screen->w / game->screen->h, 1, 250);
	game->projection = FM_PROJECTION_3D;

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

static
void
FM_Resize(FM_Game *game, unsigned int width, unsigned int height)
{
	game->screen = SDL_SetVideoMode(width, height, FM_SCREEN_BPP, FM_VIDEO_FLAGS);
	switch(game->projection)
	{
		case FM_PROJECTION_2D: FM_GL_2D(game); break;
		case FM_PROJECTION_3D: FM_GL_3D(game); break;
	}
}


/*
static
void
FM_Capture()
{
	static int i = 0;
	char buffer[128];

	ilutGLScreen();
	sprintf(buffer, "freemachines-%06i.png", i++);
	ilSaveImage(buffer);
}
*/

int
main(int argc, char *argv[])
{
	FM_Game game;
	SDL_Event event;
	Uint32 startticks, endticks, ticks;

	FM_EnterMain(&game);

	exitrequest = false;

	FM_EnterMenuIntro(&game);

	while(!exitrequest)
	{
		startticks = SDL_GetTicks();

		while(SDL_PollEvent(&event))
		{
			if(event.type == SDL_VIDEORESIZE) FM_Resize(&game, event.resize.w, event.resize.h);
			exitrequest = FM_EventHandler(&game, &event);
		}

		exitrequest = exitrequest || FM_UpdateHandler(&game);

		endticks = SDL_GetTicks();

		ticks = endticks - startticks;

		if(FM_UpdateHandler != FM_UpdateMenuIntro && ticks < 40)
			SDL_Delay(40 - ticks);

//		FM_Capture();
	}

	FM_ExitMain(&game);

	return EXIT_SUCCESS;
}
