// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "machine.h"
#include <math.h>
#include "rilmedia.h"
#include "ml.h"

static
void
FM_ODE_LoadMachine(FM_Machine *machine, dWorldID world, dSpaceID space, const float *position, float steer)
{
	unsigned int i;
	dMass mass;
	dQuaternion q;
	const dReal *anchor;

	// Chassis
	machine->body[0] = dBodyCreate(world);
	dBodySetPosition(machine->body[0], position[0], position[1], position[2]);
	dMassSetBox(&mass, 1, 2.5, 1, 4);
	dMassAdjust(&mass, 1);
	dBodySetMass(machine->body[0], &mass);
	machine->geom[0] = dCreateBox(space, 2.5, 1, 5.5);
	dGeomSetData(machine->geom[0], machine);
	dGeomSetBody(machine->geom[0], machine->body[0]);

	// Wheels
	dQFromAxisAndAngle(q, 0, 1, 0, M_PI * 0.5);
	dMassSetSphere(&mass, 0.05, 0.25);
	dMassAdjust(&mass, 0.05);
	for(i = 1; i < 5; i++)
	{
		machine->body[i] = dBodyCreate(world);
		dBodySetQuaternion(machine->body[i], q);
		dBodySetMass(machine->body[i], &mass);
		machine->geom[i] = dCreateSphere(space, 0.25);
		dGeomSetData(machine->geom[i], machine);
		dGeomSetBody(machine->geom[i], machine->body[i]);
	}
	dBodySetPosition(machine->body[1], position[0] - 1.5, position[1] - 0.5, position[2] + 1.25);
	dBodySetPosition(machine->body[2], position[0] + 1.5, position[1] - 0.5, position[2] + 1.25);
	dBodySetPosition(machine->body[3], position[0] - 1.5, position[1] - 0.5, position[2] - 1.5);
	dBodySetPosition(machine->body[4], position[0] + 1.5, position[1] - 0.5, position[2] - 1.5);

	// Hinges
	for(i = 0; i < 4; i++)
	{
		machine->joint[i] = dJointCreateHinge2(world, NULL);
		dJointAttach(machine->joint[i], machine->body[0], machine->body[i + 1]);
		anchor = dBodyGetPosition(machine->body[i + 1]);
		dJointSetHinge2Anchor(machine->joint[i], anchor[0], anchor[1], anchor[2]);
		dJointSetHinge2Axis1(machine->joint[i], 0, 1, 0);
		dJointSetHinge2Axis2(machine->joint[i], 1, 0, 0);
		dJointSetHinge2Param(machine->joint[i], dParamSuspensionERP, 0.4);
		dJointSetHinge2Param(machine->joint[i], dParamSuspensionCFM, 0.8);
	}
	for(i = 0; i < 2; i++)
	{
		dJointSetHinge2Param(machine->joint[i], dParamLoStop, 0);
		dJointSetHinge2Param(machine->joint[i], dParamHiStop, 0);
	}
}

static
void
FM_ODE_FreeMachine(FM_Machine *machine)
{
	unsigned int i;
	for(i = 0; i < 5; i++)
		dGeomDestroy(machine->geom[i]);
}

static
void
FM_AL_LoadMachine(FM_Machine *machine, const char *engine, const float *position)
{
	machine->buffer = rilMediaALCreateBufferFromOgg(engine);
	alGenSources(1, &machine->source);
	alSourcei(machine->source, AL_BUFFER, machine->buffer);
	alSourcei(machine->source, AL_LOOPING, AL_TRUE);
	alSourcefv(machine->source, AL_POSITION, position);
	alSourcef(machine->source, AL_GAIN, 10);
	alSourcePlay(machine->source);
}

static
void
FM_AL_FreeMachine(FM_Machine *machine)
{
	alDeleteSources(1, &machine->source);
	alDeleteBuffers(1, &machine->buffer);
}

FM_Machine *
FM_LoadMachine(dWorldID world, dSpaceID space, float cfgspeed, float cfgsteer, const float *position, float steer, const char *name)
{
	FM_Machine *machine;
	char buffer[128];

	machine = malloc(sizeof(FM_Machine));

	machine->cfgspeed = cfgspeed;
	machine->cfgsteer = cfgsteer;
	machine->checkpoint = 0;

	sprintf(buffer, PACKAGE_DATADIR "/machines/%schassis.btm.bz2", name);
	btmLoad(&machine->chassis, buffer);

	sprintf(buffer, PACKAGE_DATADIR "/machines/%swheel.btm.bz2", name);
	btmLoad(&machine->wheel, buffer);

	FM_ODE_LoadMachine(machine, world, space, position, steer);

	sprintf(buffer, PACKAGE_DATADIR "/machines/%sengine.ogg", name);
	FM_AL_LoadMachine(machine, buffer, position);

	return machine;
}

void
FM_FreeMachine(FM_Machine *machine)
{
	alSourceStop(machine->source);

	FM_AL_FreeMachine(machine);

	btmUnload(&machine->chassis);
	btmUnload(&machine->wheel);

	FM_ODE_FreeMachine(machine);

	free(machine);
}

void
FM_ControlMachine(FM_Machine *machine, int up, int down, int left, int right)
{
	unsigned int i;
	float speed = machine->cfgspeed * (up ? 1 : (down ? -1 : 0));
	float steer = machine->cfgsteer * (left ? -1 : (right ? 1 : 0));
	if(steer)
	{
		for(i = 2; i < 4; i++)
		{
			dJointSetHinge2Param(machine->joint[i], dParamVel, steer);
			dJointSetHinge2Param(machine->joint[i], dParamFMax, 4);
			dJointSetHinge2Param(machine->joint[i], dParamLoStop, -0.75);
			dJointSetHinge2Param(machine->joint[i], dParamHiStop, 0.75);
			dJointSetHinge2Param(machine->joint[i], dParamFudgeFactor, 0.1);
		}
	}
	else
	{
		for(i = 2; i < 4; i++)
		{
			dJointSetHinge2Param(machine->joint[i], dParamLoStop, 0);
			dJointSetHinge2Param(machine->joint[i], dParamHiStop, 0);
			dJointSetHinge2Param(machine->joint[i], dParamFudgeFactor, 0.1);
		}
	}
	for(i = 0; i < 4; i++)
	{
		dJointSetHinge2Param(machine->joint[i], dParamVel2, speed);
		dJointSetHinge2Param(machine->joint[i], dParamFMax2, 2);
	}
	for(i = 0; i < 2; i++)
	{
		dJointSetHinge2Param(machine->joint[i], dParamLoStop, 0);
		dJointSetHinge2Param(machine->joint[i], dParamHiStop, 0);
		dJointSetHinge2Param(machine->joint[i], dParamFudgeFactor, 0.01);
	}
}

bool
FM_UpdateMachine(FM_Machine *machine, const RILcamera *camera)
{
	const dReal *p;
	float v;
	unsigned int i;
	bool frustum;

	v = 0;
	for(i = 0; i < 4; i++)
		v += mlVectorNormFunc(dBodyGetAngularVel(machine->body[i + 1]));
	alSourcef(machine->source, AL_PITCH, (mlClampf(v / 80.0, 0, 1.5) + 0.5));

	p = dBodyGetPosition(machine->body[0]);
	alSourcefv(machine->source, AL_POSITION, p);
	frustum = rilCameraFrustumPoint(camera, p);
	return frustum;
}

void
FM_RenderMachine(FM_Machine *machine)
{
	size_t i;
	BTMmodel *model;
	const dReal *p;
	const dReal *r;
	float m[16];

	for(i = 0; i < 5; i++)
	{
		p = dBodyGetPosition(machine->body[i]);
        	r = dBodyGetRotation(machine->body[i]);
		m[ 0] = r[ 0];m[ 1] = r[ 4];m[ 2] = r[ 8];m[ 3] = 0;
		m[ 4] = r[ 1];m[ 5] = r[ 5];m[ 6] = r[ 9];m[ 7] = 0;
		m[ 8] = r[ 2];m[ 9] = r[ 6];m[10] = r[10];m[11] = 0;
		m[12] = p[ 0];m[13] = p[ 1];m[14] = p[ 2];m[15] = 1;
		if(i) model = &machine->wheel;
		else model = &machine->chassis;
		glPushMatrix();
			glMultMatrixf(m);
			btmRender(model);
		glPopMatrix();
	}
}

