// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include "btm.h"
#include <stdlib.h>
#include <string.h>
#include <libgen.h>
#include <IL/ilut.h>
#include "btmio.h"
#include "ml.h"

static
void
btmGenTextures(BTMmodel *model);

static
void
btmDeleteTextures(BTMmodel *model);

static
void
btmGenDisplayLists(BTMmodel *model);

static
void
btmDeleteDisplayLists(BTMmodel *model);

static
void
btmLoadString(char **name, BTMFILE *stream)
{
	char *line = NULL;
	size_t n;
	char *sname;

	btmgetline(&line, &n, stream);
	sname = strchr(line, ' ');
	sname++;
	*name = malloc((strlen(sname) + 1) * sizeof(char));
	strcpy(*name, sname);
	free(line);
}

static
void
btmLoadFace(BTMface *face, BTMFILE *stream)
{
	char *line = NULL;
	size_t n;
	unsigned int i;

	// Skip {
	btmgetline(&line, &n, stream);
	free(line);

	// Parse mat
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\t\t\tmat %i", &face->mat);
	free(line);

	// Parse verts
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\t\t\tverts %i %i %i", &face->verts[0], &face->verts[1], &face->verts[2]);
	free(line);

	// Skip {
	btmgetline(&line, &n, stream);
	free(line);

	// Parse uv
	for(i = 0; i < 3; i++)
	{
		btmgetline(&line, &n, stream);
		sscanf(line, "\t\t\t\t\tuv %f %f %f", &face->uv[i][0], &face->uv[i][1], &face->uv[i][2]);
		free(line);
	}

	// Skip }
	btmgetline(&line, &n, stream);
	free(line);
	
	// Parse no
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\t\t\tno %f %f %f", &face->no[0], &face->no[1], &face->no[2]);
	free(line);

	// Skip }
	btmgetline(&line, &n, stream);
	free(line);
}

static
void
btmLoadVertex(BTMvertex *vertex, BTMFILE *stream)
{
	char *line = NULL;
	size_t n;

	// Skip {
	btmgetline(&line, &n, stream);
	free(line);

	// Parse co
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\t\t\tco %f %f %f", &vertex->co[0], &vertex->co[1], &vertex->co[2]);
	free(line);

	// Parse no
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\t\t\tno %f %f %f", &vertex->no[0], &vertex->no[1], &vertex->no[2]);
	free(line);

	// Skip }
	btmgetline(&line, &n, stream);
	free(line);
}

static
void
btmLoadMaterial(BTMmaterial *material, BTMFILE *stream)
{
	char *line = NULL;
	size_t n;

	// Skip {
	btmgetline(&line, &n, stream);
	free(line);

	// Parse name
	btmLoadString(&material->name, stream);
	
	// Parse texture
	btmLoadString(&material->texture, stream);

	// Parse alpha
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\t\t\talpha %f", &material->alpha);
	free(line);

	// Parse emit
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\t\t\temit %f", &material->emit);
	free(line);

	// Parse ambient
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\t\t\tambient %f %f %f", &material->ambient[0], &material->ambient[1], &material->ambient[2]);
	free(line);

	// Parse diffuse
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\t\t\tdiffuse %f %f %f", &material->diffuse[0], &material->diffuse[1], &material->diffuse[2]);
	free(line);

	// Parse specular
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\t\t\tspecular %f %f %f", &material->specular[0], &material->specular[1], &material->specular[2]);
	free(line);

	// Skip }
	btmgetline(&line, &n, stream);
	free(line);
}

static
void
btmLoadMesh(BTMmesh *mesh, BTMFILE *stream)
{
	char *line = NULL;
	size_t n;
	unsigned int i;

	// Skip begin mesh
	btmgetline(&line, &n, stream);
	free(line);
	
	// Parse name
	btmLoadString(&mesh->name, stream);

	// Skip begin materials
	btmgetline(&line, &n, stream);
	free(line);

	// Parse materials length
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\tlength %i", &mesh->nmaterials);
	free(line);

	// Load materials
	mesh->materials = malloc(mesh->nmaterials * sizeof(BTMmaterial));
	for(i = 0; i < mesh->nmaterials; i++)
		btmLoadMaterial(&mesh->materials[i], stream);

	// Skip end
	btmgetline(&line, &n, stream);
	free(line);


	// Skip begin vertices
	btmgetline(&line, &n, stream);
	free(line);

	// Parse vertices length
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\tlength %i", &mesh->nvertices);
	free(line);

	// Load vertices
	mesh->vertices = malloc(mesh->nvertices * sizeof(BTMvertex));
	for(i = 0; i < mesh->nvertices; i++)
		btmLoadVertex(&mesh->vertices[i], stream);

	// Skip end
	btmgetline(&line, &n, stream);
	free(line);

	// Skip begin faces
	btmgetline(&line, &n, stream);
	free(line);

	// Parse faces length
	btmgetline(&line, &n, stream);
	sscanf(line, "\t\t\tlength %i", &mesh->nfaces);
	free(line);

	// Load faces
	mesh->faces = malloc(mesh->nfaces * sizeof(BTMface));
	for(i = 0; i < mesh->nfaces; i++)
		btmLoadFace(&mesh->faces[i], stream);

	// Skip end
	btmgetline(&line, &n, stream);
	free(line);


	// Skip end
	btmgetline(&line, &n, stream);
	free(line);
}

static
void
btmLoadObject(BTMobject *object, BTMFILE *stream)
{
	char *line = NULL;
	size_t n;

	// Skip {
	btmgetline(&line, &n, stream);
	free(line);
	
	// Parse name
	btmLoadString(&object->name, stream);

	btmLoadMesh(&object->mesh, stream);

	// Skip }
	btmgetline(&line, &n, stream);
	free(line);
}

void
btmLoad(BTMmodel *model, const char *filename)
{
	BTMFILE *stream;
	stream = btmfopenread(filename);
	char *line;
	char *tmp;
	size_t n;
	unsigned int i;

	tmp = strdup(filename);
	model->name = strdup(basename(tmp));
	free(tmp);

	tmp = strdup(filename);
	model->path = strdup(dirname(tmp));
	free(tmp);

	// Skip begin objects
	btmgetline(&line, &n, stream);
	free(line);

	// Parse objects length
	btmgetline(&line, &n, stream);
	sscanf(line, "\tlength %i", &model->nobjects);
	free(line);
	model->objects = malloc(model->nobjects * sizeof(BTMobject));
	model->textures = malloc(model->nobjects * sizeof(GLuint *));

	// Load objects
	for(i = 0; i < model->nobjects; i++)
		btmLoadObject(&model->objects[i], stream);

	// Skip end
	btmgetline(&line, &n, stream);
	free(line);

	btmfclose(stream);

	btmGenTextures(model);
	btmGenDisplayLists(model);
}

static
void
btmUnloadMaterial(BTMmaterial *material)
{
	free(material->name);
	free(material->texture);
}

static
void
btmUnloadMesh(BTMmesh *mesh)
{
	unsigned int i;
	free(mesh->name);
	for(i = 0; i < mesh->nmaterials; i++)
		btmUnloadMaterial(&mesh->materials[i]);
	free(mesh->materials);
	free(mesh->vertices);
	free(mesh->faces);
}

static
void
btmUnloadObject(BTMobject *object)
{
	free(object->name);
	btmUnloadMesh(&object->mesh);
}

void
btmUnload(BTMmodel *model)
{
	unsigned int i;

	btmDeleteDisplayLists(model);
	btmDeleteTextures(model);

	for(i = 0; i < model->nobjects; i++)
		btmUnloadObject(&(model->objects[i]));
	free(model->objects);

	free(model->path);
	free(model->name);
}

static
void
btmSetMaterial(BTMmaterial *material)
{
	glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, material->ambient);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, material->diffuse);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, material->specular);
	//glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, material->ambient);
}

static
void
btmGenDisplayLists(BTMmodel *model)
{
	size_t i, j, k, vertindex;
	model->displaylists = malloc(model->nobjects * sizeof(GLuint *));
	for(k = 0; k < model->nobjects; k++)
	{
		model->displaylists[k] = glGenLists(1);
		glNewList(model->displaylists[k], GL_COMPILE);
		for(j = 0; j < model->objects[k].mesh.nfaces; j++)
		{
			btmSetMaterial(&model->objects[k].mesh.materials[model->objects[k].mesh.faces[j].mat]);
			glBindTexture(GL_TEXTURE_2D, model->textures[k][model->objects[k].mesh.faces[j].mat]);
			glBegin(GL_TRIANGLES);
			for(i = 0; i < 3; i++)
			{
				vertindex = model->objects[k].mesh.faces[j].verts[i];
				glNormal3fv(model->objects[k].mesh.vertices[vertindex].no);
				glTexCoord2fv(model->objects[k].mesh.faces[j].uv[i]);
				glVertex3fv(model->objects[k].mesh.vertices[vertindex].co);
			}
			glEnd();
		}
		glEndList();
	}
}


static
void
btmDeleteDisplayLists(BTMmodel *model)
{
	size_t i;
	for(i = 0; i < model->nobjects; i++)
		glDeleteLists(model->displaylists[i], 1);

	free(model->displaylists);
}

static
void
btmGenTextures(BTMmodel *model)
{
	unsigned int i, j;
	char *data;
	ILuint image;

	model->textures = malloc(model->nobjects * sizeof(GLuint *));
	for(j = 0; j < model->nobjects; j++)
	{
		model->textures[j] = malloc(model->objects[j].mesh.nmaterials * sizeof(GLuint));
		for(i = 0; i < model->objects[j].mesh.nmaterials; i++)
		{
			ilGenImages(1, &image);
			ilBindImage(image);
			asprintf(&data, "%s/%s", model->path, model->objects[j].mesh.materials[i].texture);
			ilLoadImage(data);
			free(data);
			model->textures[j][i] = ilutGLBindTexImage();
			ilutGLBuildMipmaps();
			ilDeleteImages(1, &image);
		}
	}
}

static
void
btmDeleteTextures(BTMmodel *model)
{
	unsigned int i;
	for(i = 0; i < model->nobjects; i++)
	{
		glDeleteTextures(model->objects[i].mesh.nmaterials, model->textures[i]);
		free(model->textures[i]);
	}
	free(model->textures);
}

void
btmRender(const BTMmodel *model)
{
	unsigned int i;
	for(i = 0; i < model->nobjects; i++)
		glCallList(model->displaylists[i]);
}

void
btmRenderWithCamera(const RILcamera *camera, const BTMmodel *model)
{
	unsigned int vertindex, i, j, k;
	float triangle[3][3];

	for(k = 0; k < model->nobjects; k++)
	{
		for(j = 0; j < model->objects[k].mesh.nfaces; j++)
		{
			for(i = 0; i < 3; i++)
			{
				vertindex = model->objects[k].mesh.faces[j].verts[i];
				mlPointCopy(model->objects[k].mesh.vertices[vertindex].co, triangle[i]);
			}
			if(rilCameraFrustumTriangle(camera, &triangle[0][0]))
			{
				btmSetMaterial(&model->objects[k].mesh.materials[model->objects[k].mesh.faces[j].mat]);
				glBindTexture(GL_TEXTURE_2D, model->textures[k][model->objects[k].mesh.faces[j].mat]);
				glBegin(GL_TRIANGLES);
				for(i = 0; i < 3; i++)
				{
					vertindex = model->objects[k].mesh.faces[j].verts[i];
					glNormal3fv(model->objects[k].mesh.vertices[vertindex].no);
					glTexCoord2fv(model->objects[k].mesh.faces[j].uv[i]);
					glVertex3fv(model->objects[k].mesh.vertices[vertindex].co);
				}
				glEnd();
			}
		}
	}
}
