// Copyright (C) 2008 Juan Manuel Borges Caño

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif

#include "ML/ml.h"
#include <stdio.h>
#include <string.h>
#include <math.h>

void
mlVectorRepr(const float *a, char *repr)
{
	if(asprintf(&repr, "(%f, %f, %f)", a[0], a[1], a[2])<0)
		repr = NULL;
}

void
mlVectorPrint(const float *a)
{
	printf("(%f, %f, %f)\n", a[0], a[1], a[2]);
}

void
mlVectorCopy(const float *a, float *result)
{
	memcpy(result, a, 3 * sizeof(float));
}

void
mlVectorLoad(float *a, float x, float y, float z)
{
	a[0] = x;
	a[1] = y;
	a[2] = z;
}

void
mlVectorAdd(const float *a, const float *b, float *result)
{
	unsigned int i;
	for(i = 0; i < 3; i++)
		result[i] = a[i] + b[i];
}

void
mlVectorAddInPlace(float *a, const float *b)
{
	unsigned int i;
	for(i = 0; i < 3; i++)
		a[i] += b[i];
}

void
mlVectorSub(const float *a, const float *b, float *result)
{
	unsigned int i;
	for(i = 0; i < 3; i++)
		result[i] = a[i] - b[i];
}

void
mlVectorSubInPlace(float *a, const float *b)
{
	unsigned int i;
	for(i = 0; i < 3; i++)
		a[i] -= b[i];
}

void
mlVectorScalarProduct(const float *a, float b, float *result)
{
	unsigned int i;
	for(i = 0; i < 3; i++)
		result[i] = a[i] * b;
}

void
mlVectorScalarProductInPlace(float *a, float b)
{
	unsigned int i;
	for(i = 0; i < 3; i++)
		a[i] *= b;
}

void
mlVectorNorm(const float *a, float *result)
{
	*result = sqrt(a[0] * a[0] + a[1] * a[1] + a[2] * a[2]);
}

float
mlVectorNormFunc(const float *a)
{
	return sqrt(a[0] * a[0] + a[1] * a[1] + a[2] * a[2]);
}

void
mlVectorDotProduct(const float *a, const float *b, float *result)
{
	*result = a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
}

float
mlVectorDotProductFunc(const float *a, const float *b)
{
	return a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
}

void
mlVectorCrossProduct(const float *a, const float *b, float *result)
{
	result[0] = a[1] * b[2] - a[2] * b[1];
	result[1] = a[2] * b[0] - a[0] * b[2];
	result[2] = a[0] * b[1] - a[1] * b[0];
}

void
mlVectorCrossProductInPlace(float *a, const float *b)
{
	float result[3];
	mlVectorCrossProduct(a, b, result);
	mlVectorCopy(result, a);
}

void
mlVectorNormal(const float *a, float *result)
{
	float norm;
	mlVectorNorm(a, &norm);
	mlVectorScalarProduct(a, 1.0 / norm, result);
}

void
mlVectorNormalInPlace(float *a)
{
	mlVectorScalarProductInPlace(a, 1.0 / mlVectorNormFunc(a));
}

void
mlVectorMatrixProduct(const float *a, const float *b, float *result)
{
	result[0] = a[0] * b[0] + a[1] * b[3] + a[2] * b[6];
	result[1] = a[0] * b[1] + a[1] * b[4] + a[2] * b[7];
	result[2] = a[0] * b[2] + a[1] * b[5] + a[2] * b[8];
}

void
mlVectorMatrixProductInPlace(float *a, const float *b)
{
	float result[3];
	mlVectorMatrixProduct(a, b, result);
	mlVectorCopy(result, a);
}

void
mlVectorRotate(const float *a, const float *axis, float angle, float *result)
{
	float matrix[9];
	float c = cosf(angle * M_PI / 180.0);
	float s = sinf(angle * M_PI / 180.0);
	matrix[0] = axis[0] * axis[0] * (1 - c) + c;
	matrix[1] = axis[0] * axis[1] * (1 - c) - axis[2] * s;
	matrix[2] = axis[0] * axis[2] * (1 - c) + axis[1] * s;
	matrix[3] = axis[1] * axis[0] * (1 - c) + axis[2] * s;
	matrix[4] = axis[1] * axis[1] * (1 - c) + c;
	matrix[5] = axis[1] * axis[2] * (1 - c) - axis[0] * s;
	matrix[6] = axis[0] * axis[2] * (1 - c) - axis[1] * s;
	matrix[7] = axis[1] * axis[2] * (1 - c) + axis[0] * s;
	matrix[8] = axis[2] * axis[2] * (1 - c) + c;
	mlVectorMatrixProduct(a, matrix, result);
}

void
mlVectorRotateInPlace(float *a, const float *axis, float angle)
{
	float result[3];
	mlVectorRotate(a, axis, angle, result);
	mlVectorCopy(result, a);
}

void
mlVectorAngle(const float *a, const float *b, float *angle)
{
	float dp;
	float na, nb;
	mlVectorDotProduct(a, b, &dp);
	mlVectorNorm(a, &na);
	mlVectorNorm(b, &nb);
	*angle = 180.0 / M_PI * acosf(dp/(na * nb));
}

float
mlVectorAngleFunc(const float *a, const float *b)
{
	float dp;
	float na, nb;
	mlVectorDotProduct(a, b, &dp);
	mlVectorNorm(a, &na);
	mlVectorNorm(b, &nb);
	return 180.0 / M_PI * acosf(dp/(na * nb));
}
