/* 
   scm-procedures.c: messenger procedures that are exported to
   guile environment
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */


#include <stdio.h>
#include <string.h>
#include <readline/readline.h>
#include <libguile.h>
#include <assert.h>
#include <yahoo2.h>

#include <sys/stat.h>

#include "freehoo.h"
#include "extension.h"
#include "fh-utils.h"
#include "scm-procedures.h"
#include "yahoo-adapter.h"
#include "yahoo-backend.h"
#include "interpreter.h"
#include <config.h>
#include "compat.h"

extern struct dict_words *word_list;

/*
SCM
ex_set_some_thing_x (SCM scm_some_thing)
{
  int some_thing_length;
  
  set_some_thing (scm_to_locale_string (scm_some_thing, &some_thing_length));
  
  return SCM_UNSPECIFIED;
}
*/

/* returns current freehoo version */
SCM
ex_version (void)
{
  return (scm_from_locale_string (VERSION));
}

/* this function should be used to print messages at run- time from the
   scheme environment 
*/
SCM
ex_display (SCM scm_message)
{
  char *str;
  PRINTF_MESSAGE ("%s\n", (str = scm_to_locale_string (scm_message)));
  free(str);
  return SCM_UNSPECIFIED;
}

SCM
ex_toggle_x (SCM scm_what2toggle)
{
  char * str = scm_to_locale_string(scm_what2toggle);
  if (strcasecmp (str, 
		  "bell") == 0)
    toggle_bell ();
  else if (strcasecmp (str,
		       "who") == 0)
    toggle_who ();
  else if (strcasecmp (str, 
		       "session") == 0)
    toggle_session ();
  else if (strcasecmp (str,
		       "status") == 0)
    toggle_status ();
  else
    scm_wrong_type_arg (str,
			0, SCM_UNSPECIFIED);
  free(str);
  return SCM_UNSPECIFIED;
}

SCM
ex_hook_return (void)
{
  set_hook_return (1);
  return SCM_UNSPECIFIED;
}

SCM
ex_bell (void)
{
#if defined (HAVE_RL_DING)
  rl_ding ();
#else
#if defined (HAVE_DING)
  ding ();
#else
/* don't worry, 'else' will never happen. configure script exits if
   both functions are missing */
  assert (0);
#endif
#endif
  return SCM_UNSPECIFIED;
}

SCM
ex_set_default_status_x (SCM scm_status)
{
  if (get_current_status () == -1) 
    set_current_status (scm_to_long (scm_status));
  return SCM_UNSPECIFIED;
}

SCM
ex_get_global_extensions_directory (void)
{
  return (scm_from_locale_string (get_global_extensions_directory ()));
}

SCM
ex_get_local_extensions_directory (void)
{
  return (scm_from_locale_string (get_local_extensions_directory ()));
}

SCM
ex_load (SCM scm_filename)
{
  char *filename = scm_to_locale_string (scm_filename);
  fh_load (filename);
  free(filename);
  return SCM_UNSPECIFIED;
}

SCM
ex_get_home_dir (void)
{
  return (scm_from_locale_string (get_home_directory ()));
}

SCM
ex_get_config_dir (void)
{
  return (scm_from_locale_string (get_config_directory ()));
}

SCM
ex_get_config_filename (void)
{
  return (scm_from_locale_string (get_config_filename ()));
}

SCM
ex_get_download_filename (void)
{
  return (scm_from_locale_string (get_download_filename ()));
}

SCM
ex_set_default_login_id_x (SCM scm_default_login_id)
{
  char * login = scm_to_locale_string (scm_default_login_id);
  set_default_login_id (login);
//  free(login);
  return SCM_UNSPECIFIED;
}

SCM
ex_get_default_login_id (void)
{
  return scm_from_locale_string (get_default_login_id ());
}

SCM
ex_set_default_password_x (SCM scm_default_password)
{
  char *passwd = scm_to_locale_string (scm_default_password);
  set_default_password (passwd);
//  free (passwd);
  return SCM_UNSPECIFIED;
}

SCM
ex_set_current_target_buddy_x (SCM scm_current_target_buddy, SCM scm_current_target_buddy_mode)
{
  char *current_target_buddy = NULL;
  char *mode_str =  NULL;
  
  current_target_buddy_mode_t mode = current_target_buddy_mode_send; // default to unset-able mode
  
  if (scm_current_target_buddy != SCM_BOOL_F)
    {
      current_target_buddy = 
	scm_to_locale_string (scm_current_target_buddy);
    }
  if (scm_current_target_buddy_mode != SCM_BOOL_F)
    {
       if(!strcasecmp ( (mode_str = scm_to_locale_string (scm_current_target_buddy_mode)), "receive"))
          {
             mode = current_target_buddy_mode_receive;
          }
    }
      
  set_current_target_buddy (current_target_buddy,mode);
//  if (current_target_buddy) free (current_target_buddy);
  if (mode_str) free (mode_str);
  return SCM_UNSPECIFIED;
}

SCM
ex_send_message (SCM scm_to, SCM scm_message)
{
  char *to, *msg;

  to = scm_to_locale_string (scm_to);
  msg = scm_to_locale_string (scm_message);
  send_message (to,
		msg);
  return SCM_UNSPECIFIED;
}

SCM
ex_send_message_no_hook (SCM scm_to, SCM scm_message)
{
  char *to, *msg;

  to = scm_to_locale_string (scm_to);
  msg = scm_to_locale_string (scm_message);
  send_message_no_hook (to,
			msg);
  return SCM_UNSPECIFIED;
}

SCM
ex_register_command_x (SCM scm_command)
{
  register_command (scm_command);
  return SCM_UNSPECIFIED;
}

SCM
ex_unregister_command_x (SCM scm_command)
{
  unregister_command (scm_command);
  return SCM_UNSPECIFIED;
}

SCM
ex_set_prompt_x (SCM scm_prompt)
{
  char *prompt;
  set_default_prompt ((prompt = scm_to_locale_string (scm_prompt)));
//  free (prompt);
  return SCM_UNSPECIFIED;
}

SCM
ex_get_buddy_list ()
{
  int i = 0;
  SCM scm_buddy_list;
  yahoo_account *temp_buddy;

  scm_buddy_list = SCM_UNSPECIFIED;

  while ((temp_buddy = (yahoo_account *) get_nth_buddy(i)))
    {
      if (!temp_buddy)
	break;
	  
      if (scm_is_true (scm_list_p (scm_buddy_list)) != 1)
	{
	  scm_buddy_list = 
	    scm_list_n (scm_from_locale_string (temp_buddy->yahoo_id), 
		     SCM_UNDEFINED);
	}
      else
	{
	  scm_buddy_list = 
	    scm_append (scm_list_n (scm_buddy_list,
				    scm_list_n (scm_from_locale_string (temp_buddy->yahoo_id), SCM_UNDEFINED), SCM_UNDEFINED));
	}
      i++;
    }
  return scm_buddy_list;
}


SCM
ex_add_buddy (SCM scm_buddy, SCM scm_group, SCM scm_msg)
{
  char *buddy =  (scm_to_locale_string (scm_buddy));
  char *group =  (scm_to_locale_string (scm_group));
  char *msg = (scm_to_locale_string (scm_msg));
  
  /*
    ==========================
    1. The following function prototype is changed from previous
       libyahoo2 versions.
    2. The changes is the extra parameter (char *) at the end, For
       now, NULL string is passed, with which its working fine. This
       needs a revisit and approprite string to be passed if required.
  */
  yahoo_add_buddy (get_fh_session ()->id, buddy, 
		   ((group == NULL) ? FH_DEFAULT_GROUP : group),
		   ((msg == NULL) ? "" : msg));
  free(buddy); free(group); free(msg);
  return SCM_UNSPECIFIED;
}


SCM
ex_fh_quit ()
{
  fh_quit ();
  return SCM_UNSPECIFIED;
}


SCM
ex_fh_logoff ()
{
  yahoo_logoff (get_fh_session ()->id);
  return SCM_UNSPECIFIED;
}

SCM
ex_fh_dict_add_word (SCM word)
{
    char *w = scm_to_locale_string (word);
    struct dict_words *dw = word_list;
    struct dict_words *newnode = calloc (1, sizeof (word_list));

    newnode->word = w;

    if (!dw)
      word_list = newnode;
    else
      while (dw)
	{
	  int ret = strcmp (dw->word, w);

	  if (ret == 0) {
            free (newnode->word);
            free (newnode);
            break;
	  }
	  
	  if(ret < 0) {
            newnode->word = dw->word;
            newnode->next = dw->next;
            dw->next = newnode;
            dw->word = w;
            break;
        }
	  if (ret > 0) {
            if (dw->next) {
	      dw = dw->next;
            } else {
	      dw->next = newnode;
	      break;
	    }
	  }
	}
    return SCM_UNSPECIFIED;
}

SCM
ex_fh_dict_add_word_sorted (SCM word)
{
    char *w = scm_to_locale_string (word);
    struct dict_words *newnode = calloc (1, sizeof (word_list));

    newnode->word = w;

    newnode->next = word_list;
    word_list = newnode;

    return SCM_UNSPECIFIED;
}

SCM
ex_fh_dict_del_word (SCM word)
{
  char *w = scm_to_locale_string (word);
  struct dict_words * dw = word_list;
  struct dict_words *delnode =NULL;

  while(dw->next) {
    if( ! strcmp(dw->next->word,w) ) {
       delnode = dw->next;
       dw->next = delnode->next;
       break;
    }
    dw = dw->next;
  }
  if(delnode) {
      free(delnode->word);
      free(delnode);
  }
  return SCM_UNSPECIFIED;
}

