##    Fonty Python Copyright (C) 2006, 2007, 2008 Donn.C.Ingle
##    Contact: donn.ingle@gmail.com - I hope this email lasts.
##
##    This file is part of Fonty Python.
##    Fonty Python is free software: you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation, either version 3 of the License, or
##    (at your option) any later version.
##
##    Fonty Python is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with Fonty Python.  If not, see <http://www.gnu.org/licenses/>.

import os, sys, locale, codecs, glob
import Image, ImageFont, ImageDraw 
import fontybugs, fpsys
from pathcontrol import *

class FontItem( object ):
    """
    Represents a single font file. It has the ability to provide a font image
    and query a font file for the family name and style.
    
    Ancestor to the specific classes per font type.
    Never instantiate one directly.
    """
    def __init__(self, glyphpaf ):
        self.glyphpaf = glyphpaf
        self.name = os.path.basename ( glyphpaf )
        
        self.ticked = False # State of the tick/cross symbol.
        self.inactive = False # Set in fpsys.markInactive()
        self.msg = "" #Say something unique when I draw this item.  
        
        ## These are lists to cater for sub-faces
        self.family, self.style =  [], []
        self.numFaces = 0
        self.pilheight, self.pilwidth = 0,0
        
        ## I'm not bad in any way, until I turn bad that is :)
        self.badfont = False
        ## If I'm bad, what should I say?
        self.badfontmsg = ""
        ## What kind of bad to the bone am I?
        ## One of FILE_NOT_FOUND, PIL_IO_ERROR, PIL_UNICODE_ERROR, PIL_CANNOT_RENDER
        self.badstyle = "" 
        
        ## We need the family name and style to be fetched
        ## because we have that filter thingy in the gui
        ## and it uses those strings to search for terms.
        ##
        ## We also want to know what flavour of bad this item will be
        ## and we must open the file and query it to know that.
        self.__queryFontFamilyStyleFlagBad()
        
    def __queryFontFamilyStyleFlagBad( self ):
        """
        Get the family, style and size of entire font.
        
        If this font has a problem (PIL can't read it) then we set the
        badfont flag, along with the badstyle & badfontmsg vars.
        
        badstyle: FILE_NOT_FOUND, PIL_IO_ERROR, PIL_UNICODE_ERROR
        
        The last kind of badstyle is set in generatePilFont() and is
        set to: PIL_CANNOT_RENDER
        
        This is tricky because FontImage.getname() is fragile
        and apt to segfault. As of Dec 2007 I have reported
        the bug, and a patch has been written but I don't
        know how it will be implemented yet.
        
        NB: InfoFontItem overrides this so it does not happen
        in that case. This is why I made this a method and not
        simply part of the __init__ above.
        """
        
        i = 0
        ## Step through all subfaces.
        #print "BUILDING:", self.glyphpaf
        while True:
            try:
                fileDoesExist = os.path.exists(self.glyphpaf)
                if not fileDoesExist:
                    fpsys.logBadStrings( self.glyphpaf )
                    self.badfont = True
                    self.badfontmsg = _("Font cannot be found, you should purge it.")
                    self.badstyle = "FILE_NOT_FOUND"
                    ## If the multi face font is damaged after the
                    ## first face, then this won't catch it...
                    break # it's at the end of the sub-faces 
            except:
                print _("Unhandled error:\nPlease move (%s) away from here and report this to us.") % self.glyphpaf
                raise
            try:
                font = ImageFont.truetype(self.glyphpaf, 16, index=i, encoding="unicode" )  
            except IOError: 
                """
                Means the ttf file cannot be opened or rendered by PIL !
                NOTE: Sets badfont
                """
                if i == 0: # fubar on the first face:
                    #print "IOERROR:", self.glyphpaf
                    fpsys.logBadStrings( self.glyphpaf )
                    self.badfont = True
                    self.badfontmsg = _("Font may be bad and it cannot be drawn.")
                    self.badstyle = "PIL_IO_ERROR"
                    ## If the multi face font is damaged after the
                    ## first face, then this won't catch it...
                break # it's at the end of the sub-faces
                if False: # using this to remark all the remarks. It collapses in my editor.
                    pass
                    ##            ## I did not want to do this next part, it *must* slow things down.
                    ##            ## I found a font throwing a MemoryError (Onsoku Seinen Plane.ttf)
                    ##            ## that only happens upon the .text() command.
                    ##            ## So, to know if this will happen to others, I am forced to render
                    ##            ## a bit of text here and now. :(
                    ##            ##
                    ##            ## UPDATE: It's not working. Onsoku *only* barfs on "TE" and not
                    ##            ## "A" or even chr(0) to chr(255) all in a string...
                    ##            ## So, it's virtually impossible to know at this point what will
                    ##            ## cause the MemoryError in the rendering step.
                    ##            ## So, I am remarking this stuff and will invent a new flag called
                    ##            ## specialvoodooerror which will be handled as a special case in the
                    ##            ## Fitmap drawing routines in the gui.
                    ##            try:
                    ##                print "RENDERING A"
                    ##                text = "A"
                    ##                pilimage = Image.new("RGB", (font.getsize( text )), (255,255,255)) 
                    ##                drawnFont = ImageDraw.Draw( pilimage ) # Draws INTO pilimage 
                    ##                drawnFont.text((0,0) , text, font=font, fill=0 ) # Causes MemoryError
                    ##            except MemoryError:
                    ##                print "CAUGHT IT FOR:", self.glyphpaf
                    ##                fpsys.logBadStrings( self.glyphpaf )
                    ##                self.badfont = True
            except UnicodeEncodeError:
                """
                NOTE: Sets badfont
                """                
                ## Aw man! I thought I had this taped. This error does not *seem* to
                ## be related to the encoding param passed to ImageFont. I have
                ## mailed the PIL list about this.
                ##
                ## In the meantime, this will have to be flagged as a bad font.
                fpsys.logBadStrings( self.glyphpaf )
                self.badfont = True
                self.badfontmsg = _("Font may be bad and it cannot be drawn.")
                #self.badstyle = "BAD_UNICODE"
                ## Changed to BAD_OPEN because it seems that if the glyphpaf
                ## is bad unicode, then this error fires before the IOError
                ## which means that this font may also not be on that path
                ## at all (i.e. BAD_OPEN) and I don't want to let users 
                ## work with bad fonts like this, so I am foreclosing the
                ## bad unicode distinction.
                self.badstyle = "PIL_UNICODE_ERROR"
                break

            except:
                ## What next on the error pile?
                raise
                ## Abort the app because this is an unhandled PIL error of some kind.
            if not self.badfont:
                ## No point Try-ing here, this segfaults when style/family is Null.
                fpsys.logSegfaulters( self.glyphpaf ) # record the potential destroyer of apps!
                self.family.append( font.getname()[0] )
                self.style.append( font.getname()[1] )
                
                i += 1
                
        self.numFaces = i
                
    def generatePilFont( self, width, maxHeight, enc="unicode" ):
        """
        This function seems too similar to the __queryFontFamilyStyleFlagBad one
        and in many ways it is. I am forced to work with PIL and it's not ideal
        at the moment.
        
        This function is called from the GUI in a tight loop. It provides (generates)
        pilimage objects with the font's text rendered onto them.
        
        Fonts that cause errors are marked 'badfont' and provide no image.
        They can then be 'displayed' and can be put into Pogs etc., but they cannot
        be seen.
        
        """
        paf, points, text = self.glyphpaf, fpsys.config.points, " " + fpsys.config.text + " "
        i = 0
        while (True):
            try:
                font = ImageFont.truetype(paf, points,index=i, encoding=enc) 

                pilheight = int( font.getsize( fpsys.config.text )[1] )# Must int for PIL.
                pilheight = min( pilheight, maxHeight )
                pilimage = Image.new("RGB", (width,pilheight), (255,255,255)) 
                
                if self.inactive:
                    col = (230, 230, 230) 
                else:
                    col = 0
                    
                ## text = os.path.basename(self.glyphpaf)
                
                ## Well, I have since discovered that some fonts
                ## cause a MemoryError on the next command:
                drawnFont = ImageDraw.Draw( pilimage ) # Draws INTO pilimage
                drawnFont.text((0,0) , text, font=font, fill=col) 
                
                ## All is well, so we step ahead to the next *potential* sub-face
                ## and return the font image data.
                i += 1
                yield pilimage, pilheight  

            except MemoryError:
                """
                NOTE: Sets badfont

                This one CAN ONLY BE CAUGHT HERE.
                **IDEALLY**, it should have been caught in __queryFontFamilyStyleFlagBad
                but for reasons explained there, it cannot.
                So, we have a badfont flag being set here too :(
                """
                self.badfontmsg = _("Font cannot be drawn.")
                self.badstyle = "PIL_CANNOT_RENDER"
                fpsys.logBadStrings( self.glyphpaf )
                self.badfont = True
                break
            ## These two must be caught, but are already known about 
            ## from the exact same test in __queryFontFamilyStyleFlagBad
            except IOError: 
                ## The font at index (i==0) cannot be opened.
                break                
            except UnicodeEncodeError:
                ## Already handled in __queryFontFamilyStyleFlagBad
                break
                
    def __str__( self ):
        return self.glyphpaf
        
## Create some subclasses to represent the fonts that we support:        
class InfoFontItem( FontItem ):
    """
    This class is only instantiated in wxgui.CreateFitmaps 
    It's used to indicate when a Folder or Pog is EMPTY and
    if a single font is bad in some way.
    
    It's the only Font Item in the target or source view list
    at that time.
    """
    def __init__( self, TYPE, glyphpaf="" ):
        ## TYPE is: EMPTY (no fonts to display)
        FontItem.__init__( self, glyphpaf )
        self.TYPE = TYPE
    def __queryFontFamilyStyleFlagBad( self ):
        """
        Overridden so that it does not happen for this class.
        """
        pass
    def errorText ( self ):
        if self.TYPE == "EMPTY":
            l1 = _("There are no fonts to see here, move along.")
            l2 = _("(Check your filter!)")
            return ( l1, l2 )

class TruetypeItem( FontItem ):
    def __init__( self, glyphpaf ):
        FontItem.__init__( self, glyphpaf )
        
class TruetypeCollectionItem( FontItem ):
    def __init__( self, glyphpaf ):
        FontItem.__init__( self, glyphpaf )

class OpentypeItem( FontItem ):
    def __init__( self, glyphpaf ):
        FontItem.__init__( self, glyphpaf )
      
class Type1Item( FontItem ):
    def __init__( self, glyphpaf, pfmpaf ):
        FontItem.__init__( self, glyphpaf )
        self.pfmpaf = pfmpaf
        
def itemGenerator( fromObj, sourceList ):
    """
    Prepare for hell...
    This is a *generator* function that yields a FontItem 
    instantiated according to the type of the font.
    Call it once-off; or in a loop to get one FontItem after another.
    Pass it a sourceList that contains pafs.
    """
    def ext(s): return s.split(".")[-1].upper()
    def stripExt(s): return s[:s.rfind(".")]
    
    listOfItemsGenerated = []
    
    
    ## The sourceList that comes in MUST be exactly like 
    ## one that os.listdir() would supply.
    ## When this is called from Folder, all is well.
    
    ## When this is called from Pog, we have fetched a list
    ## of pafs from a text file. 
    ## For the Type1 special case, this means that the PFM 
    ## files are missing. (PFM goes with PFB and we only record
    ## the PFB files in the Pog text file.)
    ## They *may* or may not actually exist as files and
    ## we need to go and look for them.
    
    ## This is horribly complicated and really needs an overhaul...
    
    if isinstance( fromObj, Pog ):
        ## Any PFB files in sourceList?
        pfblist = [f for f in sourceList if ext(f) == "PFB" ]
        if pfblist: # Yes, so find the related PFM files
            ## Each of them may be in their own directory, so we must loop
            for pfbpaf in pfblist:
                ## paf : /some/path/somefont.pfb
                originalDir = os.path.dirname( pfbpaf ) # /some/path
                filename = os.path.basename( pfbpaf ) # somefont.pfb
                pfbfilename = stripExt( filename ) # somefont
                wildcard = os.path.join( originalDir, pfbfilename + u".[Pp][Ff][Mm]" )
                listOfpfms = glob.glob( wildcard )
                ## I am only looking for one match
                thePFM = None
                for pfmpaf in listOfpfms:
                    if pfbfilename in pfmpaf:
                        thePFM = pfmpaf
                        break
                if thePFM:
                    ## We have found a PFM that matches the filename of the PFB
                    sourceList.append( thePFM )
    
    ## Make a unique list of all PFM or PFB files in sourceList and strip the extensions.
    ## This results in only those paths/to/files that are unique files and I will call them
    ## a "font group". They have the same path and the same filename.
    ## I don't grok the set() stuff, but it seems to strip duplicates out and that's what
    ## I want.
    ## I have to strip the extension because, with it, it makes each string unique anyway.
    ## I want to identify /some/path/file1.pfb and /some/path/file1.pfm as /some/path/file1
    ## So this halves the number of Type1 related entries in the list.
    T1FontGroupNamesOnly = \
    list( set( [ stripExt(f) for f in sourceList if ( ext(f) == "PFB" or ext(f) =="PFM" ) ] ) )
    
    ## Do we have any Type1 nonsense to work out?
    if len(T1FontGroupNamesOnly) > 0: #yes.
        ## I can't find any clever way to do this. 
        ## I want a list like so:
        ## [ ["PFM", paf], ["PFB", paf] ]
        ##
        ## It *may* also happen that there is no PFM file at all...
        for pafstub in T1FontGroupNamesOnly:
            tmp=[]
            for paf in sourceList:
                if paf.find(pafstub) != -1:
                    tmp.append( [ext(paf), paf ] )
            ## tmp is the business
            ## Now to make a single Type1Item from this list:
            d = dict(tmp) # cool.
            p = None
            if "PFM" in d: p = d["PFM"]
            fi = Type1Item( d["PFB"], pfmpaf = p )
            #if fi.badfont:
                #fi = InfoFontItem( "BADFONT", d["PFB"] )
            listOfItemsGenerated.append(fi)
            # Jeezuz that only took me 3 days to work out. Man I'm slow.
            
    ## Do the other font types
    TTFList = [ paf for paf in sourceList if ext(paf) == "TTF" ]
    if len(TTFList) > 0:
        for paf in TTFList:
            fi = TruetypeItem( paf )
            #if fi.badfont:
            #    fi = InfoFontItem( "BADFONT", paf )            
            listOfItemsGenerated.append( fi )
                
    OTFList = [ paf for paf in sourceList if ext(paf) == "OTF" ]
    if len(OTFList) > 0:
        for paf in OTFList:
            fi = OpentypeItem( paf )
            #if fi.badfont:
                #fi = InfoFontItem( "BADFONT", paf )              
            listOfItemsGenerated.append(fi)
            
    TTCList = [ paf for paf in sourceList if ext(paf) == "TTC" ]
    if len(TTCList) > 0:
        for paf in TTCList:
            fi = TruetypeCollectionItem( paf )
            #if fi.badfont:
                #fi = InfoFontItem( "BADFONT", paf )              
            listOfItemsGenerated.append(fi)
        
    ## Sort the list: not sure this will work....
    listOfItemsGenerated.sort( cmp=locale.strcoll, key=lambda obj:obj.glyphpaf ) # Try to sort on that field.
    
    ## Supply it: This is pure magic!
    for fi in listOfItemsGenerated:
        yield fi


class BasicFontList(list):
    """
    Ancestor to the Pog and Folder classes.
    """
    def clear(self):
        del self[:] # works. It was real touch and go there for a while.
    def clearInactiveflags(self):
        for fi in self:
            fi.inactive = False

class EmptyView(BasicFontList):
    """
    Imitates an empty Pog or an empty Folder.
    """
    def __init__(self): 
        BasicFontList.__init__(self)
        
        ## Public properties:
        self.name = "EMPTY"
        self.installed = False
        self.empty = True
        
    def label(self):
        return str(self.name)
    ####
    ## generate the list
    def genList(self):
        return 
        
    def isInstalled(self):
        return False


class Folder(BasicFontList):
    """
    Represents an entire Folder (from a path given by user clicking on the
    GenericDirCtrl or from a commanline string.)
    Contains a list of various FontItem Objects.
    Supply the path
    """
    def __init__(self, path):
        BasicFontList.__init__(self)
        ## Public var
        self.path = os.path.abspath(path) # fix relative paths

        ## NEW INFO : DEC 2007
        ## Linux is Posix and that means all filenames are stored as byte strings
        ## "Unlike Windows NT/2000/XP, which always store filenames in Unicode format, 
        ##  POSIX systems (including Linux) always store filenames as binary strings. 
        ##  This is somewhat more flexible, since the operating system itself doesn't 
        ##  have to know (or care) what encoding is used for filenames. The downside 
        ##  is that the user is responsible for setting up their environment 
        ##  ("locale") for the proper coding."
        ##
        ## On Linux: os.path.supports_unicode_filename is always == True
        ## On my system:
        ## >>> locale.getpreferredencoding()
        ## 'UTF-8'
        ##    Return the charset that the user is likely using,
        ##    according to the system configuration.
        ## On my system:
        ## >>> sys.getfilesystemencoding()
        ## 'UTF-8'
        ## This one returns the ENCODING (byte string to unicode) needed to
        ## convert filenames from the O/S *to* unicode.
        ##
        
        listOfFilenamesOnly = []
        ## If self.path is unicode, return will be list of unicode objects (with some slip-ups...)
        ## I have forced the app to be 100% unicode, so this is going to be a Unicode list.
        ## self.path comes from the GenericDirCtrl, which is part of a Unicode wxPython build
        ## therefore it always emits Unicode objects. (Or it comes from cli)
        listOfFilenamesOnly = os.listdir (  self.path  ) # Get the unicode list
        
        ## I could use this:
        ## sourceList = [ os.path.join( self.path, f ) for f in listOfFilenamesOnly ]
        ## But I won't know (when there's an error) which element of the listOfFilenamesOnly was
        ## the culprit, so I'm gonna loop this one.
        sourceList = []
        for f in listOfFilenamesOnly:
            try:
                paf = os.path.join( self.path, f )
                sourceList.append( paf )
            except:
                ## Okay, 'f' was not able to join to self.path
                ## I'ts prob a UnicodeDecodeError, but I don't care.
                ## This is one font that will never make it through
                ## to the app. It won't even become a 'badfont'
                fpsys.logBadStrings(f) # I'm keeping a list of bad names.

        #print sourceList
        
        ## Now employ the generator magic:
        for fi in itemGenerator( self, sourceList ):
            self.append(fi)
            
        if len(self) == 0:
            raise fontybugs.FolderHasNoFonts(self.path)
#TO DO            
        ## I sorted the list in the generator, so if that didn't work, perhaps this will
        ## self.sort ( locale.strcoll ) # Sort myself....
    def __str__(self):
        return str(self.path)
        
    def label( self ):
        """
        A handy way to refer to Folders to Pogs in certain circumstances.
        See around line 1296 wxgui.OnMainClick()
        Pog.label returns the name
        Folder.label returns the path
        """
        return os.path.basename( self.path )
        
class Pog(BasicFontList):
    """
    Represents an entire Pog. 
    Contains a list of various FontItems.
    Dec 2007 - adding OTF and Type 1
    Supply the pog name.
    Must call genList() if you want actual font items in the list.
    """
    def __init__(self, name ): #, progressCallback = None):
        BasicFontList.__init__(self)
        ##private props
        #self.__progressCallback = progressCallback
        self.__pc = PathControl() # My own path control.
        
        ## Public properties:
        self.name = name
        self.__installed = "dirty" #am I installed?
        #self.empty = None #do I have fonts in me?
        self.paf = self.__pc.appPath() + self.name + ".pog"
        #print "Pog __init__ says self.paf is of type:", type(self.paf)
        self.badpog = False #To be used mainly to draw icons and ask user to purge.
        
    def label(self):
        """
        A handy way to refer to Folders to Pogs in certain circumstances.
        See around line 1296 wxgui.OnMainClick()
        Pog.label returns the name
        Folder.label return the path
        These are both the full paf of the font.
        """        
        return self.name
    
    def __openfile(self):
        """
        Open my pog file. Raise PogInvalid error or return a file handle.
        """
        ## NB: NO error is raised if an empty file is opened and read...
        ## If there is some chronic hard drive problem, I assume Python will quit anyway...
        try:
            f = codecs.open( self.paf, "r", sys.getfilesystemencoding())
        except:
            raise fontybugs.PogInvalid( self.paf )
        ## Let's see what kind of line 1 we have
        line1 = f.readline()[:-1]
        self.__installed = "dirty" # unsure as to the status
        if line1.upper() == "INSTALLED": self.__installed = "yes"
        if line1.upper() == "NOT INSTALLED": self.__installed = "no"
        if self.__installed == "dirty":
            ## We have a bad pog.
            print "ABOUT TO RENAME POG"
            self.__renameBadPog()
            raise fontybugs.PogInvalid( self.paf )
        ## At this point, we have a valid pog file:
        ## It has a valid line 1
        ## It may or may not have paf lines below it.
        return f
        
        
    def isInstalled(self):
        """
        Will ONLY raise a PogInvalid error. Any other will abort app.
        """
        if self.__installed == "yes": return True
        if self.__installed == "no": return False
        ## Else it == "dirty" and:
        ## We must open the file to discover the status:
        ## Will raise an error, so don't handle it, let it propogate upwards.
        f = self.__openfile() #sets __installed flag
        f.close()
        if self.__installed == "yes": return True
        if self.__installed == "no": return False        
        
    def __renameBadPog(self):
        """
        This is a bad pog, My plan is to rename it out of the .pog namespace.
        No error detection ... yet
        """
        newpaf =  self.paf[:-4] + ".badpog" #kick out the .pog and append .badpog
        #print "Invalid Pog : \"%s\"\nRenaming it to \"%s\"" % (self.paf, newpaf)
        os.rename(self.paf, newpaf) #just going to hope this works...
        self.paf = newpaf
        
    def genList(self):
        """
        Generate the list of font items within myself.
        Access the disk. Build the object up. All attribs and the list of fonts.
        
        Passes any PogInvalid error directly through.
        """
        f = self.__openfile() #sets install flag, raises PogInvalid error.
        self.clear() #clear is in basicfontlist.py

        sourceList = []
        for paf in f: #This continues from line 2 onwards ...
            paf = paf[:-1] #Strip the damn \n from the file
            sourceList.append(paf) #[path,filename])
        f.close()  
        
        ## Now to make Fontitems out of sourceList
        for fi in itemGenerator( self, sourceList):
            self.append(fi) # store them in myself.
            
    def purge(self):
        """
        ####
        ## Purge method - remove fonts in the pog that are not on disk.
        ##
        ## Raises
        ##          PogEmpty
        ##          PogInstalled
        """
        ## can't purge an empty pog
        if len(self) == 0:
            raise fontybugs.PogEmpty # RAISED :: PogEmpty
        ## can't purge an installed pog
        if self.__installed == "yes":
            raise fontybugs.PogInstalled # RAISED :: PogInstalled
        else:
            ## Let's build a new list of all the bad font items.
            badfonts = []
            for i in self:
                try: #prevent weird errors on path test...
                    if not os.path.exists(i.glyphpaf) :
                        badfonts.append(i) 
                except:
                    pass # it's a bad through and through! It'll be axed too.
            ## Now go thru this list and remove the bad items.
            for bi in badfonts:
                #print "purging:", bi.name
                self.remove(bi) 
            self.write() 

    def install(self):
        """
        ####
        ## Install the fonts in myself to the user's fonts folder.
        ## NOTE:
        ## Even if ONLY ONE font out of a gigazillion in the pog 
        ## actually installs, the POG == INSTALLED.
        ## If we have a font that cannot be sourced, flag BADPOG
        ##
        ## For Type1 fonts - 
        ## The choice I have made is:
        ## I will ONLY reference the PFB in the Pog file.
        ## When we install a Pog, the PFM must be sought
        ## in the original folder and linked.
        ## For uninstall, simply kill the associated PFM
        ##
        ## Raises:
        ##          PogEmpty
        ##          PogAllFontsFailedToInstall
        ##          PogSomeFontsDidNotInstall
        """        
        ## We start thinking all is rosey:
        self.__installed = "yes"

        ## Now we make sure ...
        if len(self) == 0: 
            self.__installed = "no"
            raise fontybugs.PogEmpty(self.name) # RAISED :: PogEmpty
        ## Now we go through the guts of the pog, font by font:
        bugs = 0
        for fi in self:
            linkDestination = os.path.join(self.__pc.userFontPath(), fi.name) 
            
            ## Link it if it ain't already there.
            ## I am not checking for errors on os.symlink ...
            
            if not os.path.exists(linkDestination):  
                if os.path.exists(fi.glyphpaf):
                    os.symlink(fi.glyphpaf, linkDestination)  #Should do the trick.
                    ## Now, the Type1 special case, link the PFM
                    ## I could use:
                    ## myInstance = fi.__class__.__dict__       
                    ## if "pfmpaf" in myInstance
                    if isinstance( fi, Type1Item ):
                        ## It's a Type 1, does it have a pfmpaf?
                        if fi.pfmpaf:
                            linkDestination = \
                            os.path.join( self.__pc.userFontPath(), os.path.basename( fi.pfmpaf ) )
                            os.symlink( fi.pfmpaf, linkDestination )
                else:
                    bugs += 1
        if bugs == len(self): # There was 100% failure to install fonts.
            ## We flag ourselves as NOT INSTALLED
            self.__installed = "no"
            self.write()
            raise fontybugs.PogAllFontsFailedToInstall(self.name) # RAISED :: PogAllFontsFailedToInstall
        elif bugs > 0: 
            ## Some fonts did get installed, but not all. so, we are INSTALLED
            self.write()
            raise fontybugs.PogSomeFontsDidNotInstall(self.name) # RAISED :: PogSomeFontsDidNotInstall
        self.write()

    def uninstall(self):
        """
        ####
        ## Uninstall the fonts.
        ## NOTE:
        ## If any font is NOT removed POG = INSTALLED
        ## Any links that are not found = just ignore: They could have been removed by another pog, or this
        ## could have been a bad pog anyway.
        ## NO BAD POG flag EVER.
        ##
        ## Raises:
        ##          PogEmpty
        ##          PogLinksRemain
        ##          PogNotInstalled
        """        
        if len(self) == 0: raise fontybugs.PogEmpty # RAISED :: PogEmpty
        bugs = 0
        if self.__installed == "yes":
            for fi in self:
                link = os.path.join(self.__pc.userFontPath(), fi.name) #Point at the link in .fonts folder.
                ## Step one - look for the actual file (link)
                if os.path.exists(link):
                    try:
                        os.unlink(link) 
                        ## The Type1 special case - it's PFM may be here...
                        if isinstance( fi, Type1Item ):
                            ## It's a Type 1, does it have a pfmpaf?
                            ## It may be None (if this pfb happens not to have had a pfm.)
                            if fi.pfmpaf:
                                pfmlink = os.path.join(self.__pc.userFontPath(), os.path.basename(fi.pfmpaf))
                                if os.path.exists( pfmlink ):
                                    os.unlink( pfmlink )                     
                    except: # e.g. Permission denied [err 13]
                        ## Only bugs that imply that the file is THERE but CANNOT BE REMOVED
                        ## are classified as bugs. We are making a sweeping assumption here.
                        bugs += 1
            ## Okay, we are currently INSTALLED, so what is the result of the loop?
            if bugs > 0:
                ## We still have fonts in the pog that could NOT be removed, ergo we stay INSTALLED
                raise fontybugs.PogLinksRemain(self.name)  # RAISED :: PogLinksRemain
            else:
                ## Okay - there were no problems, so we are now done.
                self.__installed = "no"
                self.write() #save to disk
        else:
            ## self.__installed says we are not installed:
            raise fontybugs.PogNotInstalled(self.name) # RAISED :: PogNotInstalled
            

    def write(self) :
        """
        ####
        ## Write a pog to disk.
        """
        try:
            #f = open(self.paf, "w") 
            f = codecs.open( self.paf, 'w', sys.getfilesystemencoding() )
            i = "not installed\n"
            if self.__installed == "yes":
                i = "installed\n"
            f.write(i) 
            #Now write the font pafs
            for i in self:
                f.write(i.glyphpaf + "\n") 
            f.close() 
        except:
            raise fontybugs.PogWriteError(self.paf)

    def delete(self):
        """
        ####
        ## Delete my pogfile, then clean myself up, ready to be destroyed.        
        """
        try:
            os.unlink(self.paf)
        except:
            raise fontybugs.PogCannotDelete(self.paf)
        self.clear()
        self.__installed = "no"
