/*
 *  guile.c - Routines for setting up libguile.
 *            This file is part of the FreeLCD package.
 *  
 *  $Id: guile.c,v 1.3 2004/01/16 23:49:36 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */


#if HAVE_CONFIG_H
# include "config.h"
#endif

#include "guile.h"


#if USE_GUILE

#if HAVE_STRING_H
# include <string.h>
#endif

#if HAVE_STRINGS_H
# include <strings.h>
#endif

#if HAVE_DIRENT_H
# include <dirent.h>
#else
# if HAVE_SYS_DIRENT_H
#  include <dirent.h>
# endif
#endif

#if HAVE_TYPES_H
# include <types.h>
#else
# if HAVE_SYS_TYPES_H
#  include <sys/types.h>
# endif
#endif

#include <libguile.h>
#include <libguile/gc.h>
#include <libguile/symbols.h>

#include "../common/xmalloc.h"
extern int keep_going;

static SCM guile_hooks[10];

static SCM
_make_string (const char *text)
{
  SCM lsym;
  
  SCM_NEWCELL (lsym);
  SCM_SETLENGTH (lsym, strlen (text), scm_tc7_ssymbol);
  SCM_SETCHARS (lsym, text);

  return lsym;
}

static SCM
_flcd_server_version (void)
{
  return _make_string (PACKAGE_VERSION);
}

static SCM
_flcd_server_quit (void)
{
  keep_going = 0;

  return SCM_BOOL_T;
}

void
guile_call_hook (guile_hook id, void *data, long param1, long param2)
{
  switch (id)
    {
    case HOOK_NEXT_SCREEN:
      break;
      
    case HOOK_DATA_UPDATE:
      break;
      
    case HOOK_OPEN_CONNECTION:
      break;

    case HOOK_CLOSE_CONNECTION:
      break;
    }
}

int
guile_init ()
{
  const char    *location = getenv ("FLCDD_SCHEME_PATH");
  unsigned int  location_len;
  DIR           *directory;
  struct dirent *entry;
  char          *file_name;
  
  
  scm_c_define_gsubr ("flcd:server-version", 0, 0, 0, _flcd_server_version);  
  scm_c_define_gsubr ("flcd:server-quit"   , 0, 0, 0, _flcd_server_quit);  

  guile_hooks[0] = scm_create_hook ("flcd:hook:next-screen", 1);
  guile_hooks[1] = scm_create_hook ("flcd:hook:data-update", 2);
  guile_hooks[2] = scm_create_hook ("flcd:hook:open-connection", 2);
  guile_hooks[3] = scm_create_hook ("flcd:hook:close-connection", 1);

  if (location == NULL)
    location = "../scripts/";

  directory = opendir (location);
  if (directory == NULL)
    return 0;

  location_len = strlen (location);
  file_name = xmalloc (location_len + 256);
  memcpy (file_name, location, location_len);

  if (file_name[location_len - 1] != '/')
    {
      file_name[location_len] = '/';
      ++location_len;
    }
  
  while ((entry = readdir (directory)) != NULL)
    {
      unsigned int entry_length;

      if (entry->d_name == NULL)
        break;
      
      entry_length = strlen (entry->d_name);
      if (entry_length > 255)
        {
          /* That's a long file name by any standard. I don't trust it. */
          continue;
        }
      
      if (entry_length < 4 
          || strncasecmp (entry->d_name + entry_length - 4, ".scm", 4) != 0)
        continue;

      memcpy (file_name + location_len, entry->d_name, entry_length + 1);
      /* FIXME */
      /* scm_primitive_load (_make_string (file_name)); */
    }
  
  free (file_name);
  closedir (directory);

  return 1;
}

#else /* ! USE_GUILE */

int
guile_init ()
{ return 1; }

void
guile_call_hook (guile_hook id, void *data, long param1, long param2)
{
  (void)id;
  (void)data;
  (void)param1;
  (void)param2;
}

#endif /* USE_GUILE */
