/*
 *  connections.c : Manage connected sockets and the incoming data.
 *                  This file is part of the FreeLCD package.
 *
 *  $Id: connections.c,v 1.7 2004/01/25 15:57:27 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/** \file connections.c */

#include <errno.h>

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#else
# ifdef HAVE_SYS_UNISTD_H
#  include <sys/unistd.h>
# endif
#endif

#if HAVE_ASSERT_H
# include <assert.h>
#else
# define assert(FOO)
#endif

#if HAVE_STRING_H
# include <string.h>
#endif

#ifdef HAVE_STRINGS_H
# include <strings.h>
#endif

#include "common/xmalloc.h"
#include "common/socket.h"
#include "common/debug.h"
#include "connections.h"
#include "scheduler.h"


/** This struct keeps the state information of a connection to a client. */
typedef struct
{
  /** Socket handle. */
  int        sock;
  /** Address of the other side of the connection. */
  in_addr_t  originator;
  /** The XML parser context associated with this connection. */
  void       *xml_parser_context;
  /** A list of screens that the client has defined. It contains elements
   * of type const char *. */
  slist      registered_screens;
}
connection;

/** The list of active connections. */
static slist connectionlist;

/*------------------------------------------------------------------------*/
/* XML dictionaries  */

typedef enum
{
  SCREEN = 1, DATA, COMMAND, IMPORTANCE, REPEAT, FREQUENCY, LAYOUT
}
tag_t;

static tag_t conn_tagindex[] =
{
  SCREEN, DATA, COMMAND, IMPORTANCE, REPEAT, FREQUENCY, LAYOUT
};

static dict_pair conn_tags_d[] =
{
  { "command",    conn_tagindex + 2 },
  { "data",       conn_tagindex + 1 },
  { "frequency",  conn_tagindex + 5 },
  { "layout",     conn_tagindex + 6 },
  { "importance", conn_tagindex + 3 },
  { "repeat",     conn_tagindex + 4 },
  { "screen",     conn_tagindex + 0 }
};

static dictionary conn_tags = { conn_tags_d, 
                               sizeof (conn_tags_d) / sizeof (dict_pair) };


typedef enum
{
  NAME = 1
}
attr_t;

static attr_t conn_attrindex[] = 
{
  NAME
};

static dict_pair conn_attrs_d[] =
{
  { "name",     &conn_attrindex[0] }
};

static dictionary conn_attrs = { conn_attrs_d,
                                 sizeof (conn_attrs_d) / sizeof (dict_pair) };




/*----------------------------------------------------- _string_compare --*/

/** An internal function, used as an extra argument to slist_find_if() .
 * \param _str1 First string to compare.
 * \param _str2 Second string to compare.
 * \return 0 if the strings are not equal, any other value if they are.
 */
static int
_string_compare (const void *_str1, const void *_str2)
{
  const char *str1 = (const char *)_str1;
  const char *str2 = (const char *)_str2;

  assert (str1 != NULL);
  assert (str2 != NULL);

  return strcmp (str1, str2) == 0;
}

/*------------------------------------------------------ _handle_screen --*/

/** Handle screen definition data. If the screen was not registered yet, it
 * is added to the list. If it was registered, its settings are updated.
 * \param info The connection status associated with the client that sent
 *             the screen definition.
 * \param data Screen definition.
 */
static void
_handle_screen (connection *info, xml_node *data)
{
  sch_importance importance = IMP_NORMAL;
  int            repeat     = 10;
  const char*    name       = xmlt_get_attrib (data, NAME);
  xml_node       *found_lay = xmlt_find (data, NULL, LAYOUT);
  xml_node       *found_imp = xmlt_find (data, NULL, IMPORTANCE);
  xml_node       *found_rep = xmlt_find (data, NULL, REPEAT);
  
  if (found_lay == NULL)
    {
      /* This screen has no layout, and therefore no use. 
       * FIXME: Send an error. */
      debug ("no layout was found");

      return;
    }
  
  if (name == NULL)
    {
      /* Screens must have a name. FIXME: Error message. */
      debug ("screen has no name");

      return;
    }
  
  if (found_imp != NULL)
    {
      const char* importance_str = xmlt_get_first_cdata (found_imp);
  
      if (importance_str != NULL)
        {
          if (!strcmp (importance_str, "background"))
            importance = IMP_BACKGROUND;
  
          else if (!strcmp (importance_str, "normal"))
            importance = IMP_NORMAL;
  
          else if (!strcmp (importance_str, "notification"))
            importance = IMP_NOTIFICATION;
  
          else if (!strcmp (importance_str, "alert"))
            importance = IMP_ALERT;

          else
            {
              debug_2 ("%s is not a valid screen importance", importance_str);
              return;
            }
        }
    }
  
  if (found_rep != NULL)
    {
      const char* repeat_str = xmlt_get_first_cdata (found_rep);
  
      if (repeat_str != NULL)
        {
          if (!strcasecmp (repeat_str, "infinite"))
            repeat = -1;
  
          else
            {
              long tmp = strtol (repeat_str, 0, 10);
  
              if (errno == ERANGE || errno == EINVAL)
                {
                  /* FIXME: send an error message. */
                  return;
                }
                  
              if (tmp < 1 || tmp > 65536)
                {
                  /* FIXME: send an out of range error message. */
                  return;
                }
  
              repeat = tmp;
            }
        }
    }
  
  if (slist_iter_at_end (
      slist_find_if (&info->registered_screens, name, _string_compare)))
    {
      debug_2 ("adding screen %s to list of registered screens", name);
      slist_append (&info->registered_screens, (void *)xstrdup (name));
    }

  sch_add_screen ((void *)info, name, importance, repeat, found_lay);
}

/*-------------------------------------------------------- _handle_data --*/
static void
_handle_data (connection *info, xml_node *data)
{
  const char *name = xmlt_get_attrib (data, NAME);

  if (name == NULL)
    {
      /* FIXME: send an error message. */
      return;
    }
  
  if (slist_iter_at_end (
      slist_find_if (&info->registered_screens, name, _string_compare)))
    {
      /* No screen has been registered for this data set yet.
       * FIXME: send an error message. */
      debug_2 ("no screen named %s registered yet.", name);
      return;
    }

  debug_2 ("updating data for %s.", name);
  sch_update_data ((void *)info, name, data);
}

/*----------------------------------------------------- _handle_command --*/
static void
_handle_command (connection *info, xml_node *data)
{
  /* FIXME: implement */
  (void)info;
  (void)data;
}

/*------------------------------------------------------- _xml_callback --*/
static void 
_xml_callback (void *_info, xml_node *data)
{
  connection *info = (connection *)_info;
  
  /* Find out what kind of message we received by looking at the root tag. */
  switch (data->tag)
    {
    case SCREEN:
      _handle_screen (info, data);
      break;
    
    case DATA:
      _handle_data (info, data);
      break;

    case COMMAND:
      _handle_command (info, data);
      break;

    default:
      /* FIXME: Send back an error message or something. */
      break;
    }

  xmlt_free_document (data);
}

/*-------------------------------------------------------- _xml_compare --*/
static int
_xml_compare (const void *_conn, const void *_sock)
{
  const connection *conn = (const connection *)_conn;
  const int sock = *(const int *)_sock;

  return conn->sock == sock;
}

/*----------------------------------------------------------- conn_open --*/
void
conn_open (int sock, struct sockaddr_in* s_in)
{
  connection *new_conn = xmalloc (sizeof (connection));
  
  new_conn->sock = sock;
  new_conn->originator = s_in->sin_addr.s_addr;
  new_conn->xml_parser_context = 
    xmlt_create_context (_xml_callback, new_conn, &conn_tags, &conn_attrs);
  slist_init (&new_conn->registered_screens);
  
  slist_append (&connectionlist, new_conn);
}

/*----------------------------------------------------------- conn_data --*/
void
conn_data (int sock, char *buf, size_t size)
{
  connection *conn;
  slist_iter found = slist_find_if (&connectionlist, &sock, _xml_compare);

  if (slist_iter_at_end (found))
    {
      /* Okay this is weird. Close the socket and pretend nothing happened. */
      close (sock);
      return;
    }
  
  conn = (connection *)slist_at_iter (found);
  if (!xmlt_parse (conn->xml_parser_context, buf, size))
    {
      /* FIXME: Send something back to the client? */
      xmlt_reset_context (conn->xml_parser_context);
    }
}

/*---------------------------------------------------------- conn_close --*/
void
conn_close (int sock)
{
  slist_iter found = slist_find_if (&connectionlist, &sock, _xml_compare);

  if (!slist_iter_at_end (found))
    {
      connection *removed = slist_remove_at_iter (&connectionlist, found);

      slist_delete_special (&removed->registered_screens, free);
      xmlt_free_context (removed->xml_parser_context);
      free (removed);
    }
}

/*---------------------------------------------------------- conn_reply --*/
int
conn_reply (void *originator, const char *data, size_t len)
{
  connection *info = (connection *)originator;

  if (socket_send (info->sock, data, len) == -1)
    {
      slist_iter found = slist_find (&connectionlist, &info);

      if (!slist_iter_at_end (found))
        {
          slist_remove_at_iter (&connectionlist, found);
          slist_delete (&info->registered_screens);
          xmlt_free_context (info->xml_parser_context);
          free (info);
        }

      return 0;
    }

  return 1;
}
