/*
 *  charrenderer.c - Simple module for rendering a character-based layout
 *                   to a canvas.
 *                   This file is part of the FreeLCD package.
 *  
 *  $Id: charrenderer.c,v 1.5 2004/01/25 15:56:37 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>

#if HAVE_STRING_H
# include <string.h>
#elif HAVE_STRINGS_H
# include <strings.h>
#endif

#include "charrenderer.h"
#include "common/field_types.h"
#include "common/debug.h"

/*----------------------------------------------------------- cr_render --*/
void
cr_render (cl_layout *layout, dictionary *data, cc_canvas *canvas)
{
  cl_layout_elem *elem = layout->layout_data;
  size_t i;

  for (i = 0; i < layout->layout_data_size; ++i, ++elem)
    {
      if (elem->is_label)
        {
          cc_write_text (canvas, elem->x, elem->y, elem->utf8_data);
        }
      else
        {
          char temp[256];
          field *fld = (field *)dict_lookup (data, elem->utf8_data);

          if (fld == NULL)
            continue;

          switch (fld->type)
            {
              case F_SCALAR:
                sprintf (temp, "%li", fld->data.scalar);
                break;

              case F_PERCENTAGE:
                sprintf (temp, "%i%%", (int)(fld->data.percentage * 100));
                break;

              case F_LABEL:
                if (fld->data.text != NULL)
                    strncpy (temp, fld->data.text, 255);
                else
                    temp[0] = '\0';

                break;
                  
              case F_TIME:
                debug_2 ("time, width %i", elem->width);
                if (elem->width >= 11)
                    sprintf (temp, "%02i:%02i:%02i.%02i",
                             fld->data.time.hours, 
                             fld->data.time.minutes,
                             fld->data.time.seconds, 
                             fld->data.time.milliseconds / 10);
                
                else if (elem->width >= 8)
                    sprintf (temp, "%02i:%02i:%02i",
                             fld->data.time.hours, 
                             fld->data.time.minutes,
                             fld->data.time.seconds);

                else if (elem->width >= 5)
                    sprintf (temp, "%02i:%02i",
                             fld->data.time.hours, 
                             fld->data.time.minutes);

                else if (elem->width >= 2)
                    sprintf (temp, "%02i:%02i", fld->data.time.hours);

                break;

              default:
                debug_2 ("unknown element %i", fld->type);
                snprintf (temp, 256, "%s", "---------------------------------");
                temp[elem->width] = 0;
            }

          cc_write_text (canvas, elem->x, elem->y, temp);
        }
    }
}
