/*
 *  charcanvas.c - Canvas for character-oriented devices.
 *                 This file is part of the FreeLCD package.
 *  
 *  $Id: charcanvas.c,v 1.5 2004/01/25 15:51:14 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

#include <stdlib.h>

#include <common/xmalloc.h>
#include "charcanvas.h"

/*---------------------------------------------------- cc_create_canvas --*/
cc_canvas *
cc_create_canvas (unsigned int width, unsigned int height)
{
  size_t i;
  size_t size = width * height;
  cc_elem *element;
  cc_canvas *new = xmalloc (sizeof (cc_canvas));
   
  new->width = width;
  new->height = height;
  new->elements = xmalloc (size * sizeof (cc_elem));
  element = new->elements;
  
  for (i = 0; i < size; ++i, ++element)
    {
      element->c = L' ';
      element->damaged_flag = 0;
      element->bitmap = 0;
    }
  
  return new;
}

/*---------------------------------------------------- cc_delete_canvas --*/
void
cc_delete_canvas (cc_canvas *canvas)
{
  int i;
  int size = canvas->width * canvas->height;
  cc_elem *element = canvas->elements;
  
  for (i = 0; i < size; ++i, ++element)
    free (element->bitmap);

  free (canvas->elements);
  free (canvas);
}

/*--------------------------------------------------------- cc_put_char --*/
void
cc_put_char (cc_canvas *canvas, unsigned int x_pos, unsigned int y_pos, 
             wchar_t character)
{
  cc_elem *element = cc_get_element (canvas, x_pos, y_pos);

  if (element != NULL)
    {
      element->c = character;
      element->damaged_flag = 1;
    }
}

/*------------------------------------------------------- cc_write_text --*/
void
cc_write_text (cc_canvas *canvas, unsigned int x_pos, unsigned int y_pos,
               char *utf8)
{
  wchar_t curr;
  unsigned char u, u2, u3, u4, u5, u6;
  
  while ((u = *utf8) != '\0')
    {
      curr = 0;

      if (u < 0x80)
        {
          /* Range 0 - 7F */
          curr = u;
        }
      else if (u <= 0xc0 || (u2 = *(utf8 + 1)) & 0xc0 != 0x80)
        {
          /* Invalid character */
          return;  
        }
      else if ((u & 0xe0) == 0xc0)
        {
          /* Range 80 - 7FF */
          curr = u & 0x1f;
          curr <<= 6;
          curr |= u2 & 0x3f;
          ++utf8;
        }
      else if (u <= 0xe0 || (u3 = (*utf8 + 2)) & 0xc0 != 0x80)
        {
          /* Invalid character */
          return;  
        }
      else if ((u & 0xf0) == 0xe0)
        {
          /* Range 800 - FFFF */
          curr = u & 0x0f;
          curr <<= 6;
          curr |= u2 & 0x3f;
          curr <<= 6;
          curr |= u3 & 0x3f;
          utf8 += 2;
        }
      else if (u <= 0xf0 || (u4 = (*utf8 + 3)) & 0xc0 != 0x80)
        {
          /* Invalid character */
          return;  
        }
      else if ((u & 0xf8) == 0xf0)
        {
          /* Range 10000 - 1FFFFF */
          curr = u & 0x07;
          curr <<= 6;
          curr |= u2 & 0x3f;
          curr <<= 6;
          curr |= u3 & 0x3f;
          curr <<= 6;
          curr |= u4 & 0x3f;
          utf8 += 3;
        }
      else if (u <= 0xf8 || (u5 = (*utf8 + 4)) & 0xc0 != 0x80)
        {
          /* Invalid character */
          return;  
        }
      else if ((u & 0xfc) == 0xf8)
        {
          /* Range 200000 - 3FFFFFF */
          curr = u & 0x03;
          curr <<= 6;
          curr |= u2 & 0x3f;
          curr <<= 6;
          curr |= u3 & 0x3f;
          curr <<= 6;
          curr |= u4 & 0x3f;
          curr <<= 6;
          curr |= u5 & 0x3f;
          utf8 += 4;
        }
      else if (u <= 0xfc || (u6 = (*utf8 + 4)) & 0xc0 != 0x80)
        {
          /* Invalid character */
          return;  
        }
      else if ((u & 0xfe) == 0xfc)
        {
          /* Range 4000000 - 7FFFFFFF */
          curr = u & 0x01;
          curr <<= 6;
          curr |= u2 & 0x3f;
          curr <<= 6;
          curr |= u3 & 0x3f;
          curr <<= 6;
          curr |= u4 & 0x3f;
          curr <<= 6;
          curr |= u5 & 0x3f;
          curr <<= 6;
          curr |= u6 & 0x3f;
          utf8 += 5;
        }
      else
        {
          /* Invalid character */
          return;
        }

      cc_put_char (canvas, x_pos++, y_pos, curr);
      ++utf8;
    }
}

/*------------------------------------------------------ cc_get_element --*/
cc_elem*
cc_get_element (cc_canvas *canvas, unsigned int x_pos, unsigned int y_pos)
{
  if (x_pos >= canvas->width || y_pos >= canvas->height)
    return NULL;

  return canvas->elements + x_pos + (y_pos * canvas->width);
}

/*-------------------------------------------- cc_clear_damaged_regions --*/
void
cc_clear_damaged_regions (cc_canvas *canvas)
{
  size_t i;
  size_t size = canvas->width * canvas->height;
  cc_elem *element = canvas->elements;
  
  for (i = 0; i < size; ++i, ++element)
    element->damaged_flag = 0;

  free (canvas->elements);
  free (canvas);
}

void
cc_copy_region (cc_canvas *src, unsigned int sx, unsigned int sy,
                unsigned int width, unsigned int height, 
                cc_canvas *dest, unsigned int dx, unsigned int dy)
{
  /* \todo Implement character canvas copy region function. */  
  (void)src;
  (void)sx;
  (void)sy;
  (void)width;
  (void)height;
  (void)dest;
  (void)dx;
  (void)dy;
}
