/*
 *  dictionary.c - "Dictionary" data structure, works a bit like std::map.
 *                 This file is part of the FreeLCD package.
 *
 *  $Id: dictionary.c,v 1.5 2003/08/20 22:31:08 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#if HAVE_STRING_H
# include <string.h>
#endif

#if HAVE_ASSERT_H
# include <assert.h>
#else
# define assert(FOO)
#endif

#include "dictionary.h"

/*-------------------------------------------------------- dict_lookup --*/
void *
dict_lookup (const dictionary *d, const char *key)
{
  dict_pair *top;
  dict_pair *middle;
  dict_pair *bottom;
  int result;

  assert (d != NULL);
  assert (key != NULL);

  /* Looking up a key in the dictionary works as follows: TOP and BOTTOM
   * point to the first element and right after the last element in the
   * part that is being searched (initially, this is the entire dictionary).
   * MIDDLE always points to the element halfway between TOP and BOTTOM.
   * In every iteration of the while loop, the element at MIDDLE is compared
   * to KEY.  If they are equal, the entry in the dictionary was found, and
   * the result is returned.
   * If KEY is lexicographically less than MIDDLE, it can probably be found
   * somewhere between TOP and MIDDLE, so this becomes the new search space.
   * The middle becomes the new bottom, and MIDDLE is recalculated.  If it's
   * the other way, the new search takes place between MIDDLE and BOTTOM.
   * If the search space is empty after this, the key is not in the dictionary,
   * and the while loop exits.
   */
  top = d->dict;
  bottom = d->dict + d->size;

  while (top < bottom)
    {
      middle = top + ((bottom - top) / 2);

      result = strcmp (key, middle->key);
      if (result == 0)
        return middle->value;

      if (result < 0)
        bottom = middle;
      else
        top = middle + 1;
    }

  return NULL;
}

/*--------------------------------------------------- dict_pair_compare --*/
int
dict_pair_compare (const void* _first, const void* _second)
{
  const dict_pair *first = _first;
  const dict_pair *second = _second;

  return strcmp (first->key, second->key);
}



/*------------------------------------------------------------------------
 * UNIT TEST
 *------------------------------------------------------------------------
 */

#ifdef UNIT_TEST_DICTIONARY_C

#include <stdio.h>

int lookup (const dictionary* dict, const char* key)
{
  int *valptr = (int*)(dict_lookup (dict, key));

  return valptr ? *valptr : 999;
}

int main (int argc, char **argv)
{
  int values[] = { 5, 6, 7, 8, 9 };
  int result;
  int i;
  
  dict_pair pair_array[] = { 
      { "aardbei", values + 0 },
      { "banaan", values + 1 },
      { "citroen", values + 2 },
      { "druif", values + 3 },
      { "eland", values + 4 } };

  dictionary null = { pair_array, 0 };
  dictionary tiny = { pair_array, 1 };
  dictionary dict = { pair_array, 5 };
  
  /* First we test boundary cases: the empty dictionary, and the dictionary
   * with only one entry. */

  result = lookup (&null, pair_array[0].key);
  if (result != 999)
    {
      printf ("Lookup of parameter in empty dictionary " \
              "succeeded unexpectedly.\n");
      exit (1);
    }
  
  result = lookup (&tiny, pair_array[0].key);
  if (result != values[0])
    {
      printf ("Lookup of parameter in tiny dictionary failed.\n");
      exit (1);
    }

  result = lookup (&tiny, pair_array[1].key);
  if (result != 999)
    {
      printf ("Lookup of parameter in tiny dictionary succeeded " \
              "unexpectedly.\n");
      exit (1);
    }

  /* Then we test every element in a dictionary of size 5. */

  for (i = 0 ; i < 5 ; ++i)
    {
      result = lookup (&dict, pair_array[i].key);
      if (result != values[i])
        {
          printf ("Lookup of parameter %s returned %i, expected %i/n",
                  pair_array[i].key, result, values[i]);
          exit (1);
        }
    }

  if (lookup (&dict, "xxx") != 999)
    {
      printf ("Lookup of non-existing parameter did not fail, " \
              "but should have/n");
      exit (1);
    }

  exit (0);
}

#endif
