/*
	Copyright (C) 2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <string>
#include <strings.h>

#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "FilterInBase.h"
#include "NullInFilter.h"
#include "ZLibInFilter.h"
#include "Dicer.h"

#define FLUSH_POINT 8192

Dicer::Dicer(PROMPT_USER_t prompt_user)
{
	m_size = 0;
	m_prompt_user = prompt_user;
  m_no_prompt = false;
	m_slice_offset = 0;
	m_slice_num = 0;
  m_last_slice_num = 0;
	m_slice_file_handle = NULL;
	m_isdevicefile = false;
}

Dicer::~Dicer()
{
	if(m_slice_file_handle)
		fclose(m_slice_file_handle);
}

/*
	Function:
		SetArchivePath
		
	Arguments:
		filename - basename or devicefile of archive to open
		devicefile - true if filename is that of a device file
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Sets variables so Dicer can figure out what file contains slice #4, etc...
*/
bool Dicer::SetArchivePath(const char *filename, bool devicefile)
{
	if(m_slice_file_handle)
		return false;
	
	m_isdevicefile = devicefile;
	if(m_isdevicefile)
	{
		m_slice_path = filename;
		m_slice_base_name = "";
	}
	else
	{
		if(!SetFullName(filename))
			return false;
	}
	
	return true;
}

/*
	Function:
		SetFullName
		
	Arguments:
		full_name - path and basename for slice
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Sets the path and basename of the slice
*/
bool Dicer::SetFullName(const std::string & full_name)
{
	if(full_name.empty())
		return false;

	int last_sep = full_name.find_last_of('/');
	
	if(last_sep >= 0) {
		if(last_sep > 0)
			m_slice_path = full_name.substr(0, last_sep) + "/";
		else
			m_slice_path = "/";
		m_slice_base_name = full_name.substr(last_sep+1);
	} else {
		m_slice_path = "./";
		m_slice_base_name = full_name;
	}
	
	if(m_slice_base_name.empty())
		return false;

	int ext_sep = m_slice_base_name.find_last_of('.');
	
	if((ext_sep >= 0) && (!strcasecmp(m_slice_base_name.substr(ext_sep).c_str(), ".fbk"))) {
		fprintf(stderr, "Basename looks like an archive. Aborting.\n");
		return false;
	}

	return true;
}

/* FIXME: All these PrintSettings functions are probably outdated */
void Dicer::PrintSettings()
{
	fprintf(stderr, "Slice Size: %lli bytes\nSlice Path: %s\nSlice Base Name: %s\n", m_size, m_slice_path.c_str(), m_slice_base_name.c_str());
}

/*
	Function:
		SwitchSlice
		
	Arguments:
		slice_num - slice number to switch to
		lastslice - true if user wants the last slice
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Switches to a different slice
TODO: This function needs a few more comments
*/
bool Dicer::SwitchSlice(unsigned int slice_num, bool lastslice)
{
	if((lastslice && m_last_slice_num && (m_slice_num == m_last_slice_num)) || (slice_num == m_slice_num))
		return true; /* Requested slice we are on, return success */
	
  if(!m_no_prompt)
  {
    bool breakout = false;
    while(!breakout)
    {
      int ret;
      if(lastslice) {
        ret = m_prompt_user("Insert the LAST DISK\n'c' to continue, 'q' to cancel.\n");
      } else {
        char msgbuf[256];
        snprintf(msgbuf, 256, "Insert disk %u\n'c' to continue, 'q' to cancel.\n", slice_num);
        ret = m_prompt_user(msgbuf);
      }
      switch(ret)
      {
        case 'c':
          breakout = true;
          break;
        case 'q':
          return false;
        default:
          fprintf(stderr, "Pressed key not a valid choice!\n");
          break;
      }
    }
  }
	
	if(m_slice_file_handle) {
		fclose(m_slice_file_handle);
		m_slice_file_handle = NULL;
	}
	
	m_slice_offset = 0;
	
	if(lastslice && (!m_isdevicefile))
	{
		for(m_slice_num = 2; m_slice_num < 4294967295UL; m_slice_num++)
		{
			struct stat stbuf;
			if(stat(SliceFileName().c_str(), &stbuf))
			{
				m_slice_num--;
        m_last_slice_num = m_slice_num;
				break;
			}
		}
	}
	else
	{
    if(lastslice)
    {
      m_last_slice_num = UINT_MAX;
      m_slice_num = m_last_slice_num;
    }
    else
    {
      m_slice_num = slice_num;
    }
	}

	m_slice_file_handle = fopen(SliceFileName().c_str(), "rb");
	
	return m_slice_file_handle ? true : false;
}

/*
	Function:
		SliceFileName
		
	Arguments:
		None
	
	Returns:
		Current slice filename
		
	Remarks:
		This function probably doesn't do the right thing.
*/
std::string Dicer::SliceFileName()
{
	if(m_isdevicefile)
		return m_slice_path;

	char buf[11];
	snprintf(buf, 11, "%u", m_slice_num);
	return (m_slice_path + m_slice_base_name + std::string(".") + std::string(buf) + std::string(".fbk"));
}

/*
	Function:
		SetSize
		
	Arguments:
		size - size of slice
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Sets the size of a slice
*/
bool Dicer::SetSize(const long long int size)
{
	if(size < 0)
		return false;
	m_size = size;
	return true;
}

/*
	Function:
		HandleSlices
		
	Arguments:
		None
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Opens next slice if necessary
*/
bool Dicer::HandleSlices()
{
	if((m_slice_num == 0) || (m_size && (m_slice_offset == m_size)))
		return SwitchSlice(m_slice_num+1, false);
	
	return true;
}

/*
	Function:
		ReadAmmount
		
	Arguments:
		None
	
	Returns:
		the ammount of data to read
		
	Remarks:
		Determines the next ammount of data to read so we don't overflow slices.
*/
size_t Dicer::ReadAmmount()
{
	/* Determine if we need to flush data to disk */
	if(m_size)
  {
    size_t flush_point = MIN(FLUSH_POINT, m_size);
		return (off_t(flush_point)+m_slice_offset > m_size ? m_size-m_slice_offset : flush_point); /* Only flush up to m_size */
  }
	else
		return FLUSH_POINT;
}

/*
	Function:
		GetArchivePosition
		
	Arguments:
		None
	
	Returns:
		The offset of the archive, much like ftello()
		
	Remarks:
		Returns the offset of archive.
		FIXME: Assumes that all previous slices are exactly m_size bytes long
*/
off_t Dicer::GetArchivePosition()
{
	return ((m_size * (m_slice_num-1))+m_slice_offset);
}

/*
	Function:
		ReadData
		
	Arguments:
		buffer - where to put data
		length - length to read
	
	Returns:
    0 - Success
		-1 - Switching slices failed
    -2 - Filesystem read error
    -3 - Partial read error, corrupted data is 0 padded
		
	Remarks:
		Reads data from the archive without filtering it.
*/
int Dicer::ReadData(std::string &buffer, unsigned int length)
{
  int ret = 0;
	size_t ammt;
	unsigned int bread = 0;
	while(bread < length)
	{
		/* Swap out slices if necessary */
		if(!HandleSlices())
			return -1;
	
		ammt = MIN(length-bread, ReadAmmount());
		
    unsigned int c = 0;
		while(c < ammt)
		{
			off_t read_ammt = 0;
			char buf[FLUSH_POINT];
			
			if((read_ammt = fread(buf, 1, ammt-c, m_slice_file_handle)) != ammt-c)
				if(ferror(m_slice_file_handle)) /* Tell the user there was a filesystem read error */
					return -2;
				
			/* If something was read */
			if(read_ammt > 0)
			{
				buffer.append(buf, read_ammt);
				
				bread += read_ammt;
        c += read_ammt;
				m_slice_offset += read_ammt;
			}
			
			/* If successful, don't go any further */
			if(c == ammt)
				break;
			
      /*********** ERRORS BEYOND HERE IN THIS LOOP *******************/
      
			/* Check for EOF and switch slices if found */
			if(feof(m_slice_file_handle)) {
				if(SwitchSlice(m_slice_num+1, false)) {
					continue;
				} else {
					return -1;
				}
			}
			
      /*********** FATAL ERRORS BEYOND HERE IN THIS LOOP *************/
      
			/* If not eof then must've been an error */
			/* Seek one byte foreward */
			if(!InternalSeek(1, SEEK_CUR))
				return -2;

      ret = -3; /* Later return Partial read error */
      
			/* Pad the buffer with a 0 */
			*buf = '\0';
			buffer.append(buf, 1);
			bread++;
      c++;
      m_slice_offset++;
		}
	}
	
	return ret;
}

/*
	Function:
		Seek
		
	Arguments:
		offset - offset to seek using method
		method - currently only SEEK_END and SEEK_SET are implemented
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Seeks to an unfiltered offset in the archive.
*/
bool Dicer::Seek(off_t offset, int method)
{
	if(m_size)
	{
		switch(method)
		{
			case SEEK_END:
				/* Switch to the last slice */
				if(!SwitchSlice(0, true))
					return false;
				/* Seek to the end of the last slice */
				if(!InternalSeek(0, SEEK_END))
          return false;
        
        if(offset > m_slice_offset)
        {
          /* Seek back by 1 or more slices */
          off_t seek_back_by_n_slices = (((offset-m_slice_offset)-1)/m_size)+1;
          
          if(!SwitchSlice(m_slice_num-seek_back_by_n_slices, false))
            return false;
          if(!InternalSeek((seek_back_by_n_slices*m_size)-(offset-m_slice_offset), SEEK_SET))
            return false;
        }
        else
        {
          if(!InternalSeek(m_slice_offset-offset, SEEK_SET))
            return false;
        }
        break;
      case SEEK_SET:
        if(offset < 0)
          return false;
        if(!SwitchSlice((offset/m_size)+1, false))
          return false;
        if(!InternalSeek((offset%m_size), SEEK_SET))
          return false;
        break;
      /* TODO: Implement SEEK_CUR if needed */
		}
	}
	else
	{
		/* Don't need to worry about slices, only one slice */
		if(!InternalSeek(offset, method))
			return false;
	}
	
	return true;
}

/*
	Function:
		InternalSeek
		
	Arguments:
		offset - offset to seek using method
		method - seek method to use
	
	Returns:
		true - success
		false - failure
		
	Remarks:
		Does a fseek and updates the m_slice_offset
*/
bool Dicer::InternalSeek(off_t offset, int method)
{
	if(fseeko(m_slice_file_handle, offset, method) == -1)
		return false;
	
	if((m_slice_offset = ftello(m_slice_file_handle)) == -1)
		return false;
	
	return true;
}

