/*
	Copyright (C) 2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "FilteredMemoryCache.h"

#define FLUSH_POINT 8192

FilteredMemoryCache::FilteredMemoryCache(FilterOutBase *filter, off_t max_mem, TArchiverWriteCallback write_callback)
{
  m_write = write_callback;
  m_filter = filter;
  m_mem = NULL;
  m_max_mem = max_mem;
  m_total_bytes = 0;
  m_no_mem = false;
}

FilteredMemoryCache::~FilteredMemoryCache()
{
  if(m_filter)
    delete m_filter;
  if(m_mem)
    free(m_mem);
}

bool FilteredMemoryCache::PutData(const char *data, size_t len)
{
  if(!m_filter)
    return false;

  WriteFilter(data, len);

	return true;
}

bool FilteredMemoryCache::Flush()
{
  if(!m_filter)
    return false;

	if(!m_filter->Flush())
		return false;

	return WriteFilter(NULL, 0);
}

/* Push the data through another filter and then store it in memory */
bool FilteredMemoryCache::WriteFilter(const char *buffer, unsigned int length)
{
	std::string data;

	/* Process data in filter */
	if(buffer)
		m_filter->PutData(buffer, length);

	/* Write the data to disk if it is past the flush point */
	while(m_filter->NeedFlush(FLUSH_POINT))
	{
		if(!m_filter->GetData(data, FLUSH_POINT))
			return false;
		
		if(!MemoryWrite(data.c_str(), data.length()))
			return false;
	}
	
	data.clear();
	
	/* Flush the rest if the buffer is NULL */
	if(!buffer)
	{
		while(!m_filter->Empty())
		{
			if(!m_filter->GetData(data, FLUSH_POINT))
				return false;
			
			if(!MemoryWrite(data.c_str(), data.length()))
				return false;
		}
	}
	
	return true;
}

bool FilteredMemoryCache::MemoryWrite(const char *buffer, unsigned int length)
{
  /* Reallocate memory until we run out or reach the user specified limit.
     Once memory runs out, give up and free the entire buffer. */
  /* This keeps counting even if the memory has filled up */
  m_total_bytes += length;
  
  if(!m_no_mem)
  {
    /* No memory full conditions yet, good.
       Check for new memory full conditions now */
    if(m_total_bytes > m_max_mem)
    {
      /* Sorry, reached the user specified memory limit */
      if(m_mem)
      {
        /* Free the memory buffer */
        free(m_mem);
        m_mem = NULL;
      }
      /* Set memory full flag */
      m_no_mem = true;
    }
    else
    {
      void *new_mem = realloc(m_mem, m_total_bytes);
      if(!new_mem)
      {
        /* Oops, unable to reallocate more memory */
        /* Free the memory buffer */
        free(m_mem);
        m_mem = NULL;
        /* Set memory full flag */
        m_no_mem = true;
      }
      else
      {
        /* Append the data to the memory buffer */
        m_mem = new_mem;
        memcpy(((char*)m_mem)+((unsigned int)m_total_bytes-length), buffer, length);
      }
    }
  }
  
  return true;
}

bool FilteredMemoryCache::Dump()
{
  if(!DataAvailable())
    return false;

  off_t memread;
  for(memread = 0; memread < TotalBytes(); memread += FLUSH_POINT)
  {
    size_t memsize = (((TotalBytes()-memread) < FLUSH_POINT) ? (TotalBytes()-memread) : FLUSH_POINT);
    if(!m_write(((char*)m_mem)+(unsigned int)memread, memsize))
      return false;
  }
  
  return true;
}

