#!/usr/bin/python

# coeff.py - produces a calibration map from calibration pressure data

## Copyright (c) 2005 Grant Ingram, this program is distributed under
## the terms of the GNU General Public License

##     This program is free software; you can redistribute it and/or
##     modify it under the terms of the GNU General Public License as
##     published by the Free Software Foundation; either version 2 of
##     the License, or (at your option) any later version.

##     This program is distributed in the hope that it will be useful,
##     but WITHOUT ANY WARRANTY; without even the implied warranty of
##     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##     GNU General Public License for more details.

##     You should have received a copy of the GNU General Public
##     License along with this program; if not, write to the Free
##     Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
##     MA 02111-1307 USA

from optparse import OptionParser
import sys

version_no = 1.0
CENT_MAX_RANGE = -195

def process_args(opt_list):
  
    usage = "usage: %prog [options] arg"
    parser = OptionParser(usage,version="%prog "+str(version_no))
    
    parser.add_option("-d", "--data", type="string", dest="filename",
                      default="calib",
                      help="sets filename root to FILENAME")
    parser.add_option("-v", "--verbose",
                      action="store_true", dest="verbose",
                      help="outputs lots of information")
    parser.add_option("-q", "--quiet",
                      action="store_false", dest="verbose",
                      help="operates silently (default)")
    parser.add_option("-l", "--license",
                      action="store_true", dest="license",
                      help="prints out copyright information")
    (options, args) = parser.parse_args(opt_list)

    if options.verbose:
        print "coeff "+str(version_no)+ """ Copyright (C) 2005 Grant Ingram.
        This program comes with ABSOLUTELY NO WARRANTY; for details
        use option --license.  This is free software, and you are
        welcome to redistribute it under certain conditions.
        """
        print "Reading data from %s.pa" % options.filename
        print "Writing data to %s.cal" % options.filename
     
    if options.license:
      print "coeff "+str(version_no)
      print """
      Copyright (C) 2005 Grant Ingram

      This program is free software; you can redistribute it and/or
      modify it under the terms of the GNU General Public License
      as published by the Free Software Foundation; either version 2
      of the License, or (at your option) any later version.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program; if not, write to the Free Software
      Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
      """
    return options

def open_files(opt_list):
    global fin,fout
    fin = open(opt_list.filename+".pa",'r')
    fout = open(opt_list.filename+".cal",'w')

def close_files():
    global fin,fout
    fin.close()
    fout.close()

def read_data(opt_list):
    global fin,fout
    lines = fin.readlines()

    # find extent of comment lines in input file
    for i in xrange(len(lines)): 
        if lines[i].find("Pitch") != -1 and lines[i].find("Yaw") != -1:
            # There are i-2 comment lines to print out
            break

    # calc size of calibration map,only the probe_code actually matters
    angles=[float(s) for s in (lines[i-2].split())]
    probe_code = angles[-1]
    n_pitch_points = 1 + ( angles[1]-angles[0]) / (angles[2] )
    n_yaw_points = 1 + ( angles[4]-angles[3]) / (angles[5] )
    n_points = n_pitch_points * n_yaw_points

    # calc dynamic correction for the p-s probe not being in jet centre
    corrects=[float(s) for s in (lines[i-1].split())]
    dyn_corr = corrects[0]/corrects[1]
    if dyn_corr > 1.05 or dyn_corr < 0.95:
        print "WARNING: correction > 5%%. Value: %f" % dyn_corr

    # writes out output lines
    for j in xrange(i-2):
        fout.write(lines[j])
    output = "%1.0f %1.0f %1.0f %1.0f 0 Processed by coeff.py V%1.1f\n" \
             % (n_pitch_points,n_yaw_points,n_points,probe_code,version_no)
    fout.write(output)
    output = "Pitch\tYaw\tCpitch\tCyaw\tCtotal\tCstatic"
    fout.write(output)

    # prints out verbose information
    if opt_list.verbose == True:
        print "Probe Code:  " + str(probe_code)
        print "Dynamic Correction: " + str(dyn_corr)
        print "n_pitch_points: " + str(n_pitch_points)
        print "n_yaw_points: " + str(n_yaw_points)
        print "n_points: " + str(n_points)

    #slice data to strip comments lines
    data = lines[i+1:]
        
    return (probe_code,dyn_corr,data)

def calc_coeffs(options,probe_code,dyn_corr,data):

    # convert string data to floating point and give it a meaningful name
    # data is in order  pdyn, pcent, pleft, pright, pcent2, ptop, pbottom
    fdata = []; pdyn =[]; pcent = []; pleft = []; pright = []; ptop = []; pbot =[];
    pitch = []; yaw = [];
    for i in xrange(len(data)):
        fdata = [float(s) for s in (data[i].split())]
        pitch.append(fdata[0])
        yaw.append(fdata[1])
        pdyn.append(dyn_corr * fdata[2]) # correct pdyn
        pcent.append(fdata[3])
        pleft.append(fdata[4])
        pright.append(fdata[5])
        if pcent[-1] < CENT_MAX_RANGE: # use best ranged transducer
            pcent[-1] = fdata[6]
        ptop.append(fdata[7])
        pbot.append(fdata[8])

    for i in xrange(len(pdyn)):

        if (probe_code == 3): # three hole probe coeffs
            paverage = 0.5 * (pleft[i] + pright[i]);
            cyaw = (pleft[i] - pright[i]) / (pcent[i] - paverage);
            ctotal = (pcent[i]) / (pcent[i] - paverage);
            cstatic = (paverage - (-1 * pdyn[i])) / (pcent[i] - paverage);
            output = "\n%f \t%f \t%f \t%f" % (yaw[i], cyaw, ctotal, cstatic)
            fout.write(output)

        if (probe_code == 5): # five hole probe coeffs
            paverage = 0.25 * (pbot[i] + ptop[i] + pleft[i] + pright[i]);
            cpitch = (pbot[i] - ptop[i]) / (pcent[i] - paverage);
            cyaw = (pleft[i] - pright[i]) / (pcent[i] - paverage);
            ctotal = (pcent[i]) / (pcent[i] - paverage);
            cstatic = (paverage - (-1 * pdyn[i])) / (pcent[i] - paverage);
            output = "\n%f \t%f \t%f \t%f \t%f \t%f"\
                     % (pitch[i], yaw[i], cpitch, cyaw, ctotal, cstatic)
            fout.write(output)

def main():
  options = process_args(sys.argv[1:])
  open_files(options)
  (probe_code,dyn_corr,data) = read_data(options)
  calc_coeffs(options,probe_code,dyn_corr,data)
  close_files()


if __name__ == "__main__":
    main()
