#!/usr/bin/python

# plotcoeff.py - plots calibration maps out
# This uses biggles and the GNU plotting utilities


## Copyright (c) 2005 Grant Ingram, this program is distributed under
## the terms of the GNU General Public License

##     This program is free software; you can redistribute it and/or
##     modify it under the terms of the GNU General Public License as
##     published by the Free Software Foundation; either version 2 of
##     the License, or (at your option) any later version.

##     This program is distributed in the hope that it will be useful,
##     but WITHOUT ANY WARRANTY; without even the implied warranty of
##     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##     GNU General Public License for more details.

##     You should have received a copy of the GNU General Public
##     License along with this program; if not, write to the Free
##     Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
##     MA 02111-1307 USA


from optparse import OptionParser
import sys
import biggles
import string
import Numeric

version_no = 1.0

def process_args(opt_list):
  
    usage = "usage: %prog [opt] arg"
    parser = OptionParser(usage,version="%prog "+str(version_no))
    
    parser.add_option("-d", "--data", type="string", dest="filename",
                      default="press",
                      help="sets filename root to FILENAME")
    parser.add_option("-v", "--verbose",
                      action="store_true", dest="verbose",
                      help="outputs lots of information")
    parser.add_option("-q", "--quiet",
                      action="store_false", dest="verbose",
                      help="operates silently (default)")
    parser.add_option("-l", "--license",
                      action="store_true", dest="license",
                      help="prints out copyright information")
    parser.add_option("-m", "--maxdegree",
                      type="float", dest="maxd", default=25,
                      help="set max and min angles for plotting")
    (opt, args) = parser.parse_args(opt_list)

    if opt.verbose:
        print "plotcoeff "+str(version_no)+ """ Copyright (C) 2005 Grant Ingram.
        This program comes with ABSOLUTELY NO WARRANTY; for details
        use option --license.  This is free software, and you are
        welcome to redistribute it under certain conditions.
        """
        
        print "Reading data from %s.cal" % opt.filename
             
    if opt.license:
      print "plotcoeff "+str(version_no)
      print """
      Copyright (C) 2005 Grant Ingram

      This program is free software; you can redistribute it and/or
      modify it under the terms of the GNU General Public License
      as published by the Free Software Foundation; either version 2
      of the License, or (at your option) any later version.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program; if not, write to the Free Software
      Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
      """
    return opt

def open_files(opt_list):
    global fin
    fin = open(opt_list.filename+".cal",'r')
    
def close_files():
    global fin
    fin.close()
    

def read_data(opt_list):
    global fin,n_pitch,n_yaw
    lines = fin.readlines()

    # find extent of comment lines in input file
    for i in xrange(len(lines)): 
        if lines[i].find("Pitch") != -1 and lines[i].find("Yaw") != -1:
            break

    sizes = lines[i-1].split()
    n_pitch = int(sizes[0])
    n_yaw = int(sizes[1])
    
    #slice data to strip comments lines
    data = lines[i+1:]
    
    # prints out verbose information
    if opt_list.verbose == True:
        print "Number of points: " + str(len(data))
                
    return (data)

def plot_data(opt,data):
    global n_yaw, n_pitch

    # convert string data to floating point
    # data is in order:  Pitch, Yaw, Cpitch, Cyaw, Ctotal, Cstatic
    fdata = [];
        
    for i in xrange(len(data)):
        fdata.append([float(s) for s in (data[i].split())])

    plot_box(fdata,opt)
    plot_totstat(fdata,opt,"total")
    plot_totstat(fdata,opt,"static")

    
def plot_box(fdata,opt):  #do box plot of Cyaw and Cpitch
    global n_pitch,n_yaw    
    
    n_boxes = (n_pitch - 1) * (n_yaw - 1)
    
    g = biggles.FramedPlot()
    for i in range(n_boxes):
        a = i + n_yaw
        b = i + 1 + n_yaw
        c = i + 1
        d = i
        
        x = (fdata[a][2],fdata[b][2],fdata[c][2],fdata[d][2],fdata[a][2])
        y = (fdata[a][3],fdata[b][3],fdata[c][3],fdata[d][3],fdata[a][3])

        pitch = (fdata[a][0],fdata[b][0],fdata[c][0],fdata[d][0],fdata[a][0])
        yaw = (fdata[a][1],fdata[b][1],fdata[c][1],fdata[d][1],fdata[a][1])

        line = biggles.Curve(x, y, color = "red")

        x_ok = True; y_ok = True;

        for (j,k) in zip(pitch,yaw):
            if abs(j) >= opt.maxd:
                x_ok = False
            if abs(k) >= opt.maxd:
                y_ok = False

        if x_ok == True and y_ok == True:
            g.add(line)
            
    g.xlabel = "$C_{pitch}$ / [1]"
    g.ylabel = "$C_{yaw}$ / [1]"
    g.title = "$C_{pitch}$, $C_{yaw}$ plot for: %s $\pm$%2.0f$\degree$ limit" \
              % (opt.filename,opt.maxd)
    if opt.verbose == True: g.show()
    g.write_eps(  ("%sbox.eps" % opt.filename) )


def plot_totstat(fdata,opt,totstat):
    if totstat == "total":
        yindex = 4
    elif totstat == "static":
        yindex = 5
        
    g = biggles.FramedPlot();
    red = 0xff #set to red for +ve pitch
    blue = 0xff #set to blue for -ve pitch
        
    for i in range(n_pitch):
        x =[]; y=[];
        for j in range(n_yaw):
            if abs(fdata[i*n_yaw+j][1]) < opt.maxd  \
                   and abs(fdata[i*n_yaw+j][0]) < opt.maxd:
                x.append(fdata[i*n_yaw+j][1])
                y.append(fdata[i*n_yaw+j][yindex])
        if (len(x)>0):
            # create hex strings to vary the colour as the angle
            # changes
            if fdata[i*n_yaw+j][0] > 0:
                clr = str(hex(red))+str("00")+str("00")
                red = red - 16
                clr = string.atoi(clr,16)
            else:
                clrstring =  hex(blue).strip("0x")
                clr = "0x"+"00"+"00"+clrstring
                blue = blue - 16
                clr = string.atoi(clr,16)
            g.add( biggles.Curve(x, y, color = clr ))
            
    g.xlabel = "yaw / [$\degree$]"
    g.ylabel = "$C_{%s}$ / [1]" % totstat
    g.title = "$C_{%s}$ plot for: %s $\pm$%2.0f$\degree$limit"\
                  % (totstat,opt.filename,opt.maxd)
    if opt.verbose == True:
        g.show()
    g.write_eps( ("%s%s.eps" % (opt.filename,totstat)) )



    
def main():
  opt = process_args(sys.argv[1:])
  open_files(opt)
  (data) = read_data(opt)
  plot_data(opt,data)
  close_files()


if __name__ == "__main__":
    main()

