#!/usr/bin/python

# flow.py - calculates flow variables from calibration data and a
# measurement file

## Copyright (c) 2004,2005 Grant Ingram, this program is distributed
## under the terms of the GNU General Public License

##     This program is free software; you can redistribute it and/or
##     modify it under the terms of the GNU General Public License as
##     published by the Free Software Foundation; either version 2 of
##     the License, or (at your option) any later version.

##     This program is distributed in the hope that it will be useful,
##     but WITHOUT ANY WARRANTY; without even the implied warranty of
##     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##     GNU General Public License for more details.

##     You should have received a copy of the GNU General Public
##     License along with this program; if not, write to the Free
##     Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
##     MA 02111-1307 USA

from optparse import OptionParser
import sys
from math import sqrt

version_no = 1.00

#calibration map variables
pitch = []; yaw = [];
cpitch = []; cyaw = [];
ctotal = [];  cstatic = [];

# experimental variables
axial = []; tang = []; radial = []; 
turret = []; upstream = []; centre = []; 
left = []; right = []; cent2 = []; 
top = []; bot = [];

# output from the program
# the prefix m stands for measured
m_pitch = []; m_yaw = [];
m_ptot = []; m_pstat = [];
        
def process_args(opt_list):
    usage = "usage: %prog [options] arg"
    parser = OptionParser(usage,version="%prog "+str(version_no))
    
    parser.add_option("-d", "--data", type="string", dest="dataname",
                      default="press",
                      help="sets measurement filename root to dataname")
    parser.add_option("-c", "--calib", type="string", dest="calibname",
                      default="calib",
                      help="sets calibration filename root to calibname")
    parser.add_option("-o", "--output", type="string", dest="outname",
                      default="flow",
                      help="sets output filename root to outname")
    parser.add_option("-v", "--verbose",
                      action="store_true", dest="verbose",
                      help="outputs lots of information")
    parser.add_option("-q", "--quiet",
                      action="store_false", dest="verbose",
                      help="operates silently (default)")
    parser.add_option("-l", "--license",
                      action="store_true", dest="license",
                      help="prints out copyright information")
    (options, args) = parser.parse_args(opt_list)

    if options.verbose:
        print "flow.py Version"+str(version_no)
        print """Copyright (C) 2004,2005 Grant Ingram.
        This program comes with ABSOLUTELY NO WARRANTY; for details
        use option --license.  This is free software, and you are
        welcome to redistribute it under certain conditions.
        """
             
    if options.license:
      print """
      Copyright (C) 2004,2005 Grant Ingram

      This program is free software; you can redistribute it and/or
      modify it under the terms of the GNU General Public License
      as published by the Free Software Foundation; either version 2
      of the License, or (at your option) any later version.

      This program is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      GNU General Public License for more details.

      You should have received a copy of the GNU General Public License
      along with this program; if not, write to the Free Software
      Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
      """
    return options

def open_files(opt_list):
    global fin,fout,fcalib
    fin = open(opt_list.dataname+".pa",'r')
    fcalib = open(opt_list.calibname+".cal",'r')
    fout = open(opt_list.outname+".t5c",'w')

def close_files():
    global fin,fout,fcalib
    fin.close()
    fcalib.close()
    fout.close()

def read_calib(opt_list):
    global fcalib, n_pitch_pts, n_yaw_pts, probe_code
    lines = fcalib.readlines()

    # find extent of comment lines in input file
    # file format is explained in the manual
    for i in xrange(len(lines)): 
        if lines[i].find("Pitch") != -1 and lines[i].find("Yaw") != -1:
            break

    # read in size of calibration map
    calib_ranges =(lines[i-1].split())
    probe_code = float(calib_ranges[3])
    n_pitch_pts = int(calib_ranges[0])
    n_yaw_pts = int(calib_ranges[1])
    n_pts = int(calib_ranges[2])

    # save output lines
    calib_comment = lines[0]

    #slice data to strip comment lines
    data = lines[i+1:]

    #append to calibration map variables
    for i in range(len(data)):
        fdata = [float(s) for s in (data[i].split())]
        pitch.append(fdata[0])
        yaw.append(fdata[1])
        cpitch.append(fdata[2])
        cyaw.append(fdata[3])
        ctotal.append(fdata[4])
        cstatic.append(fdata[5])

    # prints out verbose information
    if opt_list.verbose == True:
        print "Calibration:  " + calib_comment
        print "Probe Code:  " + str(probe_code)
        print "n_pitch_pts: " + str(n_pitch_pts)
        print "n_yaw_pts: " + str(n_yaw_pts)
        print "n_pts: " + str(n_pts)
        
    return (calib_comment)
    
def read_data(opt_list):
    global fin
    lines = fin.readlines()

    # find extent of comment lines in input file
    for i in xrange(len(lines)): 
        if lines[i].find("axial") != -1 and lines[i].find("tangential") != -1:
            break

    # save output line and probe code, and expt "dimensions"
    data_comment = []
    data_comment.append(lines[0])
    data_comment.append(lines[i-1])

    # prints out verbose information
    if opt_list.verbose == True:
        print "Experiment:  " + data_comment[0]
        print "'Dimensions':  " + data_comment[1]

    #slice data to strip comments lines
    data = lines[i+1:]

    #append to experiment variables
    for i in range(len(data)):
        fdata = [float(s) for s in (data[i].split())]
        # Data is ordered as follows: axial, tangential, radial,
        # theta, upstream, centre, left, right, cent2, top, bottom        
        axial.append(fdata[0])
        tang.append(fdata[1])
        radial.append(fdata[2])
        turret.append(fdata[3])
        upstream.append(fdata[4])
        centre.append(fdata[5])
        left.append(fdata[6])
        right.append(fdata[7])
        cent2.append(fdata[8])
        top.append(fdata[9])
        bot.append(fdata[10])
        
    return (data_comment)

def calc_flowvariables(opts):
    # step through measured values
    for i in range(len(axial)):
        if opts.verbose == True:
            print "On point: ",i
            
        # Calculate calibration coefficients 
        pavg = 0.25 * (left[i] + right[i] + top[i] + bot[i])
        m_cyaw = (left[i] - right[i]) / (centre[i] - pavg)
        m_cpitch = (bot[i] - top[i]) / (centre[i] - pavg)
        m_pitch_tmp,m_yaw_tmp = get_pitch_yaw(m_cpitch,m_cyaw)
        m_ctotal,m_cstatic = get_total_static(m_pitch_tmp,m_yaw_tmp)

        # this part depends on your probe connections, in our case we
        # connect relative to upstream dynamic head
        ptot_tmp = centre[i] -m_ctotal*(centre[i]-pavg)
        pstat_tmp = pavg - m_cstatic*(centre[i]-pavg)

        m_pitch.append(m_pitch_tmp)
        m_yaw.append(m_yaw_tmp)
        m_ptot.append(ptot_tmp)
        m_pstat.append(pstat_tmp)

        if opts.verbose == True:
            print "Pitch: %5.2f deg, Yaw: %5.2f deg, Ptot: %5.2f Pa, Pstat: %5.2f Pa"\
                  % (m_pitch_tmp,m_yaw_tmp,ptot_tmp,pstat_tmp)


def get_pitch_yaw(m_cpitch,m_cyaw):
          
    n_boxes = (n_pitch_pts - 1) * (n_yaw_pts - 1)

    for i in range(n_boxes):
        # for each box in the calibration map get list of x and y points
        a = i + n_yaw_pts
        b = i + 1 + n_yaw_pts
        c = i + 1
        d = i

        # check we are not at the end of a row
        # if we are at the end of the row move to next i
        if ((i+1)%n_yaw_pts==0):
            continue
        
        verticies = (((cpitch[a],cyaw[a]),(cpitch[b],cyaw[b])),
                     ((cpitch[b],cyaw[b]),(cpitch[c],cyaw[c])),
                     ((cpitch[c],cyaw[c]),(cpitch[d],cyaw[d])),
                     ((cpitch[d],cyaw[d]),(cpitch[a],cyaw[a])))
        

        if boxtest(verticies,m_cpitch,m_cyaw)== True:
            m_pitch_tmp = semibilinear(cpitch[a],cyaw[a],cpitch[b],cyaw[b]
                                        ,cpitch[c],cyaw[c],cpitch[d],cyaw[d]
                                        ,pitch[a],pitch[d]
                                        ,m_cpitch,m_cyaw)
            
            m_yaw_tmp = semibilinear(cyaw[a],cpitch[a],cyaw[d],cpitch[d]
                                        ,cyaw[c],cpitch[c],cyaw[b],cpitch[b]
                                        ,yaw[d],yaw[c]
                                        ,m_cyaw,m_cpitch)

            break
        
    else:
        # point is not in the calibration map
        m_pitch_tmp= -999.0
        m_yaw_tmp = -999.0
        if opts.verbose == True:
            print "Error: point "+str(i)+"not found in calibration map"
            print "Measured Cpitch:",m_cpitch
            print "Measured Cyaw:",m_cyaw

    return (m_pitch_tmp,m_yaw_tmp)
            
        
def boxtest(verticies,mx,my):
    # we have a polygon described by verticies, the point of this
    # routine is to find out if mx and my are inside the box
    # The solution to this is from the comp.graphics FAQ

    # draw a ray off into infinity
    ray = ((mx,my),(1e99,my))
    cross = 0
    
    for i in verticies:
        ret_value = intersect(i,ray)
        #intersect returns three values, only need the 1st here
        if ret_value[0] == True:
            cross = cross +1
           
    if cross % 2 == 0: #even not in box
        return False
    else:              #odd inside the box
        return True    
        
def intersect(AB,CD):
    #we use the notation from Question 1.03 of the comp.graphics FAQ
    Ax = AB[0][0]; Ay=AB[0][1]; Bx = AB[1][0]; By=AB[1][1];
    Cx = CD[0][0]; Cy=CD[0][1]; Dx = CD[1][0]; Dy=CD[1][1];
    
    r = ((Ay-Cy)*(Dx-Cx)-(Ax-Cx)*(Dy-Cy)) / ( (Bx-Ax)*(Dy-Cy)-(By-Ay)*(Dx-Cx))
    s = ((Ay-Cy)*(Bx-Ax)-(Ax-Cx)*(By-Ay)) / ( (Bx-Ax)*(Dy-Cy)-(By-Ay)*(Dx-Cx))

    if r >= 0 and r <= 1 and 0 <= s and s <= 1:
        # point intersects return value of point
        Px = Ax +r*(Bx-Ax)
        Py = Ay +r*(By-Ay)
        return (True,Px,Py)
    else:
        return (False,None,None)
    
def semibilinear(x0,y0,x1,y1,x2,y2,x3,y3,gamma_i,gamma_j,xm,ym):
    # does a sort of bilinear interpolation, where a) the variable
    # only changes in one direction and b) isn't on a regular grid
    xi = x0 + ((ym-y0)/(y1-y0))*(x1-x0)
    xj = x3 + ((ym-y3)/(y2-y3))*(x2-x3)
    gamma_m = gamma_i + ((xm-xi)/(xj-xi))*(gamma_j-gamma_i)
    return gamma_m

def write_output(opts,calib_comments,data_comments):
    #writes the output to a t5c file, the format is designed to be
    #compatable with other work at Durham University
    fout.write(data_comments[0])
    fout.write(calib_comments)
    output = "Created using flow.py Version "+str(version_no)+"\n"
    fout.write(output)
    fout.write(data_comments[1])
    fout.write("axial\ttang\tradial\tturret\tpitch\tyaw\tptot\tpstat")
    for i in range(len(axial)):
        output = "\n%f \t%f \t%f \t%f \t%f \t%f \t%f \t%f" % \
                 (axial[i], tang[i], radial[i], turret[i], m_pitch[i], m_yaw[i],
                  m_ptot[i], m_pstat[i])
        fout.write(output)


def get_total_static(m_pitch_tmp,m_yaw_tmp):
    # trap error condition where no values are found in calibration map
    if m_pitch_tmp == -999:
        m_ctotal = 215; m_cstatic = 0;
        return
        
    #a,b,c,d = 0,1,2,3 but you can't use numbers as variable names!
    (a,b,c,d) = find_indicies(m_pitch_tmp,m_yaw_tmp)

    m_ctotal = bilinear(pitch[a],yaw[a],pitch[b],yaw[b],
                         pitch[c],yaw[c],pitch[d],yaw[d],
                         ctotal[a],ctotal[b],ctotal[c],ctotal[d]
                         ,m_pitch_tmp,m_yaw_tmp)
    m_cstatic = bilinear(pitch[a],yaw[a],pitch[b],yaw[b],
                         pitch[c],yaw[c],pitch[d],yaw[d],
                         cstatic[a],cstatic[b],cstatic[c],cstatic[d]
                         ,m_pitch_tmp,m_yaw_tmp)
    
    return (m_ctotal,m_cstatic)


def find_indicies(m_pitch_tmp,m_yaw_tmp):

    #construct array of pitch angles to search through
    pitchvalues = []
    for i in range(n_pitch_pts):
        pitchvalues.append(pitch[i*n_yaw_pts])
    
    #find bounding indicies within pitch angles and yaw angles
    #this makes assumptions about the arrangement of data in the calibration map
    for i in range(n_yaw_pts):
        if pitchvalues[i] < m_pitch_tmp < pitchvalues[i+1]:
            pitch_lwr = i
            pitch_upr = i+1
            break

    for i in range(n_yaw_pts):
        if yaw[i] < m_yaw_tmp < yaw[i+1]:
            yaw_lwr = i
            yaw_upr = i+1
            break

    #construct indicies from the info gathered
    a = pitch_lwr*n_yaw_pts+yaw_lwr
    b = pitch_lwr*n_yaw_pts+yaw_upr
    c = pitch_upr*n_yaw_pts+yaw_upr
    d = pitch_upr*n_yaw_pts+yaw_lwr
    return (a,b,c,d)

def bilinear(x0,y0,x1,y1,x2,y2,x3,y3,gamma_0,gamma_1,gamma_2,gamma_3,xm,ym):
    xi = x1; xj=x2;
    gamma_i = gamma_0 + ((ym-y0)/(y1-y0))*(gamma_1-gamma_0)
    gamma_j = gamma_3 + ((ym-y3)/(y2-y3))*(gamma_2-gamma_3)
    gamma_m = gamma_i + ((xm-xi)/(xj-xi))*(gamma_j-gamma_i)
    return gamma_m
    
def main():
  options = process_args(sys.argv[1:])
  open_files(options)
  calib_comments = read_calib(options)
  data_comments = read_data(options)
  calc_flowvariables(options)
  write_output(options,calib_comments,data_comments)
  close_files()
  

if __name__ == "__main__":
    main()
