/* 
   Copyright (C) 2009, 2010, 2011, 2012, 2013 German A. Arias <german@xelalug.org>

   This file is part of FísicaLab application

   FísicaLab is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 3 of the License, or (at your option) any later version.
 
   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/

#import <math.h>
#import "ChalkboardView.h"
#import "FLKinematics.h"
#import "FLKinematicsCircularMotion.h"
#import "FLStatics.h"
#import "FLStaticsRigidBodies.h"
#import "FLDynamics.h"
#import "FLCalorimetry.h"
#import "FLDynamicsCircularMotion.h"
#import "ObjectData.h"

@interface ObjectData ( Mathematics )
-(NSString *)processEntry: (NSString *)anEntry;
-(NSString *)processAngleEntry: (NSString *)anEntry;
-(NSString *)operatePolynomial: (NSString *)anEntry;
-(NSString *)operateTerm: (NSString *)anEntry;
-(NSString *)operateFunction: (NSString *)anEntry;
-(BOOL)isNumericArray: (NSArray *)anArray;
-(BOOL)isNumeric: (NSString *)anEntry;
@end

@implementation ObjectData ( Mathematics )
-(NSString *)processEntry: (NSString *)anEntry
{
  NSString *result;
  NSArray *data;

  data = [anEntry componentsSeparatedByCharactersInSet: plus_minusSet];
  
  if ([data count] > 1)
    {
      result = [self operatePolynomial: anEntry];
    }
  else
    {
      data = [anEntry componentsSeparatedByCharactersInSet:
			multiplication_divisionSet];
      
      if ([data count] > 1)
	{
	  result = [self operateTerm: anEntry];
	}
      else if ([anEntry hasSuffix: @")"])
	{
	  result = [self operateFunction: anEntry];
	}
      else
	{
	  return anEntry;
	}
    }

  if ([self isNumeric: result])
    {
      result = [NSString stringWithFormat: @"%.3f", [result doubleValue]];
    }

  return result;
}

-(NSString *)processAngleEntry: (NSString *)anEntry
{
  int j = -1;
  NSArray *array;
  NSString *data = [NSString stringWithString: anEntry];

  if ([data hasPrefix: @"acos("] &&
      [data hasSuffix: @")"])
    {
      data = [data stringByDeletingPrefix: @"acos("];
      data = [data stringByDeletingSuffix: @")"];

      j = 0;
    }
  else if ([data hasPrefix: @"asin("] &&
	   [data hasSuffix: @")"])
    {
      data = [data stringByDeletingPrefix: @"asin("];
      data = [data stringByDeletingSuffix: @")"];

      j = 1;
    }
  else
    {
      j = 2;
    }

  array = [data componentsSeparatedByString: @"/"];
      
  if ( ([array count] == 2) &&
       [self isNumericArray: array] &&
       ([[array objectAtIndex: 1] doubleValue] != 0) )
    {
      double angle;

      switch (j)
	{
	case 0:
	  {
	    angle = 180*acos([[array objectAtIndex: 0] doubleValue]/
			     [[array objectAtIndex: 1] doubleValue])/M_PI;
	  }
	  break;
	case 1:
	  {
	    angle = 180*asin([[array objectAtIndex: 0] doubleValue]/
			     [[array objectAtIndex: 1] doubleValue])/M_PI;
	  }
	  break;
	case 2:
	  {
	    angle = 180*atan2([[array objectAtIndex: 0] doubleValue],
			      [[array objectAtIndex: 1] doubleValue])/M_PI;
	  }
	  break;
	}
  
      return [NSString stringWithFormat: @"%.3f", angle];
    }
  else
    {
      return anEntry;
    }
}

-(NSString *)operatePolynomial: (NSString *)anEntry
{
  int x;
  double sum = 0;
  BOOL numeric = YES;
  NSString *result;
  NSMutableString *data = [NSMutableString stringWithString: anEntry];
  NSMutableString *operations = [NSMutableString string];
  NSMutableArray *terms = [NSMutableArray arrayWithArray:
 		            [anEntry componentsSeparatedByCharactersInSet: plus_minusSet]];
  NSArray *factors;

  /*
   * Check that the final factor is not an empty string.
   * If is, this mean that the final character is an
   * operator. And this is not allowed.
   */
  if ([[terms lastObject] length] == 0)
    {
      numeric = NO;
    }

  // Get a string with all the operators in polynomial.
  if (numeric)
    {
      for (x = 0; x < [terms count]; x++)
	{
	  if ([data hasPrefix: @"+"] || [data hasPrefix: @"-"])
	    {
	      [operations appendString: [data substringToIndex: 1]];
	      [data deleteCharactersInRange: NSMakeRange (0,1)];
	    }

	  if ([data hasPrefix: [terms objectAtIndex: x]])
	    {
	      [data deletePrefix: [terms objectAtIndex: x]];
	    }
	}
    }

  // Make the operations in all terms.
  if (numeric)
    {
      if ([[terms objectAtIndex: 0] length] == 0)
	{
	  [terms replaceObjectAtIndex: 0 withObject: @"0"];
	}

      for (x = 0; x < [terms count]; x++)
	{
	  factors = [[terms objectAtIndex: x]
		      componentsSeparatedByCharactersInSet:
			multiplication_divisionSet];

	  if ([factors count] > 1)
	    {
	      result = [self operateTerm: [terms objectAtIndex: x]];

	      if ([self isNumeric: result])
		{
		  [terms replaceObjectAtIndex: x
				   withObject: result];
		}
	      else
		{
		  numeric = NO;
		  break;
		}
	    }
	  else if ([[terms objectAtIndex: x] hasSuffix: @")"])
	    {
	      result = [self operateFunction: [terms objectAtIndex: x]];

	      if ([self isNumeric: result])
		{
		  [terms replaceObjectAtIndex: x
				   withObject: result];
		}
	      else
		{
		  numeric = NO;
		  break;
		}
	    }
	  else if (![self isNumeric: [terms objectAtIndex: x]])
	    {
	      numeric = NO;
	      break;
	    }
	}
    }

  // Sum the terms.
  if (numeric)
    {
      if ([terms count] > [operations length])
	{
	  sum += [[terms objectAtIndex: 0] doubleValue];
	  [terms removeObjectAtIndex: 0];
	}

      for (x = 0; x < [terms count]; x++)
	{
	  if ([operations hasPrefix: @"+"])
	    {
	      [operations deletePrefix: @"+"];

	      sum += [[terms objectAtIndex: x] doubleValue];
	    }
	  else if ([operations hasPrefix: @"-"])
	    {
	      [operations deletePrefix: @"-"];

	      sum -= [[terms objectAtIndex: x] doubleValue];
	    }
	}
    }

  // Return the result if is numeric. Otherwise
  // return the original string.
  if (numeric)
    {
      return [NSString stringWithFormat: @"%f", sum];
    }
  else
    {
      return anEntry;
    }
}

-(NSString *)operateTerm: (NSString *)anEntry
{
  int x;
  double result = 0;
  BOOL numeric = YES;
  NSString *factor;
  NSMutableString *data = [NSMutableString stringWithString: anEntry];
  NSMutableString *operations = [NSMutableString string];
  NSMutableArray *factors = [NSMutableArray arrayWithArray:
			      [anEntry componentsSeparatedByCharactersInSet:
			      multiplication_divisionSet]];

  /*
   * Check that the first and final factors are not an empty string.
   * If one of these is empty, this mean that the first/final character
   * is an operator. And this is not allowed.
   */
  if ( ([[factors objectAtIndex: 0] length] == 0) ||
       ([[factors lastObject] length] == 0) )
    {
      numeric = NO;
    }

  // Get a string with all the operators in term.
  if (numeric)
    {
      for (x = 0; x < [factors count]; x++)
	{
	  if ([data hasPrefix: @"*"] || [data hasPrefix: @"/"])
	    {
	      [operations appendString: [data substringToIndex: 1]];
	      [data deleteCharactersInRange: NSMakeRange (0,1)];
	    }

	  if ([data hasPrefix: [factors objectAtIndex: x]])
	    {
	      [data deletePrefix: [factors objectAtIndex: x]];
	    }
	}
    }

  // Operate the funtions and check that factors/results are numerics.
  if (numeric)
    {
      for (x = 0; x < [factors count]; x++)
	{
	  if ([[factors objectAtIndex: x] hasSuffix: @")"])
	    {
	      factor = [self operateFunction: [factors objectAtIndex: x]];

	      if ([self isNumeric: factor])
		{
		  [factors replaceObjectAtIndex: x
				     withObject: factor];
		}
	      else
		{
		  numeric = NO;
		  break;
		}
	    }
	  else if (![self isNumeric: [factors objectAtIndex: x]])
	    {
	      numeric = NO;
	      break;
	    }
	}
    }

  // Make the operations.
  if (numeric)
    {
      result = [[factors objectAtIndex: 0] doubleValue];

      for (x = 1; x < [factors count]; x++)
	{
	  if ([operations hasPrefix: @"*"])
	    {
	      [operations deletePrefix: @"*"];

	      result *= [[factors objectAtIndex: x] doubleValue];
	    }
	  else if ([operations hasPrefix: @"/"])
	    {
	      [operations deletePrefix: @"/"];

	      result /= [[factors objectAtIndex: x] doubleValue];
	    }
	}
    }

  // Return the result if is numeric. Otherwise
  // return the original string.
  if (numeric)
    {
      return [NSString stringWithFormat: @"%f", result];
    }
  else
    {
      return anEntry;
    }
}

-(NSString *)operateFunction: (NSString *)anEntry
{
  int j = -1;
  double number = 0;
  BOOL numeric = YES;
  NSMutableString *data = [NSMutableString stringWithString: anEntry];

  // First check the kind of funtion
  if ([data hasPrefix: @"cos("] &&
      [data hasSuffix: @")"])
    {
      [data deletePrefix: @"cos("];
      [data deleteSuffix: @")"];
      
      j = 0;
    }
  else if ([data hasPrefix: @"sin("] &&
	   [data hasSuffix: @")"])
    {
      [data deletePrefix: @"sin("];
      [data deleteSuffix: @")"];
      
      j = 1;
    }
  else if ([data hasPrefix: @"tan("] &&
	   [data hasSuffix: @")"])
    {
      [data deletePrefix: @"tan("];
      [data deleteSuffix: @")"];
      
      j = 2;
    }
  else if ([data hasPrefix: @"sqrt("] &&
	   [data hasSuffix: @")"])
    {
      [data deletePrefix: @"sqrt("];
      [data deleteSuffix: @")"];

      j = 3;
    }
  else if ([data hasPrefix: @"hypot("] &&
	   [data hasSuffix: @")"])
    {
      [data deletePrefix: @"hypot("];
      [data deleteSuffix: @")"];

      j = 4;
    }
  else if ([data hasPrefix: @"leg("] &&
	   [data hasSuffix: @")"])
    {
      [data deletePrefix: @"leg("];
      [data deleteSuffix: @")"];

      j = 5;
    }
  else if ([data hasPrefix: @"rd("] &&
	   [data hasSuffix: @")"])
    {
      [data deletePrefix: @"rd("];
      [data deleteSuffix: @")"];

      j = 6;
    }

  // Operate if there are numeric arguments
  switch (j)
    {
    case 0 ... 3:
      {
	if ([self isNumeric: data])
	  {
	    if (j == 0)
	      {
		// cosine
		number = cos(M_PI*[data doubleValue]/180);
	      }
	    else if (j == 1)
	      {
		// sine
		number = sin(M_PI*[data doubleValue]/180);
	      }
	    else if (j == 2)
	      {
		// tangent
		number = tan(M_PI*[data doubleValue]/180);
	      }
	    else
	      {
		// square root
		number = sqrt([data doubleValue]);
	      }
	  }
	else
	  {
	    numeric = NO;
	  }
      }
      break;
    case 4:
    case 5:
      {
	NSArray *factors;
	factors = [data componentsSeparatedByString: @","];

	if ([self isNumericArray: factors] &&
	    ([factors count] == 2) )
	  {
	    if (j == 4)
	      {
		// hypotenuse
		number = sqrt([[factors objectAtIndex: 0] doubleValue]*
			      [[factors objectAtIndex: 0] doubleValue] +
			      [[factors objectAtIndex: 1] doubleValue]*
			      [[factors objectAtIndex: 1] doubleValue]);
	      }
	    else
	      {
		// leg
		number = sqrt([[factors objectAtIndex: 0] doubleValue]*
			      [[factors objectAtIndex: 0] doubleValue] -
			      [[factors objectAtIndex: 1] doubleValue]*
			      [[factors objectAtIndex: 1] doubleValue]);
	      }
	  }
	else
	  {
	    numeric = NO;
	  }
      }
      break;
    case 6:
      {
	NSArray *factors;
	factors = [data componentsSeparatedByString: @","];

	if ([self isNumericArray: factors] &&
	    ([factors count] == 3) )
	  {
	    // radius
	    number = [[factors objectAtIndex: 1] doubleValue]*
	      [[factors objectAtIndex: 2] doubleValue]/
	      ( [[factors objectAtIndex: 0] doubleValue] +
		[[factors objectAtIndex: 1] doubleValue] );
	  }
	else
	  {
	    numeric = NO;
	  }
      }
      break;
    }
  
  // Return the result if there are numeric arguments. Otherwise
  // return the original string.
  if (numeric)
    {
      return [NSString stringWithFormat: @"%f", number];
    }
  else
    {
      return anEntry;
    }
}

-(BOOL)isNumericArray: (NSArray *)anArray
{
  BOOL numeric = YES;
  NSString *object;
  NSEnumerator *enumerator = [anArray objectEnumerator];

  while ((object = [enumerator nextObject]))
    { 
      if (![self isNumeric: object])
	{
	  numeric = NO;
	  break;
	}
    }

  return numeric;
}

-(BOOL)isNumeric: (NSString *)anEntry
{
  BOOL numeric = YES;
  NSCharacterSet *set;

  set = [NSCharacterSet characterSetWithCharactersInString: anEntry];
  
  if (![numbers isSupersetOfSet: set] || ([anEntry length] == 0) )
    {
      numeric = NO;
    }

  return numeric;
}
@end

@implementation ObjectData

- (void)awakeFromNib
{
  conteoReferencia = 0;
  sistema = 0;
  [dataViewer setDataSource: self] ;
  [dataViewer setDelegate: self] ;
  listaObjetos = [NSMutableDictionary dictionary] ;
  [listaObjetos retain] ;

  identificador = [[NSNumber alloc] initWithInt: 0];

  //Make the conversion factor dictionary
  NSBundle *mainBundle = [NSBundle mainBundle];
  NSString *path = [mainBundle pathForResource: @"conversionFactors" ofType: @"plist"];
  conversionDictionary = [NSMutableDictionary dictionaryWithContentsOfFile: path];
  [conversionDictionary retain];

  //Make the menu conversion factor dictionary
  path = [mainBundle pathForResource: @"menuConversionFactors" ofType: @"plist"];
  menuConversionDictionary = [NSMutableDictionary dictionaryWithContentsOfFile: path];
  [menuConversionDictionary retain];

  numbers = [NSCharacterSet characterSetWithCharactersInString: @".-+0123456789E"];
  [numbers retain];
  plus_minusSet =
    [NSCharacterSet characterSetWithCharactersInString: @"-+"];
  [plus_minusSet retain];
  multiplication_divisionSet =
    [NSCharacterSet characterSetWithCharactersInString: @"*/"];
  [multiplication_divisionSet retain];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector (setMenuForModule:)
    name: @"moduleDidChangeNotification"
    object: nil];
}

- (void) sisUnidades: (id)sender
{
  if (sistema != [[sender selectedCell] tag])
    {
      sistema = [[sender selectedCell] tag];
      [self setMenuForModule: nil];
    }
}

- (void) calcular: (id)sender
{
  /* insert your code here */
  int cod = [sender codigo: self] ;
  [dataViewer deselectAll: sender] ;
  
  if ([cleanResultsViewer state] == NSOnState)
    {
      [resultsViewer setString: @""] ;
    }
  
  switch(cod)
    {
    case 0:
      {
	id cinemaPuntual = [FLKinematics new];
	[cinemaPuntual setSystem: sistema] ;
	[cinemaPuntual setConversions: [conversionDictionary objectForKey: @"kinematicParticles"]];
	[cinemaPuntual setVisor: resultsViewer];
	[cinemaPuntual crearEcuaciones: listaObjetos] ;
	RELEASE(cinemaPuntual);
      }
      break;
    case 1:
      {
	id estatPuntual = [FLStatics new];
	[estatPuntual setSystem: sistema] ;
	[estatPuntual setConversions: [conversionDictionary objectForKey: @"staticParticles"]];
	[estatPuntual setVisor: resultsViewer];
	[estatPuntual setCells: [sender subviews]] ;
	[estatPuntual withChalkboardWidth: [sender chalkboardWidth] height: [sender chalkboardHeight]];
	[estatPuntual crearEcuaciones: listaObjetos] ;
	RELEASE(estatPuntual);
      }
      break;
    case 2:
      {
	id dinamiPuntual = [FLDynamics new];
	[dinamiPuntual setSystem: sistema] ;
	[dinamiPuntual setConversions: [conversionDictionary objectForKey: @"dynamicParticles"]];
	[dinamiPuntual setVisor: resultsViewer];
	[dinamiPuntual setCells: [sender subviews]] ;
	[dinamiPuntual withChalkboardWidth: [sender chalkboardWidth] height: [sender chalkboardHeight]];
	[dinamiPuntual crearEcuaciones: listaObjetos] ;
	RELEASE(dinamiPuntual);
      }
      break;
    case 3:
      {
	id calorGas = [FLCalorimetry new];
	[calorGas setSystem: sistema] ;
	[calorGas setConversions: [conversionDictionary objectForKey: @"heat"]];
	[calorGas setVisor: resultsViewer];
	[calorGas setCells: [sender subviews]] ;
	[calorGas withChalkboardWidth: [sender chalkboardWidth] height: [sender chalkboardHeight]];
	[calorGas crearEcuaciones: listaObjetos] ;
	RELEASE(calorGas);
      } 
      break;
    case 4:
      {
	id cinemaCircular = [FLKinematicsCircularMotion new];
	[cinemaCircular setSystem: sistema] ;
	[cinemaCircular setConversions: [conversionDictionary objectForKey: @"kinematicCircular"]];
	[cinemaCircular setVisor: resultsViewer];
	[cinemaCircular crearEcuaciones: listaObjetos] ;
	RELEASE(cinemaCircular);
      }
      break;
    case 5:
      {
	id estatRigida = [FLStaticsRigidBodies new];
	[estatRigida setSystem: sistema];
	[estatRigida setConversions: [conversionDictionary objectForKey: @"staticRigid"]];
	[estatRigida setVisor: resultsViewer];
	[estatRigida setCells: [sender subviews]];
	[estatRigida withChalkboardWidth: [sender chalkboardWidth] height: [sender chalkboardHeight]];
	[estatRigida crearEcuaciones: listaObjetos];
	RELEASE(estatRigida);
      }
      break;
    case 6:
      {
	id dynamicsCircular = [FLDynamicsCircularMotion new];
	[dynamicsCircular setSystem: sistema];
	[dynamicsCircular setConversions: [conversionDictionary objectForKey: @"dynamicsCircular"]];
	[dynamicsCircular setVisor: resultsViewer];
	[dynamicsCircular setCells: [sender subviews]];
	[dynamicsCircular withChalkboardWidth: [sender chalkboardWidth] height: [sender chalkboardHeight]];
	[dynamicsCircular makeEquationsForData: listaObjetos];
	RELEASE(dynamicsCircular);
      }
      break;
    }
}


- (void) agregarObjeto: (id)nuevo
{
  /* insert your code here */
  NSString *gravedad ;
  NSArray *listaTitulos = nil;
  NSMutableArray *listaDatos = nil;
  NSMutableArray *listaValores = [NSMutableArray array] ;
  NSMutableDictionary *datos ;
  
  NSNumber *tipo ;
  conteoReferencia = conteoReferencia + 1;
  [dataViewer deselectRow: [dataViewer selectedRow] ] ;
  
  if (sistema == 0)
    {
      gravedad = [NSString stringWithString: @"9.81"] ;
    }
  else
    {
      gravedad = [NSString stringWithString: @"32.2"] ;
    }
  
  switch ([nuevo tag])
    {
      //Objetos del 1-49 pertenecen al submodulo CinematicaPuntual
    case 1:
      {
	//Sistema de referencia
        listaTitulos = [NSArray arrayWithObjects: @"tf", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break ;
    case 2:
      {
	//Sistema movil
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), _(@"Objeto"), @"xsi", @"ysi", @"vsx", @"vsy", @"xof", @"yof", @"vxof", @"vyof", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;      
    case 3:
      {
	//Movil general
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"ax", @"ay", @"xi", @"yi", @"vxi", @"vyi", @"ti", @"xf", @"yf", @"vxf", @"vyf", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 4:
      {
	//Cañon
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"ax", @"ay", @"xi", @"yi", @"vi", @"angi", @"ti", @"xf", @"yf", @"vf", @"angf", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 5:
      {
	//Movil en X
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"ax", @"xi", @"vxi", @"ti", @"xf", @"vxf", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break; 
    case 6:
      {
	//Movil en Y
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"ay", @"yi", @"vyi", @"ti", @"yf", @"vyf", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 7:
      {
	//Sistema movil en X
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), _(@"Objeto"), @"xsi", @"vsx", @"xof", @"vxof", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 8:
      {
	//Sistema movil en Y
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), _(@"Objeto"), @"ysi", @"vsy", @"yof", @"vyof", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 9:
      {
	//Movil con velocidad constante X
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"xi", @"xf", @"ti", @"vx", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 10:
      {
	//Movil con velocidad constante Y 
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"yi", @"yf", @"ti", @"vy", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 11:
      {
	//Distancia
        listaTitulos = [NSArray arrayWithObjects: _(@"Objeto 1"), _(@"Objeto 2"), @"d", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 12:
      {
	//Velocidad relativa
        listaTitulos = [NSArray arrayWithObjects: _(@"Objeto 1"), _(@"Objeto 2"), @"v", @"ang", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 13:
      {
	//Movil radial
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"a", @"angf", @"xi", @"yi", @"vi", @"ti", @"xf", @"yf", @"vf", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 14:
      {
	//Distancia X o Y
        listaTitulos = [NSArray arrayWithObjects: @"x1 (y1)", @"x2 (y2)", @"x1 - x2 (y1 - y2)", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 15:
      {
	//Punto
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"xf", @"yf", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
      //Objetos del 50-99 pertenecen al submodulo estaticaPuntual
    case 50:
      {
	//Sistema de referencia
        listaTitulos = [NSArray arrayWithObjects: @"g", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: gravedad, nil] ;
      }
      break;
    case 51:
      {
	//Bloque
        listaTitulos = [NSArray arrayWithObjects: @"m", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 52:
    case 53:
      {
	//Bloques en un plano inclinado
        listaTitulos = [NSArray arrayWithObjects: @"m", @"ang", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 54:
    case 55:
      {
	//Poleas y puntos
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break; 
    case 56 ... 59:
      {
	//Fuerzas oblicuas
        listaTitulos = [NSArray arrayWithObjects: @"f", @"ang", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 60 ... 63:
      {
	//Fuerzas verticales y horizontales
        listaTitulos = [NSArray arrayWithObjects: @"f", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 64 ... 71:
      {
	//Fricciones
        listaTitulos = [NSArray arrayWithObjects: @"N", @"u", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 72:
      {
	//Resultante general
        listaTitulos = [NSArray arrayWithObjects: @"f", @"ang", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 73:
    case 74:
      {
	//Resultante vertical y horizontal
        listaTitulos = [NSArray arrayWithObjects: @"f", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 75 ... 78:
      {
	//Resortes
        listaTitulos = [NSArray arrayWithObjects: @"k", @"d", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 79:
      {
	//Relation of angles
	listaTitulos = [NSArray arrayWithObjects: @"ang1", @"ang2", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
      //Objetos del 100-149 pertenecen al submodulo dinamicaPuntual
    case 100:
      {
	//Sistema de referencia
        listaTitulos = [NSArray arrayWithObjects: @"g", @"t", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: gravedad, @"0", nil] ;
      }
      break;
    case 101:
      {
	//Movil general
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"m", @"vi", @"angi", @"vf", @"angf", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 102:
      {
	//Movil en X
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"m", @"vxi", @"vxf", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 103:
      {
	//Movil en Y
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"m", @"vyi", @"vyf", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 104:
    case 105:
      {
	//Bloque vertical y horizontal
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"m", @"a", @"vi", @"vf", @"d", _(@"Relativo a"), nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"sf", nil] ;
      }
      break;        
    case 106:
    case 107:
      {
	//Bloques en un plano inclinado
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"m", @"ang", @"a", @"vi", @"vf", @"d", _(@"Relativo a"), nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"sf", nil] ;
      }
      break;
    case 108:
      {
	//Polea
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 109 ... 112:
      {
	//Fuerzas oblicuas
        listaTitulos = [NSArray arrayWithObjects: @"f", @"ang", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 113 ... 116:
      {
	//Fuerzas verticales y horizontales
        listaTitulos = [NSArray arrayWithObjects: @"f", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 117 ... 124:
      {
	//Fricciones
        listaTitulos = [NSArray arrayWithObjects: @"N", @"u", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 125 ... 128:
      {
	//Contactos oblicuos
        listaTitulos = [NSArray arrayWithObjects: @"N", @"u", @"ang", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 129 ... 132:
      {
	//Contactos verticales y horizontales
        listaTitulos = [NSArray arrayWithObjects: @"N", @"u", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break; 
    case 133 ... 136:
      {
	//Resortes
        listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"k", @"xi", @"xf", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil] ;
      }
      break;                       
    case 137:
      {
	//Relacion de aceleraciones
        listaTitulos = [NSArray arrayWithObjects: @"a1", @"a2", @"z", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"-1", nil] ;
      }
      break;
    case 138:
      {
	//Colison
        listaTitulos = [NSArray arrayWithObjects: _(@"Objeto 1"), _(@"Objeto 2"), @"e", @"angn", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 139:
      {
	//Energia
        listaTitulos = [NSArray arrayWithObjects: _(@"Objeto 1"), _(@"Objeto 2"), _(@"Objeto 3"), _(@"Objeto 4"), @"W", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil] ;        
      }
      break;   
    case 140:
      {
	//Impulso
        listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"Imp", @"ang", @"fImp", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 141:
      {
	//Potencia
        listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"P", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 143:
      {
	//Movimiento relativo
        listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"asf", @"ang_asf", @"vfsf", @"ang_vfsf", @"dsf", @"ang_dsf", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;          
      //Objetos del 150-199 pertencen al submodulo calorGas
    case 150:
      {
	//Reloj
	listaTitulos = [NSArray arrayWithObjects: @"t", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 151:
    case 153:
      {
	//Calor aplicado y extraido
	listaTitulos = [NSArray arrayWithObjects: @"Q", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 152:
    case 154:
      {
	//Flujo de calor y refrigeracion
	listaTitulos = [NSArray arrayWithObjects: @"dQ/dt", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 155:
    case 156:
    case 169:
      {
	//Bloque, liquido y gas
	listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"m", @"c", @"Ti", @"Tf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 157:
      {
	//Dilatacion lineal
	listaTitulos = [NSArray arrayWithObjects: @"k", @"Li", @"Lf", @"Ti", @"Tf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 158:
      {
	//Dilatacion superficial
	listaTitulos = [NSArray arrayWithObjects: @"k", @"Si", @"Sf", @"Ti", @"Tf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 159:
      {
	//Dilatacion volumetrica
	listaTitulos = [NSArray arrayWithObjects: @"k", @"Vi", @"Vf", @"Ti", @"Tf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break; 
    case 160:
      {
	//Cambio de fase solido-liquido
	listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"m", @"cf", _(@"Sentido"), nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @">", nil] ;
      }
      break;
    case 161:
      {
	//Cambio de fase liquido-gas
	listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"m", @"cv", _(@"Sentido"), nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @">", nil] ;
      }
      break;
    case 162:
      {
	//Proceso
	listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), _(@"Objeto 1"), _(@"Objeto 2"), _(@"Objeto 3"), _(@"Objeto 4"), _(@"Objeto 5"), nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 163:
      {
	//Calorimetro
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto 1"), _(@"Objeto 2"), _(@"Objeto 3"), _(@"Objeto 4"), nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 164:
      {
	//Gas a presion constante
	listaTitulos = [NSArray arrayWithObjects: @"Vi", @"Ti", @"Vf", @"Tf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 165:
      {
	//Gas a temperatura constante
	listaTitulos = [NSArray arrayWithObjects: @"Pi", @"Vi", @"Pf", @"Vf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 166:
      {
	//Gas a volumen constante
	listaTitulos = [NSArray arrayWithObjects: @"Pi", @"Ti", @"Pf", @"Tf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 167:
      {
	//Gas general
	listaTitulos = [NSArray arrayWithObjects: @"Pi", @"Vi", @"Ti", @"Pf", @"Vf", @"Tf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 168:
      {
	//Intercambiador
	listaTitulos = [NSArray arrayWithObjects: @"TRi", @"TRf", @"dR/dt", @"cR", @"TFi", @"TFf", @"dF/dt", @"cF", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
      //Objetos del 200-249 pertenecen al submodulo cinematicaPuntualCircular
    case 200:
      {
	//Sistema de referencia
	listaTitulos = [NSArray arrayWithObjects: @"tf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 201:
      {
	//Movil circular
	listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"C", @"r", @"aci", @"at", @"angi", @"vi", @"ti", @"angf", @"vf", @"acf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 202:
      {
	//Movil polar
	listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"C", @"aa", @"ar", @"angi", @"ri", @"vai", @"vri", @"ti", @"angf", @"rf", @"vaf", @"vrf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 203:
      {
	//Velocidad angular
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"vangi", @"vangf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 204:
      {
	//Aceleracion angular
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"aang", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 205:
      {
	//Aceleracion total
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"atoti", @"angi", @"atotf", @"angf", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 206:
      {
	//Frecuencia
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"f", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 207:
      {
	//Periodo
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"T", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 208:
      {
	//Numero de vueltas
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"n", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 209:
      {
	//Centro
	listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"x", @"y", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 210:
      {
	//Distancia
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto 1"), _(@"Objeto 2"), @"d", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 211:
      {
	//Longitud de arco
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"s", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 212:
      {
	//Coordenada
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"x", @"y", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 213:
      {
	//Velocidad relativa
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto 1"), _(@"Objeto 2"), @"v", @"ang", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil] ;
      }
      break;
      //Objetos del 250-299 pertenecen al submodulo estaticaRigida
    case 250:
      {
	//Sistema de referencia
	listaTitulos = [NSArray arrayWithObjects: @"g", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: gravedad, nil] ;
      }
      break;
    case 251:
      {
	//Elemento punto
	listaTitulos = [NSArray arrayWithObjects: _(@"Points"), @"x", @"y", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 252:
      {
	//Viga
	listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"m", @"lc", @"ang", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 253:
      {
	//Solido
	listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), @"m", @"xc", @"yc", @"ang", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil] ;
      }
      break;
    case 254:
      {
	//Par
	listaTitulos = [NSArray arrayWithObjects: @"M", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 255:
      {
	//Elementos de viga
	listaTitulos = [NSArray arrayWithObjects: _(@"Beam"), @"l", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 256:
      {
	//Elementos de solido
	listaTitulos = [NSArray arrayWithObjects: _(@"Solid"), @"x", @"y", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 257 ... 260:
      {
	//Fuerzas oblicuas
	listaTitulos = [NSArray arrayWithObjects: @"f", @"ang", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 261 ... 264:
      {
	//Fuerzas verticales y horizontales
	listaTitulos = [NSArray arrayWithObjects: @"f", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 265 ... 268:
      {
	//Fricciones oblicuas
	listaTitulos = [NSArray arrayWithObjects: @"N", @"ang", @"u", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 269 ... 272:
      {
	//Fricciones verticales y horizontales
	listaTitulos = [NSArray arrayWithObjects: @"N", @"u", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 273:
      {
	//Resultante general
	listaTitulos = [NSArray arrayWithObjects: @"M", @"f", @"ang", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 274:
    case 275:
      {
	//Resultantes con fuerza vertical y horizontal
	listaTitulos = [NSArray arrayWithObjects: @"M", @"f", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 276:
      {
	//Puntos
	listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 277 ... 280:
      {
	//Vigas oblicuas de 2 fuerzas
	listaTitulos = [NSArray arrayWithObjects: @"f", @"ang", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 281 ... 284:
      {
	//Vigas verticales y horizontales de 2 fuerzas
	listaTitulos = [NSArray arrayWithObjects: @"f", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 285 ... 288:
      {
	//Vigas oblicuas para armaduras
	listaTitulos = [NSArray arrayWithObjects: @"t", @"ang", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 289 ... 292:
      {
	//Vigas verticales y horizontales para armaduras
	listaTitulos = [NSArray arrayWithObjects: @"t", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 293:
      {
	//Armadura
	listaTitulos = [NSArray arrayWithObjects: _(@"Nombre"), nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 294:
      {
	//Nudo de armadura
	listaTitulos = [NSArray arrayWithObjects: _(@"Truss"), nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 295:
      {
	//Relation of angles
	listaTitulos = [NSArray arrayWithObjects: @"ang1", @"ang2", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil];
      }
      break;
      //Objetcs in 300-349 are of dynamics of circular motion
    case 300:
      {
	//Reference system
	listaTitulos = [NSArray arrayWithObjects: @"g", @"t", nil];
	listaDatos = [NSMutableArray arrayWithObjects: gravedad, @"1", nil];
      }
      break;
    case 301:
      {
	//Movil
	listaTitulos = [NSArray arrayWithObjects: _(@"Name"), @"m", @"y", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil];
      }
      break;
    case 302:
      {
	//Movil lineal
	listaTitulos = [NSArray arrayWithObjects: _(@"Name"), @"m", @"v", @"x", @"y", @"ang", @"a", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", nil];
      }
      break;
    case 303:
      {
	//Movil circular
	listaTitulos = [NSArray arrayWithObjects: _(@"Name"), @"m", @"vt", @"r", @"y", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil];
      }
      break;
    case 304:
      {
	//Movil perpendicular
	listaTitulos = [NSArray arrayWithObjects: _(@"Name"), @"m", @"vt", @"r", @"at", @"Ft", @"C", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @">", nil];
      }
      break;
    case 305:
      {
	//Energy
	listaTitulos = [NSArray arrayWithObjects: _(@"System i"), _(@"System f"), @"W", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil];
      }
      break;
    case 306:
      {
	//Angular momentum
	listaTitulos = [NSArray arrayWithObjects: _(@"System i"), _(@"System f"), @"M", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil];
      }
      break;
    case 307:
      {
	//Power
	listaTitulos = [NSArray arrayWithObjects: _(@"System i"), _(@"System f"), @"P", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil];
      }
      break;
    case 308:
    case 309:
      {
	//Initial and final system
	listaTitulos = [NSArray arrayWithObjects: _(@"Name"), _(@"Object 1"), _(@"Object 2"), _(@"Object 3"), _(@"Object 4"), nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil];
      }
      break;
    case 310:
      {
	//Sine of angle
	listaTitulos = [NSArray arrayWithObjects: @"y", @"r", @"ang", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil];
      }
      break;
      //Elements 311, 312, 313 are available.
    case 314:
      {
	//Angles
	listaTitulos = [NSArray arrayWithObjects: @"ang1", @"ang2", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil];
      }
      break;
    case 315:
      {
	//Couple
	listaTitulos = [NSArray arrayWithObjects: @"M", @"f", @"d", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil];
      }
      break;
    case 316:
      {
	//Triangle of accelerations
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"atot", @"ang", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil];
      }
      break;
    case 317 ... 320:
      {
	//Springs
        listaTitulos = [NSArray arrayWithObjects: _(@"Name"), @"k", @"x", nil] ;
        listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 321 ... 324:
      {
	//Oblique forces
	listaTitulos = [NSArray arrayWithObjects: @"f", @"ang", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 325 ... 328:
      {
	//Vertical and horizontal forces
	listaTitulos = [NSArray arrayWithObjects: @"f", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", nil] ;
      }
      break;
    case 330:
    case 332:
    case 341:
    case 342:
      {
	//Oblique frictions
	listaTitulos = [NSArray arrayWithObjects: @"N", @"u", @"ang", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil] ;
      }
      break;
    case 329:
    case 331:
    case 333:
      {
	//Vertical and horizontal frictions
	listaTitulos = [NSArray arrayWithObjects: @"N", @"u", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 334:
      {
	//Center
	listaTitulos = [NSArray arrayWithObjects: _(@"Name"), @"v", @"ang", @"x", @"y", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil];
      }
      break;
    case 335:
      {
	//Angular velocity
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"vang", nil] ;
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil] ;
      }
      break;
    case 336:
      {
	//Centripetal acceleration
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"ac", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil];
      }
      break;
    case 337:
      {
	//Angular acceleration
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"aang", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil];
      }
      break;
    case 338:
      {
	//Max acceleration
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"u", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", nil];
      }
      break;
    case 339:
      {
	//Polar mobile
	listaTitulos = [NSArray arrayWithObjects: _(@"Name"), @"m", @"vt", @"r", @"y", @"vr", @"ar", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", @"0", @"0", nil];
      }
      break;
    case 340:
      {
	//Lineal momentum
	listaTitulos = [NSArray arrayWithObjects: _(@"System i"), _(@"System f"), @"Fx", @"Fy", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", nil];
      }
      break;
    case 343:
      {
	//Inertia
	listaTitulos = [NSArray arrayWithObjects: _(@"System"), @"m", @"r", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", nil];
      }
      break;
    case 344:
      {
	//Absolute velocity
	listaTitulos = [NSArray arrayWithObjects: _(@"Objeto"), @"angR", _(@"Center"), @"v", @"ang", nil];
	listaDatos = [NSMutableArray arrayWithObjects: @"0", @"0", @"0", @"0", @"0", nil];
      }
      break;
    }

  if ([listaTitulos count] != [listaDatos count])
    {
      NSLog(@"Error in number of entries.");
    }
  else
    {
      tipo = [NSNumber numberWithInt: [nuevo tag] ] ;
      datos = [NSMutableDictionary dictionaryWithObjectsAndKeys: tipo, @"Tipo",
				   listaTitulos, @"Titulos",
				   listaDatos, @"Datos",
				   listaValores, @"Valores", nil] ;
      RELEASE(identificador);
      identificador = [[NSNumber alloc] initWithInt: conteoReferencia] ;
      [listaObjetos setObject: datos forKey: identificador] ;
      [nuevo setTag: [identificador intValue]] ;
      [dataViewer reloadData] ;
    }
}


- (void) borrarObjeto: (int)codigo
{
  /* insert your code here */
  RELEASE(identificador);
  identificador = [[NSNumber alloc] initWithInt: codigo] ;
  [listaObjetos removeObjectForKey: identificador] ;
  [dataViewer reloadData] ;
}

- (void) selecObjeto: (int)codigo
{
  /* insert your code here */
  [dataViewer deselectRow: [dataViewer selectedRow] ] ;
  RELEASE(identificador);
  identificador = [[NSNumber alloc] initWithInt: codigo] ;
  [dataViewer reloadData] ;
}

- (void) borrarObjetos: (BOOL) parametro
{
  conteoReferencia = 0;
  [listaObjetos removeAllObjects] ;
  [dataViewer reloadData] ;
}

- (NSString *) dataOfObject: (NSNumber *)aNumber
{
  int x;
  NSString *field;
  NSString *objectData = [NSString stringWithString: @""];
  NSMutableArray *keys, *values;
  
  keys = [[listaObjetos objectForKey: aNumber] objectForKey: @"Titulos"];
  values = [[listaObjetos objectForKey: aNumber] objectForKey: @"Datos"];

  for (x = 0; x < [keys count]; x++)
    {
      field = [NSString stringWithFormat: @"%@  =  %@ \n",
		    [[keys objectAtIndex: x] description],
		  [[values objectAtIndex: x] description]];
      
      objectData = [objectData stringByAppendingString: field];
    }

  return objectData;
}

- (void) setMenuForModule: (NSNotification *)notification
{
  id factor;
  NSMenuItem *item;
  NSMenu *factorsMenu;
  NSEnumerator *conv = nil;

  //Select the appropriate list of conversion factors
  switch ([chalkboard codigo: self])
    {
    case 0:
      {
	conv = [[[menuConversionDictionary objectForKey: @"kinematicParticles"]
		  objectAtIndex: sistema] keyEnumerator];
      }
      break;
    case 1:
      {
	conv = [[[menuConversionDictionary objectForKey: @"staticParticles"]
		  objectAtIndex: sistema] keyEnumerator];
      }
      break;
    case 2:
      {
	conv = [[[menuConversionDictionary objectForKey: @"dynamicParticles"]
		  objectAtIndex: sistema] keyEnumerator];
      }
      break;
    case 3:
      {
	conv = [[[menuConversionDictionary objectForKey: @"heat"]
		  objectAtIndex: 0] keyEnumerator];
      }
      break;
    case 4:
      {
	conv = [[[menuConversionDictionary objectForKey: @"kinematicCircular"]
		  objectAtIndex: sistema] keyEnumerator];
      }
      break;
    case 5:
      {
	conv = [[[menuConversionDictionary objectForKey: @"staticRigid"]
		  objectAtIndex: sistema] keyEnumerator];
      }
      break;
    case 6:
      {
	conv = [[[menuConversionDictionary objectForKey: @"dynamicsCircular"]
		  objectAtIndex: sistema] keyEnumerator];
      }
      break;
    default:
      break;
    }

  /* Set a transient menu with the conversion factors available
     in selected module */
  if (conv != nil)
    {
      factorsMenu = AUTORELEASE([[NSMenu alloc] initWithTitle: _(@"Factores")]);
      
      while ((factor = [conv nextObject]))
	{
	  item = [factorsMenu addItemWithTitle: [factor description]
			      action: @selector(insertFactor:)
			      keyEquivalent: @""];
	  [item setTarget: self];
	}
      
      [dataViewer setMenu: factorsMenu];
    }
  else
    {
      [dataViewer setMenu: nil];
    }
}

- (void) insertFactor: (id)sender
{
  //Add the selected conversion factor
  int y = [dataViewer selectedRow];

  if (y >= 0)
    {
      NSString *newValue;
      NSCell *cell = [[dataViewer tableColumnWithIdentifier: @"Datos"] dataCellForRow: y];
      NSArray *text = [[cell stringValue] componentsSeparatedByString: @"@"];
      NSMutableArray *lista = [[listaObjetos objectForKey: identificador]
				             objectForKey: @"Datos"];
      
      newValue = [[text objectAtIndex: 0] stringByTrimmingSpaces];
      newValue = [newValue stringByAppendingString: @" @ "];
      newValue = [newValue stringByAppendingString: [sender title]];
      [lista replaceObjectAtIndex: y withObject: newValue];
      [dataViewer reloadData];
      [dataViewer deselectAll: self];
    }
}

- (int) numberOfRowsInTableView: (NSTableView*)aTableView
{
  return [[[listaObjetos objectForKey: identificador] objectForKey: @"Titulos"] count] ;
}

- (id) tableView: (NSTableView*)aTableView objectValueForTableColumn: (NSTableColumn*)aTableColumn row: (int)rowIndex
{
  id imprimir ;     
  NSMutableDictionary *objeto = [listaObjetos objectForKey: identificador] ;
  NSMutableArray *lista = [objeto objectForKey: [aTableColumn identifier] ] ;
  imprimir = [lista objectAtIndex: rowIndex] ;
  
  if ([[aTableColumn identifier] isEqualToString: @"Titulos"])
    {
      imprimir = [imprimir description];
    }
  
  return imprimir ;
}

- (void) tableView: (NSTableView*)aTableView setObjectValue: (id)anObject forTableColumn: (NSTableColumn*)aTableColumn row: (int)rowIndex
{
  NSRange range;
  NSString *factor = nil;
  NSString *entry = [[anObject description] stringByTrimmingSpaces];
  NSMutableDictionary *objeto = [listaObjetos objectForKey: identificador];
  NSMutableArray *lista = [objeto objectForKey: [aTableColumn identifier]];

  range = [entry rangeOfString: @"@"];
  if (range.length > 0)
    {
      factor = [entry substringFromIndex: range.location];
      entry = [[entry substringToIndex: range.location]
		stringByTrimmingSpaces];
    }

  if ([[[[objeto objectForKey: @"Titulos"] objectAtIndex: rowIndex]
	 description] hasPrefix: @"ang"])
    {
      entry = [self processAngleEntry: entry];
    }
  else
    {
      entry = [self processEntry: entry];
    }
  
  if (factor != nil)
    {
      entry = [NSString stringWithFormat: @"%@ %@",
			entry,
			factor];
    }

  [lista replaceObjectAtIndex: rowIndex
		   withObject: entry];
  [dataViewer reloadData];
}

- (BOOL) tableView: (NSTableView*)aTableView shouldEditTableColumn: (NSTableColumn*)aTableColumn row: (int)rowIndex
{
  [chalkboard controlCursor: self];
  return YES;
}

- (void) tableView: (NSTableView*)aTableView 
   willDisplayCell: (id)aCell 
    forTableColumn: (NSTableColumn*)aTableColumn 
	       row: (int)rowIndex
{
  [aCell setEditable: YES];
}

- (void) dealloc
{
  RELEASE(listaObjetos) ;
  RELEASE(conversionDictionary);
  RELEASE(menuConversionDictionary);
  RELEASE(numbers);
  RELEASE(plus_minusSet);
  RELEASE(multiplication_divisionSet);
  RELEASE(identificador) ;
  [super dealloc] ;
}

@end
