/*
  Copyright (C) 2008 Ben Asselstine
  Written by Ben Asselstine

  This file is part of fileschanged.

  fileschanged is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  fileschanged is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with fileschanged; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301  USA
*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include "fileschanged.h"
#include <argz.h>
#include <error.h>
#include <locale.h>
#include "opts.h"
#include <unistd.h>
#include <sys/stat.h>
extern struct arguments_t arguments;

struct action_to_code_assoc_t
{
  enum fileschanged_action_enum_t id;
  char *code;
};
 
static int
show_event(char *filename, enum fileschanged_action_enum_t id, void *hook)
{
static struct action_to_code_assoc_t actions[FC_ACTION_MAX]=
{
    { FC_CREATED_FILE, "A"},
    { FC_CREATED_DIR, "A"},
    { FC_CHANGED_FILE, "M"},
    { FC_CHANGED_DIR, "M"},
    { FC_DELETED_FILE, "R"},
    { FC_DELETED_DIR, "R"},
    { FC_STARTEXECUTING_FILE, "E"},
    { FC_STOPEXECUTING_FILE, "X"},
};
  if (arguments.fileschanged.exec_cmd)
    {
      struct sigaction sa;
      sa.sa_handler = SIG_IGN;
#ifdef SA_NOCLDWAIT
      sa.sa_flags = SA_NOCLDWAIT;
#else
      sa.sa_flags = 0;
#endif
      sigemptyset (&sa.sa_mask);
      sigaction (SIGCHLD, &sa, NULL);
      if (!fork ())
	{
	  execlp (arguments.fileschanged.exec_cmd, 
		  arguments.fileschanged.exec_cmd, 
		  actions[id].code, filename, NULL);
	}
    }
  else
    {
      if (arguments.fileschanged.showaction)
	{
	  fprintf (stdout, "%s ", actions[id].code);
	}
      fprintf (stdout, "%s\n", filename);
      fflush (stdout);
    }
  return 0;
}

static void 
sigHandler (int sig)
{
  if (( sig == SIGINT ) || (sig == SIGPIPE))
    ;
  else
    signal(sig, SIG_IGN);
}

static int 
init_locales (const char *name)
{
#ifdef ENABLE_NLS
  setlocale (LC_ALL, "");
  if (!bindtextdomain (PACKAGE_NAME, LOCALEDIR))
    error (1, 0, "Error: Couldn't bind textdomain");
  if (!textdomain (PACKAGE_NAME))
    error(1, 0, "Error: Couldn't set textdomain");
#endif
  return 0;
}
static int 
add_it_to_the(char **filelist, size_t *filelist_len, char *filename)
{
  return argz_add (filelist, filelist_len, filename);
}

static int 
for_every_filename (int (*for_every_file)(int (*)(char **, size_t *, char *), char **filelist, size_t *filelist_len), int (*add_it_to_the)(char **filelist, size_t *filelist_len, char *filename), char **filelist, size_t *filelist_len)
{
  for_every_file (add_it_to_the, filelist, filelist_len);
  return 0;
}

static int 
on_the_command_line (int (*process_file)(char **filelist, size_t *filelist_len, char *filename), char **filelist, size_t *filelist_len)
{
  unsigned int i;
  int retval;
  for(i = 0; i < arguments.arraylen; i++)
    {
      retval = process_file (filelist, filelist_len, arguments.args[i]);
    }
  return 0;
}

static void 
chop (char *line)
{
  char *tmp;
  tmp = strpbrk (line, "\r\n");
  if (tmp)
    tmp[0] = '\0';
  return;
}

static int 
in_filelist_file (int (*process_file)(char **filelist, size_t *filelist_len, char *filename), char **filelist, size_t *filelist_len)
{
  //read the list of files from a file.
  FILE *fileptr;
  char *filename;
  char *line = NULL;
  size_t n = 0;
  int retval;
  filename = arguments.fileschanged.filelist_filename;
  if (strcmp (filename, "-") == 0)
    fileptr = stdin;
  else
    {
      fileptr = fopen (filename, "r");
      if (!fileptr)
	error(1, 0, _("Error: couldn't open '%s'\n"), filename);
    }
  while (getline (&line, &n, fileptr) > 0)
    {
      chop (line);
      if (line[0] != '#')
	retval = process_file (filelist, filelist_len, line);
    }
  if (line)
    free (line);

  if (fileptr != stdin)
    fclose (fileptr);
  return 0;
}
int 
main (int argc, char **argv)
{
  fileschanged_t f;
  char *filelist = NULL;
  size_t filelist_len = 0;
  int err = 0;
  unsigned int count = 0;

  if (init_locales (PACKAGE_NAME))
    error(1, 0, "Error: Couldn't set locale");

  parse_opts (argc, argv, &arguments);
  signal (SIGINT, sigHandler);

  if (arguments.fileschanged.filelist)
    err = for_every_filename (in_filelist_file, add_it_to_the,
			      &filelist, &filelist_len);
  else if (arguments.fileschanged.filestomonitor)
    err = for_every_filename (on_the_command_line, add_it_to_the,
			      &filelist, &filelist_len);

  if (!err)
    {
      //bail if there's nothing to do
      count = argz_count (filelist, filelist_len);
      if (count == 0)
	error(1, 0, _("Error: No files to monitor."));

      err = fileschanged_init (&f, filelist, filelist_len, 
			       arguments.fileschanged.recursive, 
			       arguments.fileschanged.dereference_symlinks, 
			       arguments.fileschanged.filechangetimeout);
      if (err)
	{
	  error (0, 0, "Couldn't initialize fileschanged! (%m)");
	  return 1;
	}

      setlinebuf (stdout);

      err = fileschanged_monitor (f, NULL, show_event);

      //gets here by SIGINT or SIGPIPE

      fileschanged_destroy (f);
    }
  exit (err);
}
