/*
  Copyright (C) 2008 Ben Asselstine
  Written by Ben Asselstine

  This file is part of fileschanged.

  fileschanged is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  fileschanged is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with fileschanged; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301  USA
*/

#ifndef FILESCHANGED_H
#define FILESCHANGED_H
typedef void* fileschanged_t;

/*
 * enum fileschanged_action_enum_t:
 *
 * This enumeration lists the events that fileschanged generates. 
 *
 * Callers will receive this action enumeration in the callback they provide.
 * Don't get too excited about STARTEXECUTING and STOPEXECUTING -- FAM doesn't
 * actually support those events on x86.
 * 
 */
enum fileschanged_action_enum_t
{
	FC_CREATED_FILE,
	FC_CREATED_DIR,
	FC_CHANGED_FILE,
	FC_CHANGED_DIR,
	FC_DELETED_FILE,
	FC_DELETED_DIR,
	FC_STARTEXECUTING_FILE,
	FC_STOPEXECUTING_FILE,
	FC_ACTION_MAX,
};

#include <sys/types.h>

/* 
 * fileschanged_init: instantiate a new fileschanged object.
 *
 * In order to make a new fileschanged object, we need to know a few things.
 *
 * 1. We need to know the files we want to monitor.  This is provided in 
 * FILELIST_ARGZ and FILELIST_ARGZ_LEN.  These files need to present when 
 * this function is called.  Files and Directories that are not found will 
 * not be monitored.
 * To learn more about argz, try `$ man -k argz'.
 *
 * 2. We need to know if we're going to recurse into subdirectories of
 * directories.  Set RECURSE to 1, to enable recursion.
 *
 * 3. We need to know if we're going to follow symbolic links or not.
 * Set DEREFERENCE_SYMLINKS to 1, to enable the following of symlinks.
 *
 * 4. We need to know how long to wait for a file to stop changing before
 * it's considered changed.  The TIMEOUT parameter controls this.  If it is 0
 * the file is considered changed whenever FAM tells fileschanged that the
 * file has changed.  The TIMEOUT parameter is in seconds.
 */
int fileschanged_init (fileschanged_t *f, char *filelist_argz, size_t filelist_argz_len, unsigned int recurse, unsigned int dereference_symlinks, int timeout);

/*
 * fileschanged_monitor: Here we pass the newly created fileschanged object,
 * and we give it a callback HANDLER, along with a HOOK to some data we'd 
 * like to get back when HANDLER is called.
 *
 * HANDLER will be called every time something has happened to the files we're 
 * monitoring.  HANDLER has a char * passed to it -- that's the name of the 
 * file, an action (previously described), and the HOOK we passed it earlier i
 * in the fileschanged_monitor call.
 *
 * fileschanged_monitor continues monitoring until it receives a SIGINT.
 *
 */
int fileschanged_monitor (fileschanged_t f, void *hook, int (*handler)(char *, enum fileschanged_action_enum_t, void *));

/*
 * fileschanged_destroy:  Use this to discard of your fileschanged object.
 */
void fileschanged_destroy (fileschanged_t f);
#endif
