/***************************************************************
 *             Finite Element Method Object Library            *
 *                         tutorial 1                          *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 	2.6.0                  *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2006 CREUSE Emmanuel
 * copyright © 2006 SOUALEM Nadir
 * copyright © 2012 COLLARD Christophe
 * copyright © 2006,2012 Centre National de la Recherche Scientifique
 * copyright © 2006 Arts et Métiers ParisTech
 * copyright © 2006 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2006 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \class tuto1
    \brief tuto1 example \n

    \htmlonly 
    <FONT color="#838383">

    tuto1 belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2006 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly  2006 SOUALEM Nadir \n
	     copyright \htmlonly &#169; \endhtmlonly 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; 2006, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
	     copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 2.6.0
    \date 2006-2012
    \bug none
    \warning none
*/


#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__STRING_H)
#include <string>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__TIME_H)
#include <time.h>
#endif

#if !defined(__MATHS_H)
#include "MOL++/maths.h"
#endif

#if !defined(__VECTORS_H)
#include "MOL++/vectors.h"
#endif

#if !defined(__MESH_TRI_2D_H)
#include "FEMOL++/meshes/mesh_tri_2d.h"
#endif

#if !defined(__TRIANGLE_H)
#include "FEMOL++/meshes/triangle.h"
#endif

#if !defined(__P1C2D_H)
#include "FEMOL++/elements/p1c2d.h"
#endif

#if !defined(__AFFICHE_H)
#include "tests/affiche.h"
#endif

// Second member
long double f_p1c (long double x, long double y)
{
  return y * (1 - y) * (2 * exp(y) + x * (3 + x) * exp(x)) + x * (1 - x) * (2 * exp(x) + y * (3 + y) * exp(y));
}

// Exact solution u_p1c
long double u_p1c (long double x, long double y)
{
  return (exp(x) + exp(y)) * x * (1 - x) * y * (1 - y);
}

// d/dx (u_p1c)
long double dudx_p1c (long double x, long double y)
{
  return y * (1 - y) * ((1 - 2*x) * exp(y) + (1 - x - pow (x, 2)) * exp(x));
}

// d/dy (u_p1c)
long double dudy_p1c (long double x, long double y)
{
  return x * (1 - x) * ((1 - 2*y) * exp(x) + (1 - y - pow (y, 2)) * exp(y));
}


using namespace std;
using namespace mol;
using namespace femol;


//====
main()
//====
{
  cout << "-----------------------------------------\n";
  cout << "-----------------------------------------\n";
  cout << " BEGINNING tuto1 \n";
  cout << "-----------------------------------------\n";
  cout << "-----------------------------------------\n";
  cout << endl;

  //-----------
  // Mesh files
  //-----------
  int nb_meshs = 7;
  char* mesh[nb_meshs];
  mesh[0] = "../../tests/FEMOL++/data/22.txt";
  mesh[1] = "../../tests/FEMOL++/data/42.txt";
  mesh[2] = "../../tests/FEMOL++/data/82.txt";
  mesh[3] = "../../tests/FEMOL++/data/162.txt";
  mesh[4] = "../../tests/FEMOL++/data/322.txt";
  mesh[5] = "../../tests/FEMOL++/data/642.txt";
  mesh[6] = "../../tests/FEMOL++/data/1282.txt";

  //------------------------------------
  // Nb points for numerical integration
  //------------------------------------
  int NbIntPts = 1;

  //----------------
  // Local variables
  //----------------
  long double L2_error, semi_H1_error, estiglob, ratio;
  long double full_H1_error;
  long double estiloc2, estiglob2;
  long double L2_error_anc, semi_H1_error_anc;
  long double full_H1_error_anc;
  long double estiloc;
  int ndof,ndof_anc;

  //-----------------------------------
  // Local files (destroyed at the end)
  //-----------------------------------
  ofstream file1 ("tuto1.dat");

  //-------------------------------------------------------------------
  // Local files containing gnuplot instructions (destroyed at the end)
  //-------------------------------------------------------------------
  ofstream file2 ("visu.gnu");
  file2 << "set logscale xy" << endl;
  file2 << "plot [1:20000] [0.0000060:50] \"tuto1.dat\"  using 1:2 title \"L2 Error\" with linespoints, \"tuto1.dat\" using 1:3 title \"Estimator\" with linespoints, \"tuto1.dat\" using 1:4 title \"Full H1 Error\" with linespoints" << endl;
  file2 << "pause -1 \"Press Enter\" " << endl;
  file2.close ();

  ofstream file3 ("visu2.gnu");
  file3 << "plot [0:20000] [0:0.5] \"tuto1.dat\"  using 1:5 title \" Ratio Full H1 Error / Estimator\" with linespoints " << endl;
  file3 << "pause -1 \"Press Enter\" " << endl;
  file3.close ();

  //----------
  // Main loop
  //----------
  for (int num_mesh=0; num_mesh<=nb_meshs-1; num_mesh++)
    { cout << "---------------------------------------------" << endl;
      cout << "Mesh number " << num_mesh+1 << " / " << nb_meshs << endl;
      cout << "---------------------------------------------" << endl;

      //------------------------------------
      // Déclaration of p1c2d finite element
      //------------------------------------
      cout << "Finite Element Loading... ";
      p1c2d<long double> Th(mesh[num_mesh]);
      cout << green << "OK !" << reset;

      //-------------------
      // Mesh visualisation
      //-------------------
      //cout << "Mesh visualisation... "<< endl;
      //Th.visu2d();
      //cout << green << "OK !" << reset;

      //------------------------------------------
      // Déclaration of the Global Stifness Matrix
      //------------------------------------------
      //matrix<long double> DuDv(Th.GlobalDofNumber(),Th.GlobalDofNumber());
      ndof = Th.GlobalDofNumber ();
      cout << "Number of DoF = " << ndof << endl;
      spmatrix<long double> DuDv (ndof, ndof, Th.NbMaxSegmentsPerNode()+1);

      //------------------------------------------
      // Computation of the Global Stifness Matrix
      //------------------------------------------
      cout << "Computation of the Global Stifness Matrix... ";
      Th.DUDV (DuDv);
      cout << green << "OK !" << reset;

      //-----------------------------------
      // Computation on the Right-Hand-Side
      //-----------------------------------
      cout << "Computation on the Right-Hand-Side... ";
      vector<long double> s = Th.B (f_p1c,NbIntPts);
      cout << green << "OK !" << reset;

      //------------------------------
      // Dirichlet Boundary Conditions
      //------------------------------
      cout << "Dirichlet Boundary Conditions... ";
      Th.Dirichlet (DuDv, s, u_p1c);
      cout << green << "OK !" << reset;

      //-------------------------
      // Linear system resolution
      //-------------------------
      vector<long double> x0 (ndof);
      vector<long double> uh (ndof);
      int iter;
      cout << "Linear system resolution... ";
      uh = conjugate_gradient (DuDv, s, x0, iter, 1.e-08);
      cout << green << "OK !" << reset;

      //------------------------------------------------
      // Computation of errors and orders of convergence
      //------------------------------------------------
      L2_error_anc = L2_error;
      semi_H1_error_anc = semi_H1_error;
      full_H1_error_anc = full_H1_error;

      cout << "Computation of errors... ";
      L2_error = Th.error_L2 (u_p1c, uh, NbIntPts);
      semi_H1_error = Th.error_semi_H1 (dudx_p1c, dudy_p1c, uh, NbIntPts);
      full_H1_error = sqrt (pow (L2_error, 2) + pow (semi_H1_error, 2));
      cout << green << "OK !" << reset;
      if (num_mesh > 0)
	{ cout << "Comparison between meshes " << num_mesh << " and " << num_mesh+1 << ": " << endl;
	  cout << endl;
	  cout << "Order of convergence in h for L2 norm = " << -2 * ((log (L2_error) - log (L2_error_anc)) / (log ((long double) ndof) - log ((long double)ndof_anc))) << endl;
	  cout << "Order of convergence in h for full H1 norm= " << -2 * ((log(full_H1_error) - log (full_H1_error_anc)) / (log ((long double) ndof) - log ((long double) ndof_anc))) << endl;
	  cout << endl;
	}

      //-------------------------------
      // Computation of error estimator
      //-------------------------------
      cout << "Computation of error estimator... ";
      estiglob2 = 0;
      for (int numelement=1; numelement<=Th.nbelement(); numelement++)
	{ estiloc2 = pow (Th.Laplacian_Residual_L2 (f_p1c,numelement), 2) + 0.5 * pow (Th.Grad_Normal_Jump_L2 (uh, numelement), 2);
	  estiglob2 += estiloc2;
	}
      estiglob = sqrt (estiglob2);
      cout << green << "OK !" << reset;
      cout << endl;

      //----------------------------------------------
      // Computation of ratio full H1 error /estimator
      //----------------------------------------------
      ratio = full_H1_error / estiglob;

      //------------------------------
      // Results are written in a file
      //------------------------------
      file1 << ndof << "  " << L2_error << "  " << estiglob << "  " << full_H1_error << "  " << ratio << endl;

      ndof_anc = ndof;
      cout << endl;
    }

  //-------------------------
  // Visualisation of results
  //-------------------------
  system ("gnuplot visu.gnu");
  system ("gnuplot visu2.gnu");

  //------------------------------
  // Closing and removing of files
  //------------------------------
  file1.close ();
  remove ("tuto1.dat");
  remove ("visu.gnu");
  remove ("visu2.gnu");

  cout << endl;
  cout << "-----------------------------------------\n";
  cout << "-----------------------------------------\n";
  cout << " END tuto1 \n";
  cout << "-----------------------------------------\n";
  cout << "-----------------------------------------\n";
  //-----------------------------------------
}
