/***************************************************************
 *             Finite Element Method Object Library            *
 *              class p1c2d : declaration for p1c2d            *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 2.6.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2003,2004,2005,2006 CREUSE Emmanuel
 * copyright © 2003,2004,2005,2006 SOUALEM Nadir
 * copyright © 2011 COLLARD Christophe
 * copyright © 2003,2004,2005,2006,2011 Centre National de la Recherche Scientifique
 * copyright © 2003,2004,2005,2006 Arts et Métiers ParisTech
 * copyright © 2003,2004,2005,2006 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2003,2004,2005,2006 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2003,2004,2005,2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2011 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \class p1c2d
    \brief p1c2d library \n

    \htmlonly 
    <FONT color="#838383">

    p1c2d belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2003, 2004, 2005, 2006 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly  2003, 2004, 2005, 2006 SOUALEM Nadir \n
	     copyright \htmlonly &#169; \endhtmlonly 2011 Christophe COLLARD \n
             copyright \htmlonly &#169; 2003, 2004, 2005, 2006, 2011 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
	     copyright \htmlonly &#169; 2011 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 2.6.0
    \date 2003-2011
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type p1c_2d
#endif

#ifndef _p1c2d_h
#define _p1c2d_h

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__VECTORS_H)
#include "MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "MOL++/matrix.h"
#endif

#if !defined(__SPMATRIX_H)
#include "MOL++/spmatrix.h"
#endif

#if !defined(__MESH_TRI_2D_H)
#include "FEMOL++/meshes/mesh_tri_2d.h"
#endif

#if !defined(__INTEGRATION_H)
#include "MOL++/integration.h"
#endif


/*
#if !defined(__DATA_H)
#include "../data/data.h"
#endif
*/
using namespace std;
using namespace mol;

namespace femol
{


//====================================================
template <class T> class p1c2d : public mesh_tri_2d<T>
//====================================================
{
  using mesh<triangle<T>,T>::alive;
  using mesh<triangle<T>,T>::dimspace_;
  using mesh<triangle<T>,T>::nbvertex_;
  using mesh<triangle<T>,T>::nbelement_;
  using mesh<triangle<T>,T>::vertices;
  using mesh<triangle<T>,T>::element;

  using mesh_tri_2d<T>::nbsegments_;
  using mesh_tri_2d<T>::SetOfEdges_;
  using mesh_tri_2d<T>::Type_of_edge_;
  using mesh_tri_2d<T>::EdgesPerTriangle_;
  using mesh_tri_2d<T>::NbTrianglesPerEdge_;
  using mesh_tri_2d<T>::TrianglesPerEdge_;
  using mesh_tri_2d<T>::TrianglesPerTriangle_;
  using mesh_tri_2d<T>::NumberOfNeighboorOfTriangles_;
  using mesh_tri_2d<T>::NumLocNodeInTriangle_;
  using mesh_tri_2d<T>::NormalDirection_;
  using mesh_tri_2d<T>::hminT_;
  using mesh_tri_2d<T>::hE_;
  using mesh_tri_2d<T>::hminE_;
  using mesh_tri_2d<T>::NormEdge_;
  using mesh_tri_2d<T>::NormalUnitVector_;
  using mesh_tri_2d<T>::NbMaxSegmentsPerNode_;
  using mesh_tri_2d<T>::NbSegmentsPerNode_;

  private:
    int DofNumber_; // Number of DDL par element (3 for p1c2d)
    int GlobalDofNumber_; // Number of DDL for the whole mesh (nbvertex_ for p1c2d)
    matrix<T> RefMassMatrix_; // 3*3 Mass matrix on the reference element
    vector<T> BuildBloc(T (*f)(T,T),int,integration<T>) const;

  public:
    p1c2d (); // Default Constructor
    p1c2d (const char*, const char* = ""); // Constructor
    p1c2d (const p1c2d<T>&); // Copy Constructor

    p1c2d<T>& operator = (const p1c2d<T>&); //p1c2d affectation
    int DofNumber() const {return DofNumber_;}
    int GlobalDofNumber() const {return GlobalDofNumber_;}
    matrix<T> RefMassMatrix() const {return RefMassMatrix_;}
    vector<T> BasisFunction (T, T) const;
    matrix<T> DiffBasisFunction (T xref=0, T yref=0) const;
    virtual int operator () (int, int) const;
    template <class Tf> friend bool operator == (const p1c2d<Tf>&, const p1c2d<Tf>&);
    template <class Tf> friend bool operator != (const p1c2d<Tf>&, const p1c2d<Tf>&);
    matrix<T> LocalMassMatrix (int) const;      // 3*3 Mass matrix on a given element
    matrix<T> LocalStiffnessMatrix (int) const; // 3*3 Stiffness matrix on a given element
    vector<T> Fk (T,T, int) const; // Affine transformation from the reference element to the actual one
    template <template <class> class matrix_type> void DUDV (matrix_type<T>&) const;   // Global Stiffness matrix
    vector<T> B (T (*f)(T,T), int NbIntPts) const;
    template <template <class> class matrix_type> void Dirichlet (matrix_type<T>&, vector<T>&, T (*g)(T,T)) const;   // Dirichlet BC 
    T error_L2_loc (T (*u)(T,T), const vector<T>&, int, int) const; // Local L2 error
    T error_L2 (T (*u)(T,T), const vector<T>&, int) const; // L2 error
    T error_semi_H1 (T (*dudx)(T,T), T (*dudy)(T,T), const vector<T>&, int) const; // Semi H1 error
    T error_semi_H1_loc2 (T (*dudx)(T,T), T (*dudy)(T,T), const vector<T>&, int, int) const; // Semi H1 error
    T Grad_Normal_Jump_L2 (const vector<T>&, int) const;
    T Laplacian_Residual_L2 (T (*f)(T,T), int) const;
    void refine_FE (vector<int>&); // refine the Finite Element
    void refine_FE_angle (vector<int>&, T); // refine the Finite Element
    T RT_L2Norm (T (*f) (T,T), int) const;
};


//=====Private methods for p1c2d=============================================


//=====Public methods for p1c2d==============================================


/*!
  \brief Default Constructor
  \n \n
*/

//====================================
template <class T>
p1c2d<T>::p1c2d () : mesh_tri_2d<T> ()
//====================================
{
  DofNumber_ = GlobalDofNumber_ = 0;
}


/*!
  \brief Constructor
  \n \n
  \param file1 file with nodes \n
  \param file2 file with elements (if not merged with file1)
*/

//====================================================================================
template <class T>
p1c2d<T>::p1c2d (const char* file1, const char* file2) : mesh_tri_2d<T> (file1, file2)
//====================================================================================
{
  DofNumber_ = 3;
  GlobalDofNumber_ = nbvertex_;
  RefMassMatrix_ = matrix<T> (3, 3, true, 1./24.);

  for (int i=1; i<=3; i++)
    RefMassMatrix_(i,i) *= 2;
}


/*!
  \brief Copy constructor
  \param EF p1c2d to duplicate
*/

//========================================================
template <class T>
p1c2d<T>::p1c2d (const p1c2d<T>& EF) : mesh_tri_2d<T> (EF)
//========================================================
{
  DofNumber_ = EF.DofNumber_;
  GlobalDofNumber_ = EF.GlobalDofNumber_;
  RefMassMatrix_ = EF.RefMassMatrix_;
}


/*!
  \brief Standard operator = for p1c2d \n \n
  \param EF p1c2d to copy \n
  \return reference of the left hand size p1c2d (for multiple equalities)
  \n\n
*/

//=================================================
template <class T>
p1c2d<T>& p1c2d<T>::operator = (const p1c2d<T>& EF)
//=================================================
{
  (*this).mesh_tri_2d<T>::operator = (EF);
  DofNumber_ = EF.DofNumber_;
  GlobalDofNumber_ = EF.GlobalDofNumber_;
  RefMassMatrix_ = EF.RefMassMatrix_;

  return (*this);
}


/*!
  \brief Value of the three basis functions in the reference element \n \n
  \param xref x-coordinate of the point \n
  \param yref y-coordinate of the point \n
  \return vector \f$(\varphi_1(xref,yref),\varphi_2(xref,yref),\varphi_3(xref,yref))\f$
   where \f$(\varphi_i)_{1\leq i\leq 3}\f$ is the i-th basis function.
  \n\n
*/


//======================================================
template <class T>
vector<T> p1c2d<T>::BasisFunction (T xref, T yref) const
//======================================================
{
  vector<T> P (DofNumber_);
  P[1] = 1 - xref - yref;
  P[2] = xref;
  P[3] = yref;

  return P;
}


/*!
  \param xref x-coordinate of the point \n
  \param yref y-coordinate of the point \n
  \return matrix
  \f$ \left[\begin{array}{ccc}
      \frac{\partial\varphi_1}{\partial x_1} & \frac{\partial\varphi_2}{\partial x_1} & \frac{\partial\varphi_3}{\partial x_1} \ \
      \frac{\partial\varphi_1}{\partial x_2} & \frac{\partial\varphi_2}{\partial x_2} & \frac{\partial\varphi_3}{\partial x_2}
      \end{array} \right]
  \f$
  where \f$(\varphi_i)_{1\leq i\leq 3}\f$ is the i-th basis function.
*/

//==========================================================
template <class T>
matrix<T> p1c2d<T>::DiffBasisFunction (T xref, T yref) const
//==========================================================
{
  matrix<T> DP (2, DofNumber_);
  DP (1,1) = DP (2,1) = -1;
  DP(1,2) = DP(2,3) = 1;

  return DP;
}


/*!
  \param num_element element's number
  \param j the j-th dof \f$(1\leq j\leq 3)\f$
  \return global number of the j-th of the element num_element
*/

//======================================================
template <class T>
int p1c2d<T>::operator () (int num_element, int j) const
//======================================================
{
  assert ((j >= 1) && (j <= DofNumber_) && (num_element >= 1) && (num_element <= nbelement_));
  return number (element[num_element-1][j]);
}


/*!
  \param EF1 type const p1c2d<T>&
  \param EF2 type const p1c2d<T>&
  \return true if EF1 and EF2 are equal, false otherwise.
*/

//===========================================================
template <class Tf>
bool operator == (const p1c2d<Tf>& EF1, const p1c2d<Tf>& EF2)
//===========================================================
{
  bool boolean = true;

  boolean *= (mesh_tri_2d<Tf>(EF1) == mesh_tri_2d<Tf>(EF2));
  boolean *= (EF1.DofNumber_ == EF2.DofNumber_);
  boolean *= (EF1.GlobalDofNumber_ == EF2.GlobalDofNumber_);
  boolean *= (EF1.RefMassMatrix_ == EF2.RefMassMatrix_);

  return boolean;
}


/*!
  \param EF1 type const p1c2d<T>&
  \param EF2 type const p1c2d<T>&
  \return true if EF1 and EF2 are different, false otherwise.
*/

//===========================================================
template <class Tf>
bool operator != (const p1c2d<Tf>& EF1, const p1c2d<Tf>& EF2)
//===========================================================
{
  return !(EF1 == EF2);
}


/*!
  \param num_element element's number
  \return local mass matrix of the element num_element
*/

//=========================================================
template <class T>
matrix<T> p1c2d<T>::LocalMassMatrix (int num_element) const
//=========================================================
{
  assert ((num_element >= 1) && (num_element <= nbelement_));
  return RefMassMatrix_ * element[num_element-1].AbsDet();
}


/*!
  \param num_element element's number
  \return local stifness  matrix of the element num_element
*/

//==============================================================
template <class T>
matrix<T> p1c2d<T>::LocalStiffnessMatrix (int num_element) const
//==============================================================
{
  assert ((num_element >= 1) && (num_element <= nbelement_));
  matrix<T> inv = element[num_element-1].InvMk();

  return element[num_element-1].AbsDet() * 0.5 * t(DiffBasisFunction()) * inv * t(inv) * DiffBasisFunction();
}


/*!
  \param x type T
  \param y type T
  \param i type integer
  \return vector \f$F_k(x,y,i)\f$
*/

//============================================
template <class T>
vector<T> p1c2d<T>::Fk (T x, T y, int i) const
//============================================
{
  matrix<T> mat (2,2);
  vector<T> vec(2);
  vec[1] = x;
  vec[2] = y;

  vector<T> A1A2 = element[i-1][2] - element[i-1][1];
  vector<T> A1A3 = element[i-1][3] - element[i-1][1];
  mat(1,1) = A1A2[1];
  mat(1,2) = A1A3[1];
  mat(2,1) = A1A2[2];
  mat(2,2) = A1A3[2];

  return mat * vec + element[i-1][1];
}


/*!
  \return the global stifness matrix of the p1nc2d element
*/

//==============================================
template <class T>
template <template <class> class matrix_type>
void p1c2d<T>::DUDV (matrix_type<T> & mat) const
//==============================================
{
  matrix<T> Stiff (DofNumber_, DofNumber_);
  for (int k=1; k<=nbelement_; k++)
    { Stiff = LocalStiffnessMatrix (k);
      for (int i=1; i<=DofNumber_; i++)
	for (int j=1; j<=DofNumber_; j++)
	  mat (number(element[k-1][i]), number(element[k-1][j])) += Stiff (i,j);
    }
}


/*!
  \brief Value of the contribution to the right-hand side of the variational formulation of a given element \n \n
  \param \f$ f \f$ the function   \n
  \param num_element the element on which we want to integrate \n
  \param itg the Gauss points and weight used for the numerical integration
  \return vector \f$(\int_{num_{element}} f(x,y) \varphi_1(x,y) dx dy,\int_{num_{element}} f(x,y) \varphi_2(x,y) dx dy,\int_{num_{element}} f(x,y) \varphi_3(x,y) dx dy)\f$
   where \f$(\varphi_i)_{1\leq i\leq 3}\f$ is the i-th basis function associated to the element num_element.
  \n\n
*/

//====================================================================================
template <class T>
vector<T> p1c2d<T>::BuildBloc (T (*f)(T,T), int num_element, integration<T> itg) const
//====================================================================================
{
  vector<T> res (DofNumber_);
  vector<T> Basis (DofNumber_);
  T omegak, omegal, xk, xl;
  vector<T> Fxkl;

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight(k);
      xk = itg.point(k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight(l);
          xl = (1-xk) * itg.point(l);
	  Basis = BasisFunction (xk, xl);
	  Fxkl = Fk (xk, xl, num_element);
          for (int i=1; i<=DofNumber_; i++)
	    res[i] += omegak * omegal * (1-xk) * f (Fxkl[1], Fxkl[2]) * Basis[i];
	 }
    }
  res *= element[num_element-1].AbsDet();

  return res;
}


/*!
  \brief Value of the right-hand side of the variational formulation \n \n
  \param \f$ f \f$ the function   \n
  \param NbIntPts the number of Gauss points to do the numerical integration \n
  \return vector v with v_i=\f$(\int_{\Omega} f(x,y) \varphi_i(x,y) dx dy \f$
   where \f$(\varphi_i)_{1\leq i\leq nbvertex}\f$ is the i-th global basis function.
  \n\n
*/

//=====================================================
template <class T>
vector<T> p1c2d<T>::B (T (*f)(T,T), int NbIntPts) const
//=====================================================
{
  assert (NbIntPts>0);
  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  vector<T> B_ (GlobalDofNumber_);
  vector<T> Bloc (DofNumber_);

  for (int k=1; k<=nbelement_; k++)
    { Bloc = BuildBloc (f, k, itg);
      for (int i=1; i<=DofNumber_; i++)
	B_[number(element[k-1][i])] += Bloc[i];
    }

  return B_;
}


/*!
  \brief Modifies the linear system to put non homogeneous Dirichlet BC \n \n
  \param MAT the matrix of the linear system \n
  \param F the Right-Hand-Side of the linear system \n
  \param g the function to find the value of the solution on the boundary
  \return MAT and F are modified
  \n\n
*/

//=============================================================================
template <class T>
template <template <class> class matrix_type>
void p1c2d<T>::Dirichlet (matrix_type<T> & mat, vector<T>&F, T (*g)(T,T)) const
//=============================================================================
{
  for (int i=1; i<=GlobalDofNumber_; i++)
    if (mesh_tri_2d<T>::operator()(i).boundary_condition())
      { F[i] = (*g) (vertices[i-1][1], vertices[i-1][2]);
	for (int j=1; j<=GlobalDofNumber_; j++)
	  mat (i, j) = 0;
	mat (i, i) = 1;
	for (int k=1; k<=GlobalDofNumber_; k++)
	  if (k!=i)
	    { F[k] += -mat(k,i) * F[i];
	      mat (k, i) = 0;
	    }
      }
}


//==============================================================================================
template <class T>
T p1c2d<T>::error_L2_loc (T (*u)(T,T), const vector<T>& uh, int NbIntPts, int num_element) const
//==============================================================================================
{
  T LocalError = 0;
  T omegak, omegal, xk, xl, SumK;
  vector<T> Fxkl;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  vector<T> Basis (DofNumber_);

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight (l);
          xl = (1-xk) * itg.point (l);
	  Basis = BasisFunction (xk, xl);
          SumK = 0;
	  for (int i=1; i<=DofNumber_; i++)
	    SumK += uh[number(element[num_element-1][i])] * Basis[i];
	  Fxkl = Fk (xk, xl, num_element);
	  LocalError += omegak * omegal * (1-xk) * pow (u (Fxkl[1], Fxkl[2]) - SumK, 2);
	}
    }

  return sqrt (element[num_element-1].AbsDet() * LocalError);
}


/*!
  \brief Computes the L2 error between u (exact function) and u_h (discrete vector) \n \n
  \param u the exact function   \n
  \param uh the discrete solution associated to p1c2d discretization  \n
  \param NbIntPts the number of Gauss points to do the numerical integration \n
  \return \f$ ||u-u_h||_{L^2(\Omega)}\f$
  \n\n
*/

//=========================================================================
template <class T>
T p1c2d<T>::error_L2 (T (*u)(T,T), const vector<T>& uh, int NbIntPts) const
//=========================================================================
{
  T GlobalError = 0;
  T LocalError;
  T omegak, omegal, xk, xl, SumK;
  vector<T> Fxkl;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  vector<T> Basis (DofNumber_);

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { LocalError = 0;
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int l=1; l<=itg.NbIntPts(); l++)
	    { omegal = itg.weight (l);
	      xl = (1-xk) * itg.point (l);
	      Basis = BasisFunction (xk, xl);
	      SumK = 0;
	      for (int i=1; i<=DofNumber_; i++)
		SumK += uh[number(element[num_element-1][i])] * Basis[i];
	      Fxkl = Fk (xk, xl, num_element);
	      LocalError += omegak * omegal * (1-xk) * pow (u (Fxkl[1], Fxkl[2]) - SumK, 2);
	    }
	}
      GlobalError += element[num_element-1].AbsDet() * LocalError;
    }

  return sqrt (GlobalError);
}


/*!
  \brief Computes the semi-H1 error between u (exact function) and u_h (discrete vector) \n \n
  \param dudx the derivative in x of the exact function   \n
  \param dudy the derivative in y of the exact function   \n
  \param uh the discrete solution associated to p1c2d discretization  \n
  \param NbIntPts the number of Gauss points to do the numerical integration \n
  \return \f$ |u-u_h|_{1,\Omega}\f$
  \n\n
*/

//=================================================================================================
template <class T>
T p1c2d<T>::error_semi_H1 (T (*dudx)(T,T), T (*dudy)(T,T), const vector<T>& uh, int NbIntPts) const
//=================================================================================================
{
  T GlobalError = 0;
  T LocalError;
  T omegak, omegal, xk, xl;

  T duhdx, duhdy;
  T duhdxhat, duhdyhat;
  vector<T> Fxkl;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  matrix<T> DiffBasis (2, DofNumber_);
  matrix<T> InvMk;

  for (int num_element=1; num_element<=nbelement_; num_element++)
    { LocalError = 0;
      InvMk = element[num_element-1].InvMk();
      for (int k=1; k<=itg.NbIntPts(); k++)
	{ omegak = itg.weight (k);
	  xk = itg.point (k);
	  for (int l=1; l<=itg.NbIntPts(); l++)
	    { omegal = itg.weight (l);
	      xl = (1 - xk) * itg.point (l);
	      DiffBasis = DiffBasisFunction (xk, xl);
	      duhdxhat = 0;
	      duhdyhat = 0;
	      for (int i=1; i<=DofNumber_; i++)
		{ duhdxhat += uh[number(element[num_element-1][i])] * DiffBasis (1,i);
		  duhdyhat += uh[number(element[num_element-1][i])] * DiffBasis (2,i);
		}
	      duhdx = duhdxhat * InvMk (1,1) + duhdyhat * InvMk (2,1);
	      duhdy = duhdxhat * InvMk (1,2) + duhdyhat * InvMk (2,2);
	      Fxkl = Fk (xk, xl, num_element);
	      LocalError += omegak * omegal * (1 - xk) * (pow (dudx (Fxkl[1], Fxkl[2]) - duhdx, 2) + pow (dudy (Fxkl[1], Fxkl[2]) - duhdy, 2));
	    }
	}
      GlobalError += LocalError * element[num_element-1].AbsDet();
    }

  return sqrt (GlobalError);
}


//======================================================================================================================
template <class T>
T p1c2d<T>::error_semi_H1_loc2 (T (*dudx)(T,T),T (*dudy)(T,T), const vector<T>& uh, int NbIntPts, int num_element) const
//======================================================================================================================
{
  T GlobalError = 0;
  T LocalError = 0;
  T omegak, omegal, xk, xl;

  T duhdx, duhdy;
  T duhdxhat, duhdyhat;
  vector<T> Fxkl;

  integration<T> itg;
  itg.Gauss (NbIntPts, 0, 1);

  matrix<T> DiffBasis (2, DofNumber_);
  matrix<T> InvMk;

  InvMk = element[num_element-1].InvMk();

  for (int k=1; k<=itg.NbIntPts(); k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=itg.NbIntPts(); l++)
	{ omegal = itg.weight (l);
          xl = (1 - xk) * itg.point (l);
	  DiffBasis = DiffBasisFunction (xk, xl);
	  duhdxhat = 0;
	  duhdyhat = 0;
	  for (int i=1; i<=DofNumber_; i++)
	    { duhdxhat += uh[number(element[num_element-1][i])] * DiffBasis (1,i);
	      duhdyhat += uh[number(element[num_element-1][i])] * DiffBasis (2,i);
	    }
	  duhdx = duhdxhat * InvMk (1,1) + duhdyhat * InvMk (2,1);
	  duhdy = duhdxhat * InvMk (1,2) + duhdyhat * InvMk (2,2);
	  Fxkl = Fk (xk, xl, num_element);
	  LocalError += omegak * omegal * (1 - xk) * (pow (dudx (Fxkl[1], Fxkl[2]) - duhdx, 2) + pow (dudy (Fxkl[1], Fxkl[2]) - duhdy, 2));
	}
    }

  return LocalError * element[num_element-1].AbsDet();
}


//==========================================================================
template <class T>
T p1c2d<T>::Grad_Normal_Jump_L2 (const vector<T>& uh, int num_element) const
//==========================================================================
{
  T SumJumps = 0;
  T LocalJump;
  T duhdx1, duhdy1, duhdx2, duhdy2;
  T duhdxhat1, duhdyhat1, duhdxhat2, duhdyhat2;
  vector<T> normal (2);
  matrix<T> DiffBasis = DiffBasisFunction ();
  matrix<T> InvMk1, InvMk2;
  int jseg, jelement;

  InvMk1 = element[num_element-1].InvMk();
  duhdxhat1 = 0;
  duhdyhat1 = 0;
  for (int i=1; i<=DofNumber_; i++)
    { duhdxhat1 += uh[number(element[num_element-1][i])] * DiffBasis (1,i);
      duhdyhat1 += uh[number(element[num_element-1][i])] * DiffBasis (2,i);
    }
  duhdx1 = duhdxhat1 * InvMk1 (1,1) + duhdyhat1 * InvMk1 (2,1);
  duhdy1 = duhdxhat1 * InvMk1 (1,2) + duhdyhat1 * InvMk1 (2,2);

  for (int edge=1; edge<=3; edge++)
    { jseg = EdgesPerTriangle_ (num_element, edge);
      normal = NormalUnitVector_[jseg];
      if (!Type_of_edge_[jseg]) // segment interne
	{ jelement = TrianglesPerEdge_ (jseg, 1);
	  if (jelement == num_element) jelement = TrianglesPerEdge_ (jseg, 2);
	  InvMk2 = element[jelement-1].InvMk();
	  duhdxhat2 = 0;
	  duhdyhat2 = 0;
	  for (int i=1; i<=DofNumber_; i++)
	    { duhdxhat2 += uh[number(element[jelement-1][i])] * DiffBasis (1,i);
	      duhdyhat2 += uh[number(element[jelement-1][i])] * DiffBasis (2,i);
	    }
	  duhdx2 = duhdxhat2 * InvMk2 (1,1) + duhdyhat2 * InvMk2 (2,1);
	  duhdy2 = duhdxhat2 * InvMk2 (1,2) + duhdyhat2 * InvMk2 (2,2);
	  LocalJump = pow (NormEdge_[jseg], 1.5) * abs (normal[1] * (duhdx2 - duhdx1) + normal[2] * (duhdy2 - duhdy1));
	}
      else // segment frontalier
	LocalJump = powf (NormEdge_[jseg], 1.5) * abs (normal[1] * duhdx1 + normal[2] * duhdy1);

      SumJumps += pow (LocalJump, 2);
    }

  return sqrt (SumJumps);
}


//====================================================================
template <class T>
T p1c2d<T>::Laplacian_Residual_L2 (T (*f)(T,T), int num_element) const
//====================================================================
{
  int NbPtsInt = 2;
  T omegak, omegal, xk, xl;
  vector<T> Fxkl;
  integration<T> itg;
  itg.Gauss (NbPtsInt, 0, 1);
  T LocalResidu = 0;

  for (int k=1; k<=NbPtsInt; k++)
    { omegak = itg.weight (k);
      xk = itg.point (k);
      for (int l=1; l<=NbPtsInt; l++)
	{ omegal = itg.weight (l);
          xl = (1 - xk) * itg.point (l);
	  Fxkl = Fk (xk, xl, num_element);
	  LocalResidu += omegak * omegal * (1 - xk) * f (Fxkl[1], Fxkl[2]);
	}
    }

  return abs (LocalResidu) * element[num_element-1].diam() * sqrt (2 * element[num_element-1].AbsDet());
}


//=============================================
template <class T>
void p1c2d<T>::refine_FE (vector<int> & TABTRI)
//=============================================
{
  (*this).refine_mesh (TABTRI);
  GlobalDofNumber_ = nbvertex_;
}


//============================================================
template <class T>
void p1c2d<T>::refine_FE_angle (vector<int> & TABTRI, T angle)
//============================================================
{
  (*this).refine_mesh_angle (TABTRI, angle);
  GlobalDofNumber_ = nbvertex_;
}


//========================================================
template <class T>
T p1c2d<T>::RT_L2Norm (T (*f)(T,T), int num_element) const
//========================================================
{
  return (*this).Laplacian_Residual_L2 (f, num_element) / element[num_element-1].diam();
}


}


#endif
