/***************************************************************
 *                 Finite Element Method Object Library        *
 *           class convert : declaration for convertion tools  *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 0.2.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2003,2004,2005,2006 CREUSE Emmanuel
 * copyright  2003 MERCIER Denis
 * copyright  Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class convert
    \brief convert library \n

    \htmlonly 
    <FONT color="#838383">

    convert belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly 2003, 2004, 2005, 2006 Emmanuel CREUSE \n
	     copyright \htmlonly &#169; \endhtmlonly 2003 Denis MERCIER \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.2.0
    \date 2003-2006
    \bug none
    \warning none
*/


#ifndef __cplusplus
#error Must use C++ for the type node
#endif

#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__VECTORS_H)
#include "../../MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "../../MOL++/matrix.h"
#endif

#ifndef _convert_h
#define _convert_h


using namespace std;


//===========
class convert
//===========
{
  public :
    friend void make_convertion3dtetra (const char*,const char*); // to convert 3D tetrahedra meshes from NETGEN
    friend void make_convertion2dtri (const char*,const char*,const char*); // to convert 2D triangles meshes from NETGEN
    friend int  lshape_boundary2d(double,double,double,double,double,double); // return yes if the specified node belongs to the boundary
    friend int  circle_boundary2d(double,double,double,double,double); // return yes if the specified node belongs to the boundary
    friend int  threeonfourpartofcircle_boundary2d(double,double,double); // return yes if the specified node belongs to the boundary
    friend int  square_boundary2d(double,double,double); // return yes if the specified node belongs to the boundary
    friend int  halfdisk_boundary2d(double,double); // return yes if the specified node belongs to the boundary
    friend int  fourthdisk_boundary2d(double,double); // return yes if the specified node belongs to the boundary
};


//=====Public  methods for convert=============================================

/*!
  \brief Convert an output file generated by Netgen (Mesh made of tetrahedra) in an
  input file for simula+
  \n \n
  \param name of the input_file file generated by Netgen
  \param name of the output_file to be read by the constructor in Simula+
  \warning Before January, 4 2006, the name of this function was only "make_convertion"
  */
//--------------------------------------------------------------------
void make_convertion3dtetra (const char* input_file, const char* output_file)
//--------------------------------------------------------------------
{
  // NETGEN file opening 
  ifstream in_file(input_file);
  assert (!in_file.fail());
  // opening of the output file : if it does not exit, it is created. Else, it is overwritten.
  ofstream out_file(output_file,ios::trunc);
  assert (!out_file.fail());

  int number_nodes, number_tetra,number_faces;

  in_file >>number_nodes;
  matrix <double> coordinates(number_nodes,3);
  vector <int> frontier(number_nodes);   
  for (int i=1; i<=number_nodes; i++)
  for (int j=1; j<=3; j++)
      in_file>>coordinates(i,j);

  in_file >>number_tetra;  
  matrix<int> tetra_nodes(number_tetra,4);
  int x;
  for (int i=1; i<=number_tetra; i++)
  { in_file>>x;
    for (int j=1; j<=4; j++)
      in_file>>tetra_nodes(i,j);
  }

  in_file >>number_faces;
  for (int i=1; i<= number_faces; i++)
  {in_file>>x;
   for (int j=1; j<=3; j++)
     {  in_file>>x;
        frontier[x]=1;
      };
};

in_file.close() ;

//writing in FORMAT3D
out_file<<3<<endl;
out_file<<number_nodes<<endl;

out_file.precision(12);
//out_file.setf(ios::scientific);
for (int i=1; i<= number_nodes; i++)
{for (int j=1; j<= 3; j++)
{out_file<<coordinates(i,j) << "   "; 
};
out_file<<frontier[i]<<endl; 
};

out_file<<number_tetra<<endl;

for (int i=1; i<= number_tetra; i++)
{for (int j=1; j<=4; j++)
{out_file<<setw(4)<<tetra_nodes(i,j) << "   "; 
};
out_file<<endl; 
};

}
/*!
  \brief Convert an output file generated by Netgen (Mesh made of triangles) in an
  input file for simula+
  \n \n
  \param name of the input_file file generated by Netgen \n
  \param name of the output_file to be read by the constructor in Simula+ \n
  \param domain : The type of domain, to characterize the boundary nodes, which are not given by NETGEN \n \n
  \f$ \textit{domain can bee : "square", "lshape2d", "circle", "threeonfourpartofcircle"} \f$  \n \n
  \warning Pour certaines fonctions permettant de caractriser les noeuds sur une frontire curviligne 
           (unitcircle,partofunitcircle), il faut prendre garde  ce que le fichier de sortie Netgen, gnr
	   lui mme  partir d'un fichier .in2d,  NE POSSEDE PAS D'AUTRES NOEUDS SUR LA FRONTIERE QUE CEUX
	   MENTIONNES DANS LE FICHIER .in2d LUI MEME. Sinon, ces fonctions ne caractriseront pas les noeuds d'une
	   telle frontire qui seraient situs du coup sur la corde et non sur l'arc... Remarque : Tous les maillages
	   fournis dans les rpertoire du code simula+ respectent cette proprit !
	   
	   
  */
//---------------------------------------------------------------------------------------------
void make_convertion2dtri (const char* input_file, const char* output_file, const char* domain)
//---------------------------------------------------------------------------------------------
{
  // NETGEN file opening 
  ifstream in_file(input_file);
  assert (!in_file.fail());
  ofstream out_file(output_file,ios::trunc);
  assert (!out_file.fail());
  int number_nodes, number_tri;
  int toto;
  double x,y;
  double titi;
  
  
  in_file >> number_nodes;
  matrix <double> coordinates(number_nodes,2);   
  vector<int> boundary(number_nodes);
  double xleft=-1.0, xright=1.0, ybottom=-1.0, ytop=1.0;
  double centerx=0.0, centery=0.0, radius=1.0;
  
  if (domain=="lshape2d")
   {
  for (int i=1; i<=number_nodes; i++)
    { 
      in_file>>x;
      in_file>>y;
      in_file>>titi;
      coordinates(i,1)=x;
      coordinates(i,2)=y;
      if (lshape_boundary2d(x,y,xleft,xright,ybottom,ytop)) boundary[i]=1;
    }
    }
   else
   if (domain=="circle")
     {
     for (int i=1; i<=number_nodes; i++)
    { 
      in_file>>x;
      in_file>>y;
      in_file>>titi;
      coordinates(i,1)=x;
      coordinates(i,2)=y;
      if (circle_boundary2d(x,y,centerx,centery,radius)) boundary[i]=1;
    }
     }     
   else
   if (domain=="threeonfourpartofcircle")
     {
     for (int i=1; i<=number_nodes; i++)
    { 
      in_file>>x;
      in_file>>y;
      in_file>>titi;
      coordinates(i,1)=x;
      coordinates(i,2)=y;
      if (threeonfourpartofcircle_boundary2d(x,y,radius)) boundary[i]=1;
    }
     }     
   else
   if (domain=="square")
     {
     double halfside=0.5;
     for (int i=1; i<=number_nodes; i++)
    { 
      in_file>>x;
      in_file>>y;
      in_file>>titi;
      coordinates(i,1)=x;
      coordinates(i,2)=y;
      if (square_boundary2d(x,y,halfside)) boundary[i]=1;
    }
     }
   else
      if (domain=="square1") // to generate the square (0,0)-(1,0)-(1,1)-(0,1)      
     {
     double halfside=0.5;
     for (int i=1; i<=number_nodes; i++)
    { 
      in_file>>x;
      in_file>>y;
      in_file>>titi;
      coordinates(i,1)=x+0.5;
      coordinates(i,2)=y+0.5;
      if (square_boundary2d(x,y,halfside)) boundary[i]=1;
    }
     }
     else
      if (domain=="square2") // to generate the square (1,0)-(2,0)-(2,1)-(1,1)
     {
     double halfside=0.5;
     for (int i=1; i<=number_nodes; i++)
    { 
      in_file>>x;
      in_file>>y;
      in_file>>titi;
      coordinates(i,1)=(x+1.5);
      coordinates(i,2)=(y+0.5);
      if (square_boundary2d(x,y,halfside)) boundary[i]=1;
    }
     }
      else
      if (domain=="square3") // to generate the square (0,0)-(2,0)-(2,2)-(0,2)
     {
     double halfside=0.5;
     for (int i=1; i<=number_nodes; i++)
    { 
      in_file>>x;
      in_file>>y;
      in_file>>titi;
      coordinates(i,1)=x+0.5;
      coordinates(i,2)=y+0.5;
      if (square_boundary2d(x,y,halfside)) boundary[i]=1;
      coordinates(i,1)*=2.0;
      coordinates(i,2)*=2.0;
    }
     }
     
     else
      if (domain=="square4") // to generate the square (2,0)-(3,0)-(3,1)-(2,1)
     {
     double halfside=0.5;
     for (int i=1; i<=number_nodes; i++)
    { 
      in_file>>x;
      in_file>>y;
      in_file>>titi;
      coordinates(i,1)=x+0.5;
      coordinates(i,2)=y+0.5;
      if (square_boundary2d(x,y,halfside)) boundary[i]=1;
      coordinates(i,1)+=2.0;
    }
     }
     
     else
      if (domain=="halfdisk") // to generate the left unit half disk
     {
     for (int i=1; i<=number_nodes; i++)
    { 
      in_file>>x;
      in_file>>y;
      in_file>>titi;
      coordinates(i,1)=x;
      coordinates(i,2)=y;
      if (halfdisk_boundary2d(x,y)) boundary[i]=1;
    }
     }
     
     else
      if (domain=="fourthdisk") // to generate the right top unit disk
     {
     for (int i=1; i<=number_nodes; i++)
    { 
      in_file>>x;
      in_file>>y;
      in_file>>titi;
      coordinates(i,1)=x;
      coordinates(i,2)=y;
      if (fourthdisk_boundary2d(x,y)) boundary[i]=1;
    }
     }
     
   else
   {
    cout << "The domain is up to now not treated in make_convertion2dtri (see convert.h)" << "\n";
    assert(0==1);
   } 
  in_file >> toto;
  in_file >>number_tri;
  matrix<int> tri_nodes(number_tri,3);
  int m;
  for (int i=1; i<=number_tri; i++)
  { in_file>>m;
    for (int j=1; j<=3; j++)
      in_file>>tri_nodes(i,j);
  }

in_file.close() ;


//writing in FORMAT3D
out_file<<2<<endl;
out_file<<number_nodes<<endl;

out_file.precision(12);
//out_file.setf(ios::scientific);
for (int i=1; i<= number_nodes; i++)
{for (int j=1; j<= 2; j++)
{out_file<<coordinates(i,j) << "   "; 
};
out_file<<boundary[i]<<endl; 
};

out_file<<number_tri<<endl;

for (int i=1; i<= number_tri; i++)
{for (int j=1; j<=3; j++)
{out_file<<setw(4)<<tri_nodes(i,j) << "   "; 
};
out_file<<endl; 
};

}
/*!
  \brief Characterizes a node located on the boundary of an L-Shape
  \n \n
  \param x x-coordinate of the node \n
  \param y y-coordinate of the node \n
  \param xleft x-coordinate of the left side of the L-Shape \n
  \param xright x-coordinate of the right side of the L-Shape \n 
  \param ybottom y-coordinate of the bottom side of the L-Shape \n 
  \param ytop y-coordinate of the top side of the L-Shape \n
  \return rep 1 if it is on the boundary, 0 else.
  */
//--------------------------------------
int lshape_boundary2d(double x,double y, double xleft, double xright, double ybottom, double ytop)
//--------------------------------------
{
double epsilon=1e-06;

int rep;
rep=(
      (abs(x-xleft)<epsilon)
    ||(abs(x-xright)<epsilon)
    ||(abs(y-ytop)<epsilon)
    ||(abs(y-ybottom)<epsilon)
    ||((abs(x)<epsilon)&&(y<epsilon))
    ||((abs(y)<epsilon)&&(x>epsilon))
      );
return rep;
}

/*!
  \brief Characterizes a node located on the boundary of a disk
  \n \n
  \param x x-coordinate of the node \n
  \param y y-coordinate of the node \n
  \param centerx x-coordinate of the circle center  \n
  \param centery y-coordinate of the circle center  \n
  \param radius radius of the circle \n 
  \return rep 1 if it is on the boundary, 0 else.
  \warning Before January, 4 2006, the name of this function was  "unitcircle_boundary2d"
  \warning Pour cette fonction, il faut prendre garde  ce que le fichier de sortie Netgen, gnr
	   lui mme  partir d'un fichier .in2d,  NE POSSEDE PAS D'AUTRES NOEUDS SUR LA FRONTIERE QUE CEUX
	   MENTIONNES DANS LE FICHIER .in2d LUI MEME. Sinon, elle ne caractrisera pas tous les noeuds de la frontire
	   (cad ceux qui seraient situs du coup sur la corde et non sur l'arc...) Remarque : Tous les maillages
	   fournis dans les rpertoire du code simula+ respectent cette proprit !
  */
//--------------------------------------
int circle_boundary2d(double x,double y, double centerx, double centery, double radius)
//--------------------------------------
{
double epsilon=1e-06;
int rep;
rep=(abs(sqrt(pow((x-centerx),2)+pow((y-centery),2))-radius)<epsilon);
return rep;
}
/*!
  \brief Characterizes a node located on the boundary of a 3/4 part of a disk which center is (0,0)
  \n \n
  \param x x-coordinate of the node \n
  \param y y-coordinate of the node \n
  \param radius radius of the 3/4 circle \n 
  \return rep 1 if it is on the boundary, 0 else.
  \warning Before January, 4 2006, the name of this function was  "partofunitcircle_boundary2d"
  \warning Pour cette fonction, il faut prendre garde  ce que le fichier de sortie Netgen, gnr
	   lui mme  partir d'un fichier .in2d,  NE POSSEDE PAS D'AUTRES NOEUDS SUR LA FRONTIERE QUE CEUX
	   MENTIONNES DANS LE FICHIER .in2d LUI MEME. Sinon, elle ne caractrisera pas tous les noeuds de la frontire
	   (cad ceux qui seraient situs du coup sur la corde et non sur l'arc...) Remarque : Tous les maillages
	   fournis dans les rpertoire du code simula+ respectent cette proprit !
  */
//--------------------------------------
int threeonfourpartofcircle_boundary2d(double x,double y, double radius)
//--------------------------------------
{
double epsilon=1e-06;
int rep;
rep=((abs(sqrt(pow(x,2)+pow(y,2))-radius)<epsilon)||((abs(x)<epsilon)&&(y<0.0))||((abs(y)<epsilon)&&(x>0.0)));
return rep;
}
/*!
  \brief Characterizes a node located on the boundary of a square [-a,a]*[-a,a]
  \n \n
  \param x x-coordinate of the node \n
  \param y y-coordinate of the node \n
  \param a half side of the square \n
  \return rep 1 if it is on the boundary, 0 else.
  */
//--------------------------------------
int square_boundary2d(double x,double y, double a)
//--------------------------------------
{
double epsilon=1e-06;
int rep;
rep=((abs((abs(x)-a))<epsilon)||((abs(abs(y)-a))<epsilon));
return rep;
}

//--------------------------------------
int halfdisk_boundary2d(double x,double y)
//--------------------------------------
{
double epsilon=1e-06;
int rep;
rep=((abs(x)<epsilon)||(abs(sqrt(pow(x,2)+pow(y,2))-1.0)<epsilon));
return rep;
}

//--------------------------------------
int fourthdisk_boundary2d(double x,double y)
//--------------------------------------
{
double epsilon=1e-06;
int rep;
rep=((abs(x)<epsilon)||(abs(y)<epsilon)||(abs((sqrt(pow(x,2)+pow(y,2))-1.0))<epsilon));
return rep;
}

#endif

