/***************************************************************
 *                 Finite Element Method Object Library        *
 *           class q12d : declaration for q12d               *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 0.1.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2004,2006 CREUSE Emmanuel
 * copyright  2004 KALETA Olivia
 * copyright  2004 SOUALEM Nadir
 * copyright  Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class q12d
    \brief q12d library \n

    \htmlonly 
    <FONT color="#838383">

    q12d belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2004,2006 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly  2004 KALETA Olivia \n
	     copyright \htmlonly &#169; \endhtmlonly  2004 SOUALEM Nadir \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.1.0
    \date 2004,2006
    \bug none
    \warning none
*/


#ifndef __cplusplus
#error Must use C++ for the type q1_2d
#endif

#ifndef _q12d_h
#define _q12d_h

#if !defined(__VECTORS_H)
#include "../../MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "../../MOL++/matrix.h"
#endif

#if !defined(__ASSERT_H)
#include<assert.h>
#endif

#if !defined(__MESH_H)
#include "../meshes/mesh.h"
#endif

#if !defined(__INTEGRATION_H)
#include "../../MOL++/integration.h"
#endif 

#if !defined(__MESH_H)
#include "../meshes/mesh.h"
#endif

using namespace std;

//-----------------------------------------------------------------------------------------------
template <class T> class q12d:public mesh <rectangle<T>,T>
//-----------------------------------------------------------------------------------------------
{

using mesh<rectangle<T>,T>::alive;
using mesh<rectangle<T>,T>::dimspace_;
using mesh<rectangle<T>,T>::nbvertex_;
using mesh<rectangle<T>,T>::nbelement_;
using mesh<rectangle<T>,T>::vertices;
using mesh<rectangle<T>,T>::element;

private :
int DofNumber_;
int GlobalDofNumber_;
matrix<T> RefMassMatrix_;
matrix<T> DUDV_;
vector<T> B_;
vector<T> BuildBloc(T (*f)(T,T),int,integration<T>) const;
public :
q12d();
q12d(const char*,const char*);
q12d(const char*);
q12d(const q12d<T>&);
q12d<T>& operator=(const q12d<T>&);//q12d affectation
/*!
\return the number of node of the object
*/
int DofNumber()const{return (DofNumber_);};
/*!
\return the number of node of meshes
*/
int GlobalDofNumber() const {return (GlobalDofNumber_);};
/*!
\return Reference Mass Matrix
*/
matrix<T> RefMassMatrix()const {return (RefMassMatrix_);};
vector<T> BasisFunction(T,T)const;
matrix<T> DiffBasisFunction(T xref=0, T yref=0)const;
virtual int operator ()(int,int)const;
template <class FT> friend int operator ==  (const q12d<FT>&,const q12d<FT>&);
template <class FT> friend int operator !=  (const q12d<FT>&,const q12d<FT>&);
matrix<T> LocalMassMatrix(int)const;
matrix<T> LocalStiffnessMatrix(int)const;
vector<T> Fk(T,T,int) const;
matrix<T> DUDV() const;
vector<T> B(T (*f)(T,T),int NbIntPts )const;
void Dirichlet(matrix<T>& MAT,vector<T>&F,T (*g)(T,T))const;
T error_L2 (T (*u)(T,T),const vector<T>& uh,int NbIntPts) const ;
T error_semi_H1(T (*dudx)(T,T),T (*dudy)(T,T),const vector<T>& uh,int NbIntPts) const;
};


/*!
\brief default constructor of the q12d class
*/
//------------------------------------------------------------------------------------------------
template <class  T>
q12d <T>::q12d() :mesh <rectangle<T>,T> ()
//------------------------------------------------------------------------------------------------
{
DofNumber_=0;
GlobalDofNumber_=0;
}

/*!
\brief constructor of the q12d class. The arguments are 2 files
  file1 and file2. file1 must be documented by the number of elements, node 's numerotation
  of each element. file2 indicate the dimension (2 or 3), 
  number of nodes and each node coordinates with a boundary condition (1 if node is on the boundary, 0 else).
  \param file1 string
  \param file2 string
  */
//------------------------------------------------------------------------------------------------
template <class T>
q12d <T>::q12d(const char*file1,const char*file2) :mesh <rectangle<T>,T> (file1,file2)
//------------------------------------------------------------------------------------------------
{
DofNumber_=4;
GlobalDofNumber_=nbvertex_;
RefMassMatrix_=matrix<T>(4,4);

RefMassMatrix_(1,1)=1./9.;RefMassMatrix_(1,2)=1./18.;RefMassMatrix_(1,3)=1./36.;RefMassMatrix_(1,4)=1./18.;
RefMassMatrix_(2,1)=1./18.;RefMassMatrix_(2,2)=1./9.;RefMassMatrix_(2,3)=1./18.;RefMassMatrix_(2,4)=1./36.;
RefMassMatrix_(3,1)=1./36.;RefMassMatrix_(3,2)=1./18.;RefMassMatrix_(3,3)=1./9.;RefMassMatrix_(3,4)=1./18.;
RefMassMatrix_(4,1)=1./18.;RefMassMatrix_(4,2)=1./36.;RefMassMatrix_(4,3)=1./18.;RefMassMatrix_(4,4)=1./9.;
}


/*!
\brief constructor of the q12d class. It makes the same actions as the previous constructor, the 
  argument file is the concatenation of file1 and file2.
  \param file string
*/
//------------------------------------------------------------------------------------------------
template <class T>
q12d<T>::q12d(const char*file) :mesh<rectangle<T>,T>(file)
//-----------------------------------------------------------------------------------------------
{ 
DofNumber_=4;
GlobalDofNumber_=nbvertex_;
RefMassMatrix_=matrix<T>(4,4);

RefMassMatrix_(1,1)=1./9.;RefMassMatrix_(1,2)=1./18.;RefMassMatrix_(1,3)=1./36.;RefMassMatrix_(1,4)=1./18.;
RefMassMatrix_(2,1)=1./18.;RefMassMatrix_(2,2)=1./9.;RefMassMatrix_(2,3)=1./18.;RefMassMatrix_(2,4)=1./36.;
RefMassMatrix_(3,1)=1./36.;RefMassMatrix_(3,2)=1./18.;RefMassMatrix_(3,3)=1./9.;RefMassMatrix_(3,4)=1./18.;
RefMassMatrix_(4,1)=1./18.;RefMassMatrix_(4,2)=1./36.;RefMassMatrix_(4,3)=1./18.;RefMassMatrix_(4,4)=1./9.;
}

/*!
  \brief copy constructor of the q12d class
  \param EF  reference on q12d object
  */

//------------------------------------------------------------------------------------------------
template <class T>
q12d<T>::q12d(const q12d<T>& EF) :mesh<rectangle<T>,T>(EF)
//------------------------------------------------------------------------------------------------
{
DofNumber_=EF.DofNumber_;
GlobalDofNumber_=EF.GlobalDofNumber_;
RefMassMatrix_=EF.RefMassMatrix_;

}

/*!
  \brief affectation operator for the q12d class.
  \param EF reference on q12d object 
  \return  reference on q12d object
 */
//------------------------------------------------------------------------------------------------
template <class T>
q12d<T>& q12d<T>::operator =(const q12d<T>& EF)
//------------------------------------------------------------------------------------------------
{
(*this).mesh<rectangle<T>,T>::operator=(EF);
DofNumber_=EF.DofNumber_;
GlobalDofNumber_=EF.GlobalDofNumber_;
RefMassMatrix_=EF.RefMassMatrix_;

return(*this);
}
/*!
\return the vector with the basis function
\param xref type T
\param yref  type T
*/


//------------------------------------------------------------------------------------------------
template <class T>
vector<T> q12d<T>::BasisFunction(T xref,T yref)const
//-----------------------------------------------------------------------------------------------
{
vector<T> P(DofNumber_);
P[1]=1-xref-yref+(xref*yref);
P[2]=xref-(xref*yref);
P[3]=xref*yref;
P[4]=yref-(xref*yref);
return(P);
}



/*!
\return the matrix whith the derivate of the basis function
\param xref type T
\param yref  typeT
*/
//------------------------------------------------------------------------------------------------
template <class T>
matrix<T> q12d<T>::DiffBasisFunction(T xref, T yref)const
//------------------------------------------------------------------------------------------------
{
matrix<T> DP(2,DofNumber_);
DP(1,1)=-1+yref;DP(1,2)=1-yref;DP(1,3)=yref;DP(1,4)=-yref;
DP(2,1)=-1+xref;DP(2,2)=-xref;DP(2,3)=xref;DP(2,4)=1-xref;
return(DP);
}
//------------------------------------------------------------------------------------------------

/*!
   \param num_element element's number
   \param j the j-th dof 
   \return global number of the j-th of the element num_element
  */
template <class T>
int q12d<T>::operator ()(int num_element,int j) const
//------------------------------------------------------------------------------------------------
{
assert((j>=1)&&(j<=DofNumber_)&&(num_element>=1)&&(num_element<=nbelement_));
return number(element[num_element-1][j]);
}


/*!
   \param EF1 type const q12d<T>&
   \param EF2 type const q12d<T>&
   \return 1 if EF1 and  EF2 are equal, 0 else
  */
//---------------------------------------------------------------------------------------------
template <class FT>
int operator ==(const q12d<FT>& EF1,const q12d<FT>& EF2)
//------------------------------------------------------------------------------------------------
{
int boolean=1;
boolean*=(mesh<rectangle<FT>,FT>(EF1)==mesh<rectangle<FT>,FT>(EF2));
boolean*=(EF1.DofNumber_==EF2.DofNumber_);
boolean*=(EF1.GlobalDofNumber_==EF2.GlobalDofNumber_);
boolean*=(EF1.RefMassMatrix_==EF2.RefMassMatrix_);
return(boolean);
}


/*!
   \param EF1 type const q12d<T>&
   \param EF2 type const q12d<T>&
   \return 1 if EF1 and  EF2 are different, 0 else
  */
//----------------------------------------------------------------------------------------------
template <class FT>
int operator !=(const q12d<FT>& EF1,const q12d<FT>& EF2)
//-----------------------------------------------------------------------------------------------
{
return(!(EF1==EF2));
}


/*!
   \param num_element element's number
   \return local mass matrix of the element num_element
  */
//-----------------------------------------------
template <class T>
matrix<T> q12d<T>::LocalMassMatrix(int num_element) const
//----------------------------------------------
{
assert((num_element>=1)&&(num_element<=nbelement_));
return(RefMassMatrix_*element[num_element-1].AbsDet());

}

/*!
\param num_element type integer
\return local stiffnessmatrix of the element
*/
//-----------------------------------------------------------
template <class T> 
matrix<T> q12d<T>::LocalStiffnessMatrix(int num_element) const
//------------------------------------------------------------
{
vector<T> A(2),B(2);
T h,k;
A=((element[num_element-1][2])-(element[num_element-1][1]));
B=((element[num_element-1][3])-(element[num_element-1][2]));
h=A[1];
k=B[2];
matrix<T> StiffnessMatrix(4,4);

StiffnessMatrix(1,1)=StiffnessMatrix(2,2)=StiffnessMatrix(3,3)=StiffnessMatrix(4,4)=((h*h)+(k*k))/(3.*h*k);
StiffnessMatrix(1,2)=StiffnessMatrix(2,1)=StiffnessMatrix(3,4)=StiffnessMatrix(4,3)=(h/(6.*k))-(k/(3.*h));
StiffnessMatrix(1,3)=StiffnessMatrix(3,1)=StiffnessMatrix(2,4)=StiffnessMatrix(4,2)=-(((h*h)+(k*k))/(6.*h*k));
StiffnessMatrix(1,4)=StiffnessMatrix(4,1)=StiffnessMatrix(2,3)=StiffnessMatrix(3,2)=(k/(6.*h))-(h/(3.*k));
return StiffnessMatrix;
}
/*!
\param x type T
\param y  typeT
\param i type integer
\return vector
*/

//----------------------------------------------------------------------------------
template <class T>
vector<T> q12d<T>::Fk(T x,T y,int i) const
//------------------------------------------------------------------------------------------
{
matrix<T> MAT(2,2);
vector<T> VEC(2);
VEC[1]=x; VEC[2]=y;
vector<T> A1A2=element[i-1][2]-element[i-1][1];
vector<T> A2A3=element[i-1][3]-element[i-1][2];
MAT(1,1)=A1A2[1];MAT(1,2)=A2A3[1];
MAT(2,1)=A1A2[2];MAT(2,2)=A2A3[2];
return MAT*VEC+element[i-1][1];
}


/*!
   \return the global stifness matrix of the q12d element
  */
//-------------------------------------------------------------------------------------------------

template <class T>
matrix<T> q12d<T>::DUDV() const
//------------------------------------------------------------------------------------------------
{
matrix<T> Stiff(DofNumber_,DofNumber_);
matrix<T> DUDV_(GlobalDofNumber_,GlobalDofNumber_);

for (int l=1;l<=nbelement_;l++)
   {
    Stiff=LocalStiffnessMatrix(l);
    
        for (int i=1;i<=DofNumber_;i++)
        for (int j=1;j<=DofNumber_;j++)
           DUDV_((*this)(l,i),(*this)(l,j))+=Stiff(i,j);
    }
return DUDV_;
}
/*!
  \brief Value of the contribution to the right-hand side of the variational formulation of a given element \n \n
  \param \f$ f \f$ the function   \n
  \param num_element the element on which we want to integrate \n
  \param itg the Gauss points and weight used for the numerical integration
  \return vector \f$(\int_{num_{element}} f(x,y) \varphi_1(x,y) dx dy,\int_{num_{element}} f(x,y) \varphi_2(x,y) dx dy,\int_{num_{element}} f(x,y) \varphi_3(x,y) dx dy,\int_{num_{element}} f(x,y) \varphi_4(x,y) dx dy )\f$ 
   where \f$(\varphi_i)_{1\leq i\leq 4}\f$ is the i-th basis function associated to the element num_element.
  \n\n
*/
//------------------------------------------------------------------------------------------------
template <class T>
vector<T> q12d<T>::BuildBloc(T (*f)(T,T),int num_element,integration<T> itg) const
//-----------------------------------------------------------------------------------------------
{

vector<T> res(DofNumber_);
vector<T> Basis(DofNumber_);
T omegak, omegal, xk, xl;
vector<T> Fxkl;

   for (int k=1; k<=itg.NbIntPts(); k++)


    {
      omegak=itg.weight(k);
      xk=itg.point(k);

      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=itg.point(l);
	  Basis=BasisFunction(xk,xl);
	  Fxkl=Fk(xk,xl,num_element);
          for (int i=1; i<=DofNumber_; i++)
	    res[i] += omegak*omegal*f(Fxkl[1],Fxkl[2])*Basis[i];

	 }

    }
 res*=element[num_element-1].AbsDet();

return(res);
}
/*!
  \brief Value of the right-hand side of the variational formulation \n \n
  \param \f$ f \f$ the function   \n
  \param NbIntPts the number of Gauss points to do the numerical integration \n
  \return vector v with v_i=\f$(\int_{\Omega} f(x,y) \varphi_i(x,y) dx dy \f$ 
   where \f$(\varphi_i)_{1\leq i\leq nbvertex}\f$ is the i-th global basis function.
  \n\n
*/
//------------------------------------------------------------------------------------------------
template <class T>
vector<T> q12d<T>::B(T (*f)(T,T),int NbIntPts ) const
//-----------------------------------------------------------------------------------------------
{
assert (NbIntPts>0);
integration<T> itg;
itg.Gauss(NbIntPts,0.0,1.0);

vector<T> B_(GlobalDofNumber_);

vector<T> Bloc(DofNumber_);


for(int k=1;k<=nbelement_;k++)
	{
	Bloc=BuildBloc(f,k,itg);
		for(int i=1;i<=DofNumber_;i++)
		B_[(*this)(k,i)] += Bloc[i];
	}
return B_;
}
/*!
\param MAT type matrix<T>
\param F type vector<T>
\param g type T
*/


/*!
  \brief Modifies the linear system to put non homogeneous Dirichlet BC \n \n
  \param MAT the matrix of the linear system   \n
  \param F the Right-Hand-Side of the linear system \n
  \param g the function to find the value of the solution on the boundary
  \return MAT and F are modified
  \n\n
*/
//-----------------------------------------------------------------------------------------------
template <class T>
void q12d<T>::Dirichlet(matrix<T>& MAT,vector<T>&F,T (*g)(T,T)) const
//--------------------------------------------------------------------------------------------
{
for(int i=1;i<=GlobalDofNumber_;i++)
{
if(mesh<rectangle<T>,T>::operator()(i).boundary_condition())
	{
F[i]=(*g)(vertices[i-1][1],vertices[i-1][2]);
for(int j=1;j<=GlobalDofNumber_;j++)
	MAT(i,j)=0;
	MAT(i,i)=1;
	for(int k=1;k<=GlobalDofNumber_;k++)
	if(k!=i)
	{
	F[k]+=-MAT(k,i)*F[i];
	MAT(k,i)=0;
	}


	}
}
}



/*!
  \brief Computes the L2 error between u (exact function) and u_h (discrete vector) \n \n
  \param u the exact function   \n
  \param uh the discrete solution associated to q12d discretization  \n
  \param NbIntPts the number of Gauss points to do the numerical integration \n
  \return \f$ ||u-u_h||_{L^2(\Omega)}\f$
  \n\n
*/
//-------------------------------------------------------------------------------------------
template <class T>
T q12d<T>::error_L2 (T (*u)(T,T),const vector<T>& uh,int NbIntPts) const
//--------------------------------------------------------------------------------------------
{
T GlobalError=0;
T LocalError;
T omegak, omegal, xk, xl,SumK;
vector<T> Fxkl;

integration<T> itg;
itg.Gauss(NbIntPts,0.0,1.0);

vector<T> Basis(DofNumber_);

for(int num_element=1;num_element<=nbelement_;num_element++)
{

LocalError=0;


   for (int k=1; k<=itg.NbIntPts(); k++) 


    {
      omegak=itg.weight(k);
      xk=itg.point(k);

      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=itg.point(l);
	  Basis=BasisFunction(xk,xl);
          SumK=0.0;
	  for (int i=1; i<=DofNumber_; i++)
	  {SumK+=uh[(*this)(num_element,i)]*Basis[i];}
	  Fxkl=Fk(xk,xl,num_element);
	  LocalError += omegak*omegal*
          pow((u(Fxkl[1],Fxkl[2])-SumK),2); 
	 }

    }

GlobalError+=element[num_element-1].AbsDet()*LocalError;
}
return(sqrt(GlobalError));
}

/*!
  \brief Computes the semi-H1 error between u (exact function) and u_h (discrete vector) \n \n
  \param dudx the derivative in x of the exact function   \n
  \param dudy the derivative in y of the exact function   \n
  \param uh the discrete solution associated to q12d discretization  \n
  \param NbIntPts the number of Gauss points to do the numerical integration \n
  \return \f$ |u-u_h|_{1,\Omega}\f$
  \n\n
*/

//======================================================
template <class T>
T q12d<T>::error_semi_H1 (T (*dudx)(T,T),T (*dudy)(T,T),const vector<T>& uh,int NbIntPts) const
//======================================================
{
T GlobalError=0;
T LocalError;
T omegak, omegal, xk, xl;

T duhdx,duhdy;
T duhdxhat,duhdyhat;
vector<T> Fxkl;

integration<T> itg;
itg.Gauss(NbIntPts,0.0,1.0);

matrix<T> DiffBasis(2,DofNumber_);
matrix<T> InvMk;
for(int num_element=1;num_element<=nbelement_;num_element++)
{

LocalError=0;
InvMk=element[num_element-1].InvMk();

   for (int k=1; k<=itg.NbIntPts(); k++)
   

    {
      omegak=itg.weight(k);
      xk=itg.point(k);

      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=itg.point(l);
	  DiffBasis=DiffBasisFunction(xk,xl);

	  
	  duhdxhat=0.0;
	  duhdyhat=0.0;

	  for (int i=1; i<=DofNumber_; i++)
	  {
	  duhdxhat+=uh[(*this)(num_element,i)]*DiffBasis(1,i);
	  duhdyhat+=uh[(*this)(num_element,i)]*DiffBasis(2,i);
	  }

	  duhdx=duhdxhat*InvMk(1,1)+duhdyhat*InvMk(2,1);
	  duhdy=duhdxhat*InvMk(1,2)+duhdyhat*InvMk(2,2);

	  Fxkl=Fk(xk,xl,num_element);
	  LocalError += omegak*omegal*
          (pow((dudx(Fxkl[1],Fxkl[2])-duhdx),2)+
	   pow((dudy(Fxkl[1],Fxkl[2])-duhdy),2));
	 }
    }

GlobalError+=LocalError*element[num_element-1].AbsDet();
}
return(sqrt(GlobalError));
}






#endif












