/***************************************************************
 *                 Finite Element Method Object Library        *
 *           class p1nc2d : declaration for p1nc2d               *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 0.2.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2005,2006 CREUSE Emmanuel
 * copyright  2005,2006 SOUALEM Nadir
 * copyright  Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class p1nc2d
    \brief p1nc2d library \n

    \htmlonly 
    <FONT color="#838383">

    p1nc2d belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2005,2006 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly  2005,2006 SOUALEM Nadir \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.2.0
    \date 2005-2006
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type p1nc2d
#endif

#ifndef _p1nc2d_h
#define _p1nc2d_h

#if !defined(__VECTORS_H)
#include "../../MOL++/vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "../../MOL++/matrix.h"
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MESH_TRI_2D_H)
#include "../meshes/mesh_tri_2d.h"
#endif

#if !defined(__INTEGRATION_H)
#include "../../MOL++/integration.h"
#endif

using namespace std;

//=============================
template <class T> class p1nc2d:public mesh_tri_2d<T>
//=============================

{

  using mesh<triangle<T>,T>::alive;
  using mesh<triangle<T>,T>::dimspace_;
  using mesh<triangle<T>,T>::nbvertex_;
  using mesh<triangle<T>,T>::nbelement_;
  using mesh<triangle<T>,T>::vertices;
  using mesh<triangle<T>,T>::element;

  using mesh_tri_2d<T>::nbsegments_;
  using mesh_tri_2d<T>::SetOfEdges_;
  using mesh_tri_2d<T>::Type_of_edge_;
  using mesh_tri_2d<T>::EdgesPerTriangle_;
  using mesh_tri_2d<T>::NbTrianglesPerEdge_;
  using mesh_tri_2d<T>::TrianglesPerEdge_;
  using mesh_tri_2d<T>::TrianglesPerTriangle_;
  using mesh_tri_2d<T>::NumberOfNeighboorOfTriangles_;
  using mesh_tri_2d<T>::NumLocNodeInTriangle_;
  using mesh_tri_2d<T>:: NormalDirection_;
  using mesh_tri_2d<T>::hminT_;
  using mesh_tri_2d<T>::hE_;
  using mesh_tri_2d<T>::hminE_;
  using mesh_tri_2d<T>::NormEdge_;
  using mesh_tri_2d<T>::NormalUnitVector_;
  using mesh_tri_2d<T>::NbMaxSegmentsPerNode_;
  using mesh_tri_2d<T>::NbSegmentsPerNode_;
 
 private :
  int DofNumber_;
  int GlobalDofNumber_;
  matrix<T> RefMassMatrix_;
  vector<T> BuildBloc(T (*f)(T,T),int,integration<T>) const;
  public :
  p1nc2d();
  p1nc2d(const char*,const char*);
  p1nc2d(const char*);
  p1nc2d(const p1nc2d<T>&);
  p1nc2d<T>& operator=(const p1nc2d<T>&);
  int DofNumber()const{return (DofNumber_);};
  int GlobalDofNumber()const{return (GlobalDofNumber_);};
  matrix<T> RefMassMatrix()const{return(RefMassMatrix_);};
  vector<T> BasisFunction(T,T)const;
  matrix<T> DiffBasisFunction(T xref=0,T yref=0)const;
  virtual int operator ()(int,int) const;
  template <class FT> friend  int operator == (const p1nc2d<FT>&,const p1nc2d<FT>&);
  template <class FT> friend  int operator != (const p1nc2d<FT>&,const p1nc2d<FT>&);
  matrix<T> LocalMassMatrix(int)const;
  matrix<T> LocalStiffnessMatrix(int)const;
  vector<T> Fk(T,T, int) const;
  //BEFORE TO USE IT, FIRST FINISH SPMATRIX AND OPERATOR (.,.)
  //template <typename matrix_type> void DUDV(matrix_type & MAT) const; // Global Stiffness matrix
  void DUDV(matrix<T> & MAT) const; // Global Stiffness matrix
  void DUDV(spmatrix<T> & MAT) const; // Global Stiffness spmatrix
  vector<T> B(T (*f)(T,T),int NbIntPts ) const;
  //BEFORE TO USE IT, FIRST FINISH SPMATRIX AND OPERATOR (.,.)
  //template <typename matrix_type> void Dirichlet(matrix_type & MAT,vector<T>&F,T (*g)(T,T)) const; // Dirichlet BC 
  void Dirichlet(matrix<T> & MAT,vector<T>&F,T (*g)(T,T)) const; // Dirichlet BC for matrix 
  void Dirichlet(spmatrix<T> & MAT,vector<T>&F,T (*g)(T,T)) const; // Dirichlet BC for spmatrix
   T error_L2 (T (*u)(T,T),const vector<T>& uh,int NbIntPts) const ;
  T error_semi_H1(T (*dudx)(T,T),T (*dudy)(T,T),const vector<T>& uh,int NbIntPts) const;
   T Grad_Normal_Jump_L2(const vector<T>& uh, int num_element) const;
   T Grad_Tangential_Jump_L2(const vector<T>& uh, int num_element) const;
   T Laplacian_Residual_L2(T (*f)(T,T),int num_element) const;
   void refine_FE(vector<int>&); // refine the Finite Element
   void refine_FE_angle(vector<int>&,T); // refine the Finite Element
};


/*!
  \brief default constructor of the p1nc2d class
*/
//=============================
template <class T>
p1nc2d<T>::p1nc2d() :mesh_tri_2d<T>()
//=============================
{
DofNumber_=GlobalDofNumber_=0;
}


/*!
  \brief constructor of the p1nc2d class. The arguments are 2 files
  file1 and file2. file1 must be documented by the number of elements, node 's numerotation
  of each element. file2 indicate the dimension (2 or 3), 
  number of nodes and each node coordinates with a boundary condition (1 if node is on the boundary, 0 else).
  \param file1 string
  \param file2 string
  */
//=============================
template <class T>
p1nc2d<T>::p1nc2d(const char*file1,const char*file2) :mesh_tri_2d<T>(file1,file2)
//=============================
{
GlobalDofNumber_=nbsegments_;
DofNumber_=3;
RefMassMatrix_=matrix<T>(3,3);
RefMassMatrix_(1,1)=RefMassMatrix_(2,2)=RefMassMatrix_(3,3)=1./6.;
}


/*!
  \brief constructor of the p1nc2d class. It makes the same actions as the previous constructor, the 
  argument file is the concatenation of file1 and file2.
  \param file string
*/

//=============================
template <class T>
p1nc2d<T>::p1nc2d(const char*file) :mesh_tri_2d<T>(file)
//=============================
{
GlobalDofNumber_=nbsegments_;
DofNumber_=3;
RefMassMatrix_=matrix<T>(3,3);
RefMassMatrix_(1,1)=RefMassMatrix_(2,2)=RefMassMatrix_(3,3)=1./6.;


}


 /*!
  \brief copy constructor of the p1nc2 class
  \param EF  reference on p1nc2d object
  */

//=============================
template <class T>
p1nc2d<T>::p1nc2d(const p1nc2d<T>& EF) :mesh_tri_2d<T>(EF)
//=============================
{

  DofNumber_=EF.DofNumber_;
  GlobalDofNumber_=EF.GlobalDofNumber_;
  RefMassMatrix_=EF.RefMassMatrix_;
}
 
 /*!
  \brief affectation operator for the p1nc2d class.
  \param EF reference on p1nc2d object 
  \return  reference on p1nc2d object
 */
//========================================================
template <class T>
p1nc2d<T>& p1nc2d<T>::operator =(const p1nc2d<T>& EF)
//========================================================
{
 (*this).mesh_tri_2d<T>::operator=(EF);
  DofNumber_=EF.DofNumber_;
  GlobalDofNumber_=EF.GlobalDofNumber_;
  RefMassMatrix_=EF.RefMassMatrix_;
  return(*this);
}

/*!
   \param xref type T
   \param yref type T
   \return vector \f$(\varphi_1(xref,yref),\varphi_2(xref,yref),\varphi_3(xref,yref))\f$ 
   where \f$(\varphi_i)_{1\leq i\leq 3}\f$ is the i-th basis function.
  */
//=============================
template <class T>
vector<T> p1nc2d<T>::BasisFunction(T xref, T yref)const
//=============================
{
vector<T> P(DofNumber_);

P[1]=-1+2*xref+2*yref;
P[2]=1-2*xref;
P[3]=1-2*yref;
return(P);
}

/*!
   \param xref type T
   \param yref type T
   \return matrix 
   \f$ \left[\begin{array}{ccc}
	\frac{\partial\varphi_1}{\partial x_1} & \frac{\partial\varphi_2}{\partial x_1} & \frac{\partial\varphi_3}{\partial x_1} \\ 
	\frac{\partial\varphi_1}{\partial x_2} & \frac{\partial\varphi_2}{\partial x_2} & \frac{\partial\varphi_3}{\partial x_2}
\end{array} \right]
\f$ 
   where \f$(\varphi_i)_{1\leq i\leq 3}\f$ is the i-th basis function.
  */
//=============================
template <class T>
matrix<T> p1nc2d<T>::DiffBasisFunction(T xref,T yref)const
//=============================

{
matrix<T> DP(2,DofNumber_);
 DP(1,1)=2;DP(1,2)=-2;DP(1,3)=0;
 DP(2,1)=2;DP(2,2)=0;DP(2,3)=-2;
return(DP);
}

/*!
   \param x type T
   \param y type T
   \param i type integer
   \return vector \f$F_k(x,y,i)\f$ 
  */
//====================================================
template <class T>
vector<T> p1nc2d<T>::Fk(T x,T y, int i) const
//====================================================
{
  matrix<T> MAT(2,2);
  vector<T> VEC(2);
  VEC[1]=x ; VEC[2]=y;

  vector<T> A1A2=element[i-1][2]-element[i-1][1];
  vector<T> A1A3=element[i-1][3]-element[i-1][1];
  MAT(1,1)=A1A2[1];  MAT(1,2)=A1A3[1] ;
  MAT(2,1)=A1A2[2];  MAT(2,2)=A1A3[2];
  return MAT*VEC+element[i-1][1];
 
}
 /*!
   \param num_element element's number
   \return local mass matrix of the element num_element
  */
//====================================================
template <class T>
matrix<T> p1nc2d<T>::LocalMassMatrix(int num_element) const
//====================================================
{
assert((num_element>=1)&&(num_element<=nbelement_));
return(RefMassMatrix_*element[num_element-1].AbsDet());
}
 /*!
   \param num_element element's number
   \return local stifness  matrix of the element num_element
  */
//====================================================
template <class T>
matrix<T> p1nc2d<T>::LocalStiffnessMatrix(int num_element) const
//====================================================
{
assert((num_element>=1)&&(num_element<=nbelement_));
matrix<long double> inv=element[num_element-1].InvMk();
return ((element[num_element-1].AbsDet()/2.)*t(DiffBasisFunction())*inv*t(inv)*DiffBasisFunction());

}

/*!
   \param num_element element's number
   \param j the j-th dof 
   \return global number of the j-th of the element num_element
  */
//====================================================
template <class T>
int p1nc2d<T>::operator ()(int num_element,int j) const
//====================================================
{
assert((j>=1)&&(j<=DofNumber_)&&(num_element>=1)&&(num_element<=nbelement_));
return EdgesPerTriangle_(num_element,j);
}

/*!
   \param EF1 type const p1nc2d<T>&
   \param EF2 type const p1nc2d<T>&
   \return 1 if EF1 and  EF2 are equal, 0 else
  */
//==========================================================================
template <class FT>
int operator ==(const p1nc2d<FT>& EF1,const p1nc2d<FT>& EF2)
//==========================================================================
{
  int boolean=1;
  boolean*=(mesh_tri_2d<FT>(EF1)==mesh_tri_2d<FT>(EF2));
  boolean*=(EF1.DofNumber_==EF2.DofNumber_);
  boolean*=(EF1.GlobalDofNumber_==EF2.GlobalDofNumber_);
  boolean*=(EF1.RefMassMatrix_==EF2.RefMassMatrix_);
  return(boolean);
}
/*!
   \param EF1 type const p1nc2d<T>&
   \param EF2 type const p1nc2d<T>&
   \return 1 if EF1 and  EF2 are different, 0 else
  */
//==========================================================================
template <class FT>
int operator !=(const p1nc2d<FT>& EF1,const p1nc2d<FT>& EF2)
//==========================================================================
{
  return(!(EF1==EF2));
}


/*!
   \return the global stifness matrix of the p1nc2d element
  */
/*
//BEFORE TO USE IT, FIRST FINISH SPMATRIX AND OPERATOR (.,.)
//====================================================
template <class T> 
template <typename matrix_type>
void p1nc2d<T>::DUDV(matrix_type & MAT) const
//====================================================
{


matrix<T> Stiff(DofNumber_,DofNumber_);
for(int k=1;k<=nbelement_;k++)
	{
	Stiff=LocalStiffnessMatrix(k);
		for(int i=1;i<=DofNumber_;i++)
		for(int j=1;j<=DofNumber_;j++)
		        MAT.add(EdgesPerTriangle_(k,i),EdgesPerTriangle_(k,j),Stiff(i,j));
	}
}
*/
/*!
   \return the global stifness matrix of the p1nc2d element
  */
//====================================================
template <class T> 
void p1nc2d<T>::DUDV(matrix<T> & MAT) const
//====================================================
{


matrix<T> Stiff(DofNumber_,DofNumber_);
for(int k=1;k<=nbelement_;k++)
	{
	Stiff=LocalStiffnessMatrix(k);
		for(int i=1;i<=DofNumber_;i++)
		for(int j=1;j<=DofNumber_;j++)
		        MAT(EdgesPerTriangle_(k,i),EdgesPerTriangle_(k,j))+=Stiff(i,j);
	}
}
/*!
   \return the global stifness spmatrix of the p1nc2d element
  */
//====================================================
template <class T> 
void p1nc2d<T>::DUDV(spmatrix<T> & MAT) const
//====================================================
{


matrix<T> Stiff(DofNumber_,DofNumber_);
for(int k=1;k<=nbelement_;k++)
	{
	Stiff=LocalStiffnessMatrix(k);
		for(int i=1;i<=DofNumber_;i++)
		for(int j=1;j<=DofNumber_;j++)
		        MAT.add(EdgesPerTriangle_(k,i),EdgesPerTriangle_(k,j),Stiff(i,j));
	}
}

/*!
  \brief Value of the contribution to the right-hand side of the variational formulation of a given element \n \n
  \param \f$ f \f$ the function   \n
  \param num_element the element on which we want to integrate \n
  \param itg the Gauss points and weight used for the numerical integration
  \return vector \f$(\int_{num_{element}} f(x,y) \varphi_1(x,y) dx dy,\int_{num_{element}} f(x,y) \varphi_2(x,y) dx dy,\int_{num_{element}} f(x,y) \varphi_3(x,y) dx dy)\f$ 
   where \f$(\varphi_i)_{1\leq i\leq 3}\f$ is the i-th basis function associated to the element num_element.
  \n\n
*/
//====================================================
template <class T>
vector<T> p1nc2d<T>::BuildBloc(T (*f)(T,T),int num_element,integration<T> itg) const
//====================================================
{

vector<T> res(DofNumber_);
vector<T> Basis(DofNumber_);
T omegak, omegal, xk, xl;
vector<T> Fxkl;
   for (int k=1; k<=itg.NbIntPts(); k++)


    {
      omegak=itg.weight(k);
      xk=itg.point(k);

      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=(1-xk)*itg.point(l);
	  Basis=BasisFunction(xk,xl);
	  Fxkl=Fk(xk,xl,num_element);
          for (int i=1; i<=DofNumber_; i++)
	    res[i] += omegak*omegal*(1-xk)*f(Fxkl[1],Fxkl[2])*Basis[i];

	 }

    }
res*=element[num_element-1].AbsDet();
return(res);
}

/*!
  \brief Value of the right-hand side of the variational formulation \n \n
  \param \f$ f \f$ the function   \n
  \param NbIntPts the number of Gauss points to do the numerical integration \n
  \return vector v with v_i=\f$(\int_{\Omega} f(x,y) \varphi_i(x,y) dx dy \f$ 
   where \f$(\varphi_i)_{1\leq i\leq nbvertex}\f$ is the i-th global basis function.
  \n\n
*/
//======================================================
template <class T>
vector<T> p1nc2d<T>::B(T (*f)(T,T),int NbIntPts ) const
//======================================================
{
assert (NbIntPts>0);
integration<T> itg;
itg.Gauss(NbIntPts,0.0,1.0);

vector<T> B_(GlobalDofNumber_);

vector<T> Bloc(DofNumber_);


for(int k=1;k<=nbelement_;k++)
	{
	Bloc=BuildBloc(f,k,itg);
		for(int i=1;i<=DofNumber_;i++)
		B_[ EdgesPerTriangle_(k,i) ] += Bloc[i];
	}
return B_;
}
/*
//BEFORE TO USE IT, FIRST FINISH SPMATRIX AND OPERATOR (.,.)
//======================================================
template <class T> 
template <typename matrix_type>
void p1nc2d<T>::Dirichlet(matrix_type & MAT,vector<T>&F,T (*g)(T,T)) const
//======================================================
{
for(int i=1;i<=GlobalDofNumber_;i++)
{
if(Type_of_edge_[i])
	{
F[i]=(*g)(0.5*(vertices[SetOfEdges_(i,1)-1][1]
                         +vertices[SetOfEdges_(i,2)-1][1]),
                 0.5*(vertices[SetOfEdges_(i,1)-1][2]
                         +vertices[SetOfEdges_(i,2)-1][2]));
for(int j=1;j<=GlobalDofNumber_;j++)
	MAT.set(i,j,0);
	MAT.set(i,i,1);
	for(int k=1;k<=GlobalDofNumber_;k++)
	if(k!=i)
	{
	F[k]+=-MAT.output(k,i)*F[i];
	MAT.set(k,i,0);
	}


	}
}
}
*/
//======================================================
template <class T> 
void p1nc2d<T>::Dirichlet(matrix<T> & MAT,vector<T>&F,T (*g)(T,T)) const
//======================================================
{
for(int i=1;i<=GlobalDofNumber_;i++)
{
if(Type_of_edge_[i])
	{
F[i]=(*g)(0.5*(vertices[SetOfEdges_(i,1)-1][1]
                         +vertices[SetOfEdges_(i,2)-1][1]),
                 0.5*(vertices[SetOfEdges_(i,1)-1][2]
                         +vertices[SetOfEdges_(i,2)-1][2]));
for(int j=1;j<=GlobalDofNumber_;j++)
	MAT(i,j)=0.0;
	MAT(i,i)=1.0;
	for(int k=1;k<=GlobalDofNumber_;k++)
	if(k!=i)
	{
	F[k]+=-MAT(k,i)*F[i];
	MAT(k,i)=0.0;
	}


	}
}
}
//======================================================
template <class T> 
void p1nc2d<T>::Dirichlet(spmatrix<T> & MAT,vector<T>&F,T (*g)(T,T)) const
//======================================================
{
for(int i=1;i<=GlobalDofNumber_;i++)
{
if(Type_of_edge_[i])
	{
F[i]=(*g)(0.5*(vertices[SetOfEdges_(i,1)-1][1]
                         +vertices[SetOfEdges_(i,2)-1][1]),
                 0.5*(vertices[SetOfEdges_(i,1)-1][2]
                         +vertices[SetOfEdges_(i,2)-1][2]));
for(int j=1;j<=GlobalDofNumber_;j++)
	MAT.set(i,j,0);
	MAT.set(i,i,1);
	for(int k=1;k<=GlobalDofNumber_;k++)
	if(k!=i)
	{
	F[k]+=-MAT.output(k,i)*F[i];
	MAT.set(k,i,0);
	}


	}
}
}


/*!
   \param u exact solution
   \param uh vector approximation solution 
   \param NbIntPts number of points of the integration 
   \return error \f$||u-u_h||_{0,\Omega}\f$
  */
//======================================================
template <class T>
T p1nc2d<T>::error_L2 (T (*u)(T,T),const vector<T>& uh,int NbIntPts) const
//======================================================
{
T GlobalError=0;
T LocalError;
T omegak, omegal, xk, xl,SumK;
vector<T> Fxkl;

integration<T> itg;
itg.Gauss(NbIntPts,0.0,1.0);

vector<T> Basis(DofNumber_);

for(int num_element=1;num_element<=nbelement_;num_element++)
{

LocalError=0;


   for (int k=1; k<=itg.NbIntPts(); k++)


    {
      omegak=itg.weight(k);
      xk=itg.point(k);

      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=(1-xk)*itg.point(l);
	  Basis=BasisFunction(xk,xl);
          SumK=0.0;
	  for (int i=1; i<=DofNumber_; i++)
	  {SumK+=uh[EdgesPerTriangle_(num_element,i) ]*Basis[i];}
	  Fxkl=Fk(xk,xl,num_element);
	   LocalError += omegak*omegal*(1-xk)*
           pow((u(Fxkl[1],Fxkl[2])-SumK),2);
	 }

    }

GlobalError+=element[num_element-1].AbsDet()*LocalError;
}
return(sqrt(GlobalError));
}
/*!
   \param dudx first derivative of the exact solution
   \param dudy second  derivative of the exact solution
   \param uh vector approximation solution 
   \param NbIntPts number of points of the integration 
   \return l'erreur \f$|u-u_h|_{0,\Omega}\f$
 */
//======================================================
template <class T>
T p1nc2d<T>::error_semi_H1 (T (*dudx)(T,T),T (*dudy)(T,T),const vector<T>& uh,int NbIntPts) const
//======================================================
{
T GlobalError=0;
T LocalError;
T omegak, omegal, xk, xl;

T duhdx,duhdy;
T duhdxhat,duhdyhat;
vector<T> Fxkl;

integration<T> itg;
itg.Gauss(NbIntPts,0.0,1.0);

matrix<T> DiffBasis(2,DofNumber_);
matrix<T> InvMk;
for(int num_element=1;num_element<=nbelement_;num_element++)
{

LocalError=0;
InvMk=element[num_element-1].InvMk();

   for (int k=1; k<=itg.NbIntPts(); k++)


    {
      omegak=itg.weight(k);
      xk=itg.point(k);

      for (int l=1; l<=itg.NbIntPts(); l++)
	{
	  omegal=itg.weight(l);
          xl=(1-xk)*itg.point(l);
	  DiffBasis=DiffBasisFunction(xk,xl);

	  
	  duhdxhat=0.0;
	  duhdyhat=0.0;

	  for (int i=1; i<=DofNumber_; i++)
	  {
	  duhdxhat+=uh[ EdgesPerTriangle_(num_element,i) ]*DiffBasis(1,i);
	  duhdyhat+=uh[ EdgesPerTriangle_(num_element,i) ]*DiffBasis(2,i);
	  }

	  duhdx=duhdxhat*InvMk(1,1)+duhdyhat*InvMk(2,1);
	  duhdy=duhdxhat*InvMk(1,2)+duhdyhat*InvMk(2,2);

	  Fxkl=Fk(xk,xl,num_element);
	  LocalError += omegak*omegal*(1-xk)*
          (pow((dudx(Fxkl[1],Fxkl[2])-duhdx),2)+
	   pow((dudy(Fxkl[1],Fxkl[2])-duhdy),2));
	 }

    }
GlobalError+=LocalError*element[num_element-1].AbsDet();
}
return(sqrt(GlobalError));
}

//======================================================
template <class T>
T p1nc2d<T>::Laplacian_Residual_L2(T (*f)(T,T),int num_element) const
//======================================================
{
int NbPtsInt=2;
T omegak, omegal, xk, xl;
vector<T> Fxkl;
integration<T> itg;
itg.Gauss(NbPtsInt,0.0,1.0);
T LocalResidu=0.0;

for (int k=1; k<=NbPtsInt; k++)
    {
      omegak=itg.weight(k);
      xk=itg.point(k);
      for (int l=1; l<=NbPtsInt; l++)
	{
	  omegal=itg.weight(l);
          xl=(1-xk)*itg.point(l);
	  Fxkl=Fk(xk,xl,num_element);
	  LocalResidu+=omegak*omegal*(1-xk)*
          f(Fxkl[1],Fxkl[2]);
	 }
    }
 return(abs(LocalResidu)*element[num_element-1].diam()*
         sqrt(2.0*element[num_element-1].AbsDet()));
}

//======================================================
template <class T>
T p1nc2d<T>::Grad_Normal_Jump_L2(const vector<T>& uh,int num_element) const
//======================================================
{
T SumJumps=0;
T LocalJump;
T duhdx1,duhdy1,duhdx2,duhdy2;
T duhdxhat1,duhdyhat1,duhdxhat2,duhdyhat2;
vector<T> normal(2);
matrix<T> DiffBasis=DiffBasisFunction();
matrix<T> InvMk1,InvMk2;
int jseg,jelement;

InvMk1=element[num_element-1].InvMk();
duhdxhat1=0.0;
duhdyhat1=0.0;
for (int i=1; i<=DofNumber_; i++)
  {
  duhdxhat1+=uh[EdgesPerTriangle_(num_element,i)]*DiffBasis(1,i);
  duhdyhat1+=uh[EdgesPerTriangle_(num_element,i)]*DiffBasis(2,i);
  }

duhdx1=duhdxhat1*InvMk1(1,1)+duhdyhat1*InvMk1(2,1);
duhdy1=duhdxhat1*InvMk1(1,2)+duhdyhat1*InvMk1(2,2);

for(int edge=1;edge<=3;edge++)
{
 jseg=EdgesPerTriangle_(num_element,edge);
 normal=NormalUnitVector_[jseg];

 if (!Type_of_edge_[jseg]) // segment interne
 {
 jelement=TrianglesPerEdge_(jseg,1);
 if (jelement==num_element) jelement=TrianglesPerEdge_(jseg,2);

 InvMk2=element[jelement-1].InvMk();
 duhdxhat2=0.0;
 duhdyhat2=0.0;
 for (int i=1; i<=DofNumber_; i++)
	  {
	  duhdxhat2+=uh[EdgesPerTriangle_(jelement,i)]*DiffBasis(1,i);
	  duhdyhat2+=uh[EdgesPerTriangle_(jelement,i)]*DiffBasis(2,i);
	  }

 duhdx2=duhdxhat2*InvMk2(1,1)+duhdyhat2*InvMk2(2,1);
 duhdy2=duhdxhat2*InvMk2(1,2)+duhdyhat2*InvMk2(2,2);

 LocalJump=powf(NormEdge_[jseg],3.0/2.0)*
           abs(normal[1]*(duhdx2-duhdx1)+normal[2]*(duhdy2-duhdy1));
 }
 else // segment frontalier
 {
  LocalJump=0.0;
 }
SumJumps+=pow(LocalJump,2);
}
return(sqrt(SumJumps));
}
//======================================================
template <class T>
T p1nc2d<T>::Grad_Tangential_Jump_L2(const vector<T>& uh,int num_element) const
//======================================================
{
T SumJumps=0;
T LocalJump;
T duhdx1,duhdy1,duhdx2,duhdy2;
T duhdxhat1,duhdyhat1,duhdxhat2,duhdyhat2;
vector<T> tangent(2);
matrix<T> DiffBasis=DiffBasisFunction();
matrix<T> InvMk1,InvMk2;
int jseg,jelement;

InvMk1=element[num_element-1].InvMk();
duhdxhat1=0.0;
duhdyhat1=0.0;
for (int i=1; i<=DofNumber_; i++)
  {
  duhdxhat1+=uh[EdgesPerTriangle_(num_element,i)]*DiffBasis(1,i);
  duhdyhat1+=uh[EdgesPerTriangle_(num_element,i)]*DiffBasis(2,i);
  }

duhdx1=duhdxhat1*InvMk1(1,1)+duhdyhat1*InvMk1(2,1);
duhdy1=duhdxhat1*InvMk1(1,2)+duhdyhat1*InvMk1(2,2);

for(int edge=1;edge<=3;edge++)
{
 jseg=EdgesPerTriangle_(num_element,edge);
 tangent[1]=-NormalUnitVector_[jseg][2];
 tangent[2]=NormalUnitVector_[jseg][1];

 if (!Type_of_edge_[jseg]) // segment interne
 {
 jelement=TrianglesPerEdge_(jseg,1);
 if (jelement==num_element) jelement=TrianglesPerEdge_(jseg,2);

 InvMk2=element[jelement-1].InvMk();
 duhdxhat2=0.0;
 duhdyhat2=0.0;
 for (int i=1; i<=DofNumber_; i++)
	  {
	  duhdxhat2+=uh[EdgesPerTriangle_(jelement,i)]*DiffBasis(1,i);
	  duhdyhat2+=uh[EdgesPerTriangle_(jelement,i)]*DiffBasis(2,i);
	  }

 duhdx2=duhdxhat2*InvMk2(1,1)+duhdyhat2*InvMk2(2,1);
 duhdy2=duhdxhat2*InvMk2(1,2)+duhdyhat2*InvMk2(2,2);

 LocalJump=powf(NormEdge_[jseg],3.0/2.0)*
           abs(tangent[1]*(duhdx2-duhdx1)+tangent[2]*(duhdy2-duhdy1));
 }
 else // segment frontalier
 {
  LocalJump=powf(NormEdge_[jseg],3.0/2.0)*
           abs(2.0*(tangent[1]*duhdx1+tangent[2]*duhdy1));
 }
SumJumps+=pow(LocalJump,2);
}
return(sqrt(SumJumps));
}

//=============================
template <class T>
void p1nc2d<T>::refine_FE(vector<int> & TABTRI)
//=============================
{
  (*this).refine_mesh(TABTRI);
  GlobalDofNumber_=nbsegments_;
}

//=============================
template <class T>
void p1nc2d<T>::refine_FE_angle(vector<int> & TABTRI, T angle)
//=============================
{
  (*this).refine_mesh_angle(TABTRI,angle);
  GlobalDofNumber_=nbsegments_;
}

#endif

