/***************************************************************
 *                 Finite Element Method Object Library        *
 *                             tutorial 4                      *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 	0.2.0                  *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2006 CREUSE Emmanuel
 * copyright  Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class tuto4
    \brief tuto4 example \n

    \htmlonly 
    <FONT color="#838383">

    tuto4 belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2006 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.2.0
    \date 2006
    \bug none
    \warning none
*/

#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__STRING_H)
#include <string>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__ASSERT_H)
#include <time.h>
#endif

#if !defined(__VECTORS_H)
#include "../../MOL++/vectors.h"
#endif

#if !defined(__MATHS_H)
#include "../../MOL++/maths.h"
#endif

#if !defined(__INTEGRATION_H)
#include "../../MOL++/integration.h"
#endif

#if !defined(__CONVERT_H)
#include "../../FEMOL++/meshes/convert.h"
#endif

#if !defined(__MESH_TRI_2D_H)
#include "../../FEMOL++/meshes/mesh_tri_2d.h"
#endif

#if !defined(__TRIANGLE_H)
#include "../../FEMOL++/meshes/triangle.h"
#endif

#if !defined(__TRIANGLE_H)
#include "../../FEMOL++/elements/rt02d.h"
#endif

#if !defined(__AFFICHE_H)
#include "../../tests/affiche.h"
#endif

//-------------------------------------------------------
// Value of parameter p (and q computation with 1/p+1/q=1)
//-------------------------------------------------------
long double p=3.0;
long double q=p/(p-1);

//-----------------------------------
// Value of vareps for exact solution
//-----------------------------------
long double vareps=0.05;

//------------------------------------
// Nb points for numerical integration
//------------------------------------
int NbIntPts=2;

//-------------------------------
// Stop test for Picard algorithm
//-------------------------------
long double arret_picard=1.0e-06;

//-----------------------------
// Stop test for linear system
//-----------------------------
long double arret_solveur=precision;

//---------------
// Nb refinements
//---------------
int nb_raff=7; 

//-------------
// Initial mesh
//--------------
char* maillage="../../tests/FEMOL++/data/Carre/carre16.dat";

/*
long double r2(long double x, long double y)
{
return(pow(x,2)+pow(y,2));
}
*/

//-----------------------------
// Valeur de la solution exacte
//-----------------------------
/*
long double u(long double x, long double y)
{
long double d2=r2(x,y);
if (d2<=0.25)
 return(powf((0.25-d2),2.0)*exp(-d2/vareps));
else
 return 0;
}
*/
//----------------------------------------------
// Drives de la solution et dtermination de p
//----------------------------------------------
long double r2(long double x, long double y)
{
return(pow(x,2)+pow(y,2));
}
long double u(long double x, long double y)
{
long double d2=r2(x,y);
if (d2<=0.25)
 return(powf((0.25-d2),2.0)*exp(-d2/vareps));
else
 return 0;
}
long double dudr(long double x, long double y)
{
long double d2=r2(x,y);
if (d2<=0.25)
return((0.25-d2)*(2.0+1.0/(4.0*vareps)-d2/vareps)*exp(-d2/vareps)*(-2.0)*sqrt(d2));
else
return 0;
}
long double dudx(long double x, long double y)
{
long double d2=r2(x,y);
if (d2<=0.25)
 return(x/sqrt(d2)*dudr(x,y));
else
 return 0;
}
long double dudy(long double x, long double y)
{
long double d2=r2(x,y);
if (d2<=0.25)
 return(y/sqrt(d2)*dudr(x,y));
else
 return 0;
}
long double px(long double x, long double y)
{
long double d2=r2(x,y);
if (d2<=0.25)
return(-powf((0.25-d2),p-1)*powf((2.0+1.0/(4.0*vareps)-d2/vareps),p-1.0)*powf(2.0,p-1.0)*powf(d2,(p-2.0)/2.0)*
       exp(-(p-1.0)*d2/vareps)*x);
 else
 return 0;      
}
long double py(long double x, long double y)
{
long double d2=r2(x,y);
if (d2<=0.25)
return(-powf((0.25-d2),p-1)*powf((2.0+1.0/(4.0*vareps)-d2/vareps),p-1.0)*powf(2.0,p-1.0)*powf(d2,(p-2.0)/2.0)*
       exp(-(p-1.0)*d2/vareps)*y);
 else
 return 0;      
}
long double f(long double x, long double y)
{
long double d2=r2(x,y);
if (d2<=0.25)
 {
  long double a=powf(2.0,p);
  long double b=(0.25-d2);
  long double c=(2.0+1.0/vareps*(0.25-d2));
  long double d=exp(-(p-1.0)*d2/vareps);
  return(-a*(p-1.0)*powf(b,p-2.0)*powf(c,p-1.0)*d*powf(d2,p/2.0) 
         -a*(p-1.0)*powf(b,p-1.0)*powf(c,p-2.0)*d*powf(d2,p/2.0)/vareps
	 -a*(p-1.0)*powf(b,p-1.0)*powf(c,p-1.0)*d*powf(d2,p/2.0)/vareps
	 +p*powf(2.0,p-1.0)*powf(b,p-1.0)*powf(c,p-1.0)*d*powf(d2,(p-2.0)/2.0)
	);
  }
else
 return 0;
}
//==========================
main()
//==========================
{
cout << "-----------------------------------------\n";
cout << "-----------------------------------------\n";
cout << " BEGINNING tuto4 \n";
cout << "-----------------------------------------\n";
cout << "-----------------------------------------\n";
cout << endl;

//-----------------------
// Finite element loading
//-----------------------
make_convertion2dtri(maillage,"tampon.dat","square");
rt02d<long double> Th("tampon.dat");

//----------------
// Local variables
//----------------
clock_t start, end;
start = clock();    
spmatrix<long double> A;
vector<long double> B;
vector<long double> Lambda, X0;
vector<long double> U;
vector<long double> Pmp1,Pm;
long double error_p,error_u,error,error_max;
long double ESTI_U,ESTI_P;
vector<int> TABTRI;
int nbiter=0;

//-------------------
// Mesh visualisation
//-------------------
Th.visu2d();

//--------------
// Loop begining
//--------------

for (int n_raff=1; n_raff<=nb_raff; n_raff++)
{
   cout << "------------------------------ \n";
   cout << "NRAFF= " << n_raff << "\n";
   cout << "ndof= " << Th.GlobalDofNumber() << "\n";
   cout << "------------------------------ \n";
   
//-----------------------------   
// Previous objects destruction
//------------------------------   
   A.~spmatrix();
   B.~vector(); 
   Lambda.~vector(); 
   X0.~vector();  
   U.~vector();  
   Pmp1.~vector(); 
   Pm.~vector(); 
   TABTRI.~vector();
   
//---------------------   
// Objects construction
//---------------------
   A=spmatrix<long double>(Th.GlobalDofNumber(),Th.GlobalDofNumber(),5);
   B=vector<long double>(Th.GlobalDofNumber());
   Lambda=vector<long double>(Th.GlobalDofNumber());
   X0=vector<long double>(Th.GlobalDofNumber());
   U=vector<long double>(Th.nbelement());
   Pmp1=vector<long double>(Th.GlobalDofNumber());
   Pm=vector<long double>(Th.GlobalDofNumber());
   TABTRI=vector<int>(Th.nbelement());
   
//--------------------------------------------------------------------------------   
// Pmp1 initialisation (non zero to enter in the Picard loop !) 
//--------------------------------------------------------------------------------  
for(int i=1; i<=Th.GlobalDofNumber(); i++)
Pmp1[i]=1.0; 

//----------------------
// Picard loop beginning
//----------------------

for (int indice_boucle=1; ((Pmp1-Pm).norm()>arret_picard); indice_boucle++)
  {
//----------------------
// Matrix initialisation
//-----------------------
   A.~spmatrix();  
   A=spmatrix<long double>(Th.GlobalDofNumber(),Th.GlobalDofNumber(),5);
//----------
// Pm saving
//----------   
   Pm=Pmp1;
   
//-----------------   
// B Initialisation
//-----------------   
   for (int i=1; i<=Th.GlobalDofNumber(); i++) B[i]=0;

//-------------------
// A et B computation 
//-------------------  
   Th.Assemble(A,B,f,Pm,q,NbIntPts);

//------------------------------   
// Dirichlet boundary conditions
//------------------------------   
   Th.Dirichlet(A,B,u,p);

//-------------------
// lambda computation   
//------------------- 
   Lambda=conjugate_gradient(A,B,X0,nbiter,arret_solveur);

//--------------
// u computation
//--------------    
   Th.Compute_u(U,Lambda,f,Pm,q,NbIntPts);

//--------------
// p computation
//--------------   
   Th.Compute_p(Pmp1,U,Lambda,Pm,q,NbIntPts);

//-------------------
// Errors computation
//-------------------      
   error_p=Th.error_p(Pmp1,px,py,q,NbIntPts);
   error_u=Th.error_u(U,p,u,NbIntPts);
   
//-------------------   
// Results displaying 
//-------------------
   cout << "---------------------------------------------- \n";
   cout << "---------------------------------------------- \n";
   cout << "loop indice= " << indice_boucle << " \n" ;
   cout << "ndof= " << Th.GlobalDofNumber() << "\n";
   cout << "Error on p= " << error_p << "\n";
   cout << "Error on u= " << error_u << "\n";
   
//------------------------------------------   
// A posteriori error estimators computation
//------------------------------------------   
   long double ETA1=Th.eta_1(Pm,p,NbIntPts);
   long double ETA2=Th.eta_2(f,q,NbIntPts);
   if (p>=2.0)
    {
   ESTI_P=sqrt(pow(ETA1,2)+powf(ETA2,q)+ETA1*ETA2);
   ESTI_U=ETA1+powf(pow(ETA1,2)+powf(ETA2,q)+ETA1*ETA2,1.0/p);
    }
    else
    {
   ESTI_P=powf(ETA1,p/q)+powf(ETA2,2.0/q)+powf(ETA1*ETA2,1.0/q);
   ESTI_U=ETA1+ETA2+powf(ETA1,p/2.0);
    }
//-------------------   
// Results displaying 
//-------------------     
   cout << "ESTI_P = " << ESTI_P << "\n";
   cout << "ESTI_U = " << ESTI_U << "\n"; 
   cout << "Ratio error p/ ESTI_P = " << error_p/ESTI_P << "\n";
   cout << "Ratio error u/ ESTI_U = " << error_u/ESTI_U << "\n";
   cout << "Test arret= " << (Pmp1-Pm).norm() << "\n";
   cout << "------------------------------------------------ \n";
   cout << "------------------------------------------------ \n";
   }
//------------------------   
// Mesh refinement process
//------------------------       
   long double esti,esti_max=0;
   for (int i=1; i<=Th.nbelement(); i++)
    {
    esti=powf(Th.eta_11_K_p(Pm,p,NbIntPts,i)+Th.eta_13_K_p(Pm,p,i),1.0/p)
        +powf(Th.eta_2_K_q(f,q,NbIntPts,i),1.0/q);
    if (esti>esti_max) esti_max=esti;
    }
    
     for (int i=1; i<=Th.nbelement(); i++)
    {
      esti=powf(Th.eta_11_K_p(Pm,p,NbIntPts,i)+Th.eta_13_K_p(Pm,p,i),1.0/p)
          +powf(Th.eta_2_K_q(f,q,NbIntPts,i),1.0/q);
      if (esti>0.75*esti_max) TABTRI[i]=-1; 
    }
    
     Th.refine_FE_angle(TABTRI,0.5);
     cout << " OK!" << "\n";
     Th.visu2d();
     cout << "Th.GlobalDofNumber= "  << Th.GlobalDofNumber() << "\n";
     cout << "\n";
               
 }

end = clock();
cout<<"cpu time = " << (long double)(end-start)/CLOCKS_PER_SEC << endl;

cout << endl;
cout << "-----------------------------------------\n";
cout << "-----------------------------------------\n";
cout << " END tuto4 \n";
cout << "-----------------------------------------\n";
cout << "-----------------------------------------\n";

}


