
/***************************************************************
 *                 Finite Element Method Object Library        *
 *           class node : declaration for node                 *
 *                    simula+@metz.ensam.fr                    *
 *	             GNU/linux version 0.1.8	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2002,2003,2004,2005,2006 CREUSE Emmanuel
 * copyright  2002,2003,2004,2005,2006 SOUALEM Nadir
 * copyright  2002,2003 COLLARD Christophe
 * copyright  2003 DJADEL Karim
 * copyright  Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class node
    \brief node library \n

    \htmlonly 
    <FONT color="#838383">

    node belongs to Finite Element Method Object Libraries (FEMOL++) </br>
    FEMOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly  2002,2003,2004,2005,2006 CREUSE Emmanuel \n
	     copyright \htmlonly &#169; \endhtmlonly  2002,2003,2004,2005,2006 SOUALEM Nadir \n
	     copyright \htmlonly &#169; \endhtmlonly  2002,2003 COLLARD Christophe \n
	     copyright \htmlonly &#169; \endhtmlonly  2003 DJADEL Karim \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	     copyright \htmlonly &#169; \endhtmlonly Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.1.8
    \date 2002-2006
    \bug none
    \warning none
*/
#ifndef __cplusplus
#error Must use C++ for the type node
#endif

#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__VECTORS_H)
#include "../../MOL++/vectors.h"
#endif


#ifndef _node_h
#define _node_h

using namespace std;


//==============================================
template <class T> class node : public vector<T>
//==============================================
{
  protected :
    int location;  // -1 : no condition
                   // 0 : internal node
                   // 1 : boundary node

  public :
    node ();    	          // default constructor
    node (int);                   // constructor (space dimension = dim());
    node(const node<T>&);        //copy constructor
    node (const vector<T>&, int); // constructor (node coordinates, node location)
    node (int, const vector<T>&); // constructor (node location, node coordinates)
    ~node ();                     // destructor
    virtual node<T>& operator = (const node<T>&); // defines equal operator for nodes
    friend  ostream& operator << <T> (ostream&, const node<T>&); // overloads output stream for node
    friend  istream& operator >> <T> (istream&, node<T>&); //overloads input stream for node
    int& boundary_condition();    // return or change node's condition
    friend  int operator == <T> (const node<T>&, const node<T>&); //compares 2 nodes
};


//=====Private methods for node=============================================


//=====Public  methods for node=============================================

/*!
  \brief Default Constructor 
  \n \n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{node$<$double$>$ w;} \; \textit{Creates a node with type double but does not allocate memory to store the coordinates. Nevertheless, location is set to -1.} \f$ \n
  \n\n
*/
//========================================================================
template <class T> node<T>::node() : vector<T>()
//========================================================================
{
  location = -1;
}

/*!
  \brief Constructor \n \n
  \param n space dimension \n\n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{node$<$long double$>$ v(2);} \; \textit{Allocates memory for a node with type long double and with 2 coordinates initialized to zero, and location is set to -1.}  \f$ \n
*/
//========================================================================
template <class T> node<T>::node(int n) : vector<T>(n)
//========================================================================
{
  location = -1;
}

/*!
  \brief Copy constructor
  \param v node to duplicate
*/
//========================================================================
template <class T> node<T>::node(const node<T>& A) : vector<T>(A)
//========================================================================
{
  location=A.location;
}


/*!
  \brief Constructor \n \n
  \param v vector node coordinates \n 
  \param position node location \n \n
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{node$<$long double$>$ a(v,0);} \; \textit{Creates an internal node from its coordinates in v}  \f$ \n
  \f$ \textbf{node$<$long double$>$ a(v,1);} \; \textit{Creates a boundary node from its coordinates in v}  \f$ \n
*/
//========================================================================
template <class T> node<T>::node(const vector<T>& v,int position) : vector<T>(v)
//========================================================================
{
  location = position;
}

/*!
  \brief Constructor \n \n
  \param position node location \n \n
  \param v vector node coordinates \n 
  \f$ \textbf{Examples :} \f$ \n
  \f$ \textbf{node$<$long double$>$ a(0,v);} \; \textit{Creates an internal node from its coordinates in v}  \f$ \n
  \f$ \textbf{node$<$long double$>$ a(1,v);} \; \textit{Creates a boundary node from its coordinates in v}  \f$ \n
*/
//========================================================================
template <class T> node<T>::node(int position, const vector<T>& v) : vector<T>(v)
//========================================================================
{
  location = position;
}

/*!
  \brief Destructor

  Deletes node coordinates table if vector size is not null.
*/
//========================================================================
template <class T> node<T>::~node()
//========================================================================
{
  location = -1;
}

/*!
  \brief Standard operator = for node \n \n
  See the operator = of the class vectors for the vector of coordinates. \n
  It also copies the location.\n \n
  \param noeud node to copy \n
  \return reference of the left hand size vector (for multiple equalities)
  \n\n
*/
//========================================================================
template <class T> node<T>& node<T>::operator = (const node<T>& noeud)
//========================================================================
{
  assert(noeud.dim()); // vector for noeud must exist
  (*this).vector<T>::operator=(noeud);
  location = noeud.location;
  return *this;
}

/*!
  \brief  overloads output stream for node \n \n
  \param out output flow \n
  \param noeud node
*/
//========================================================================
template <class T> ostream& operator << (ostream& out, const node<T>& noeud)
//========================================================================
{
  if (noeud.dim())//&&noeud.location!=-1   ??????????????
    {
    out << "Coordinates : ";
    for (int i=1; i<=noeud.dim(); i++)
	    out << noeud[i] << " ";
    out << "Condition : "<< noeud.location<<endl;
    }
  else out << "Not defined node"<<endl;
  return out;
}

/*!
  \brief  overloads input stream for node \n \n
  \param in input flow \n
  \param noeud node
*/
//========================================================================
template <class T> istream& operator >> (istream& in, node<T>& noeud)
//========================================================================
{
  assert(noeud.dim());
  for (int i=1; i<=noeud.dim(); i++)
    in >> noeud[i];
  in >> noeud.location;
  return in;
}

/*!
  \brief  returns location \n \n
*/
//========================================================================
template <class T> int& node <T> :: boundary_condition()
//========================================================================
{
  return location;
}

/*!
  \brief  Check the equality of two nodes \n \n
  \param A node \n
  \param B node \n
  \return 1 if equal, 0 if not
*/
//========================================================================
template <class T> int operator ==(const node<T> &A,const node<T> &B)
//========================================================================
{
  int boolean=1;
  boolean*=(vector<T>(A)==vector<T>(B))&&(A.location==B.location);
  return(boolean);
}
#endif
