/*
 * Copyright 2009-2017 FableTech
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION

#include "apr_shm.h"
#include "apr_errno.h"
#include "apr_general.h"
#include "apr_lib.h"
#include "apr_strings.h"
#include "apr_time.h"

#include "httpd.h"
#include "scoreboard.h"

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#if APR_HAVE_UNISTD_H
#include <unistd.h>
#endif


#if (( \
		(AP_SERVER_MAJORVERSION_NUMBER < 2) \
	) || ( \
		(AP_SERVER_MAJORVERSION_NUMBER == 2) && \
		(AP_SERVER_MINORVERSION_NUMBER < 4) \
	) || ( \
		(AP_SERVER_MAJORVERSION_NUMBER == 2) && \
		(AP_SERVER_MINORVERSION_NUMBER == 4) && \
		(AP_SERVER_PATCHLEVEL_NUMBER < 18) \
	))
#define SIZE_OF_global_score  sizeof(global_score)
#define SIZE_OF_process_score sizeof(process_score)
#define SIZE_OF_worker_score  sizeof(worker_score)
#else
#define SIZE_OF_global_score  APR_ALIGN_DEFAULT(sizeof(global_score))
#define SIZE_OF_process_score APR_ALIGN_DEFAULT(sizeof(process_score))
#define SIZE_OF_worker_score  APR_ALIGN_DEFAULT(sizeof(worker_score))
#endif


#define GLOBAL ((global_score *)p)

#define PROCESS(i) ((process_score *)(p \
	+ SIZE_OF_global_score \
	+ sizeof(process_score) * (i) \
	))

#define WORKER(i,j) ((worker_score *)(p \
	+ SIZE_OF_global_score \
	+ SIZE_OF_process_score * GLOBAL->server_limit \
	+ sizeof(worker_score) * (GLOBAL->thread_limit * (i) + (j)) \
	))


static const char statuschr[] = { '.','S','_','R','W','K','L','D','C','G','I' };


int main(int argc, char *argv[])
{
	apr_status_t rv;
	apr_pool_t *pool;
	apr_shm_t *shm;
	char *p;
	size_t size, good_size;
	worker_score *worker;
	int i, j;

	if (argc != 2) {
		fprintf(stderr, "Usage: %s /path/to/scoreboard\n", argv[0]);
		return EXIT_FAILURE;
	}

	apr_initialize();

	if ((rv = apr_pool_create(&pool, NULL)) != APR_SUCCESS) {
		fprintf(stderr, "%s: apr_pool_create() failed: %d\n",
			argv[0], rv);
		return EXIT_FAILURE;
	}

	if ((rv = apr_shm_attach(&shm, argv[1], pool)) != APR_SUCCESS) {
		fprintf(stderr, "%s: apr_shm_attach() failed: %d\n",
			argv[0], rv);
		return EXIT_FAILURE;
	}

	if (!(p = apr_shm_baseaddr_get(shm))) {
		fprintf(stderr, "%s: apr_shm_baseaddr_get() failed.\n",
			argv[0]);
		return EXIT_FAILURE;
	}

	if ((size = apr_shm_size_get(shm)) < sizeof(*GLOBAL)) {
		fprintf(stderr, "%s: Scoreboard too small.\n",
			argv[0]);
		return EXIT_FAILURE;
	}

	good_size =
		SIZE_OF_global_score
		+ (SIZE_OF_process_score
			* GLOBAL->server_limit)
		+ (SIZE_OF_worker_score
			* GLOBAL->server_limit
			* GLOBAL->thread_limit)
#ifdef SCOREBOARD_BALANCERS
		+ (sizeof(lb_score)
			* GLOBAL->lb_limit)
#endif
		;

	if (size != good_size) {
		fprintf(stderr, "%s: Wrong scoreboard size (got %d, should have been %d).\n",
			argv[0], size, good_size);
		fprintf(stderr, "This ftss was build for %s.\n", AP_SERVER_VERSION);
		return EXIT_FAILURE;
	}

	for (i=0; i<GLOBAL->server_limit; i++) {
		if (!PROCESS(i)->pid) continue;
		for (j=0; j<GLOBAL->thread_limit; j++) {
			if (WORKER(i,j)->status == SERVER_DEAD) continue;
			if (WORKER(i,j)->status >= SERVER_NUM_STATUS) continue;
			printf("%d", PROCESS(i)->pid);
			printf("\t%c", statuschr[WORKER(i,j)->status]);
			printf("\t%.32s", WORKER(i,j)->client);
			printf("\t%.32s", WORKER(i,j)->vhost);
			printf("\t%.64s", WORKER(i,j)->request);
			printf("\n");
		}
	}

	if ((rv = apr_shm_detach(shm)) != APR_SUCCESS) {
		fprintf(stderr, "%s: apr_shm_detach() failed: %d\n",
			argv[0], rv);
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}
