/********************************************************************************
				   QueryTool.cpp
			       -------------------
	   begin		: lun jui 14 19:50:56 CEST 2003
       copyright		: (C) 2003 by Thomas Silvi
	   email		: thomas.silvi@laposte.net
*********************************************************************************/

/********************************************************************************
*                                                                               *
*   This file is part of the Escort Tools Suite program.                        *
*                                                                               *
*   Escort Tools Suite is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by        *
*   the Free Software Foundation; either version 2 of the License, or           *
*   (at your option) any later version.                                         *
*                                                                               *
*   Escort Tools Suite is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of              *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *
*   GNU General Public License for more details.                                *
*                                                                               *
*   You should have received a copy of the GNU General Public License           *
*   along with Escort Tools Suite; if not, write to the Free Software           *
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   *
*                                                                               *
*********************************************************************************/

#include "QueryTool.h"
#include "ConnectionParameters.h"
#include "ConnectionDialog.h"
#include "ConfigDialog.h"
#include "Misc.h"

#include <klocale.h>
#include <kmenubar.h>
#include <kpopupmenu.h>
#include <kmessagebox.h>
#include <kurl.h>
#include <kfiledialog.h>
#include <kkeydialog.h>

#include <qvbox.h>

#define FIRST_CONN 3

extern QArray<QString> toolsArray;

extern QString applicationName;

QueryTool::QueryTool(QWidget *parent, const char *name, ToolSuiteConfig *pConfig) 
	: ToolSkeleton(parent, name)
{
	internalCanQuit = false;
	app = 0;
	
	viewsMap = new QMap<QString,QueryToolView*>();
	config = pConfig;
	lastViewCreated = 0;
	
	connections = 0;
	
	initView();
	initActions();
	initGUI();
	
	toggleMainToolbar();
	
	// final tasks
	connections = (KPopupMenu*)(factory()->container("connections",this));

}

QueryTool::~QueryTool()
{
	app = 0;
	connections = 0;
	config = 0;
	lastViewCreated = 0;
	
	if (viewsMap != 0)
	{
		viewsMap->clear();
		delete viewsMap;
		viewsMap = 0;
	}

}

QueryToolView * QueryTool::createClient(QueryToolDoc *pDoc)
{
	QueryToolView *viewTmp;
	QString sNameTmp;
	unsigned int i;
	QWidgetList listTmp;

	viewTmp= new QueryToolView(pDoc, workspace,0,WDestructiveClose);

	if (viewTmp == 0)
	{
		return 0;
	}

	viewTmp->setApp(app);
	sNameTmp = pDoc->getName();

#ifdef DEBUG_APP
	printf("DEBUG_APP QueryTool::createClient> Client registering %s\n",sNameTmp.ascii());
#endif

	(*viewsMap)[sNameTmp] = viewTmp;
	i = connections->count();

	connections->insertItem(pDoc->getName(), this,
		SLOT(changeConnectionView(int)), 0,i);

	listTmp = workspace->windowList(QWorkspace::CreationOrder);
		
	if ((config->getBool("SetViewCfgWithCfgLastViewCreated") == true) && (lastViewCreated!=0))
	{
		QueryToolViewConfig * viewCfg;

		viewCfg = lastViewCreated->getQueryToolViewConfig();
		viewTmp->applyQueryToolViewConfig(viewCfg);

		if (viewCfg!=0)
		{
			delete viewCfg;
		}
	}

	if (listTmp.count() <= 1)
	{
		// set default view size to 80% of the parent (when click on minimum size)
		signed int dx,dy;

		dx = workspace->width();
		dy = workspace->height();

		dx = (signed int)(0.8 *dx);
		dy = (signed int)(0.8 *dy);

		viewTmp->resize(dx,dy);

		viewTmp->showMaximized();
	}
	else
	{
		// do not show the view maximized
//		viewTmp->show();
		viewTmp->showNormal();
	}


	return viewTmp;
}

void QueryTool::unregister(QString &pConnectionName)
{
	// must be always called by a QueryToolDoc instance.
	// remove the view but do not delete it
	
	if (viewsMap==0)
	{
		return;
	}
	
	viewsMap->remove(pConnectionName);
	updateConnectionMenu();
	updateLastView();
	
	return;
}


void QueryTool::setActiveView(QString pMDIid)
{
	QueryToolView *vTmp;
	
	vTmp = (*viewsMap)[pMDIid];
	
	// is there a view / a connection ?
	
	if (vTmp == 0)
	{
		// error
		return;
	}
	
	vTmp->setFocus();
	vTmp->switchView();
	
	vTmp = 0;
	
	return;
}


void QueryTool::setApp(KApplication *pApp)
{
	app = pApp;
	return;	
}


void QueryTool::closeEvent(QCloseEvent *pEvent)
{
	if (isThereBatchRunning() == 0)
	{
		if (internalCanQuit == false)
		{
			byebye();
		}
		
		// after the call of byebye the value of internalCanQuit may have changed.
		
		if (internalCanQuit == true)
		{
			KMainWindow::closeEvent(pEvent);
		}
	}
	
//	KMainWindow::closeEvent(pEvent);	
	return;	
}


void QueryTool::openQuery()
{
	QueryToolView *v;
	KURL urlTmp;
	QString strHome, directoryTmp, fileNameTmp, errorMessageTmp;
	
	v = (QueryToolView*) workspace->activeWindow();
	
	// is there a view / a a connection ?

	if (v == 0)
	{
		KMessageBox::error (this, i18n("There is no connections."));
		return;
	}
	
	// is there a batch running ?

	if ((v->isBatchRunning()) == true)
	{
		KMessageBox::error (this, i18n("Error : a batch is running !"));
		return;
	}

	directoryTmp = v->getDirectoryOpen();
	
	if (replaceTildaByHomeDirectory(directoryTmp, errorMessageTmp) == false)
	{
		KMessageBox::error (this, errorMessageTmp);
		return;
	}
	
#ifdef DEBUG_APP
	printf("DEBUG_APP QueryTool::openQuery> %s\n",directoryTmp.ascii());
#endif
	
	urlTmp = KFileDialog::getOpenURL(directoryTmp,
		i18n("*.sql|SQL Files\n*|All files"),
		this, i18n("Open File..."));
	
	if(urlTmp.isEmpty())
	{
		// CANCEL pressed
		return;
	}
	
	directoryTmp = urlTmp.directory();
	fileNameTmp = urlTmp.fileName();
	
	v->openInput(directoryTmp, fileNameTmp);
	v->switchToInput();

	return;	
}


void QueryTool::saveQueryResult()
{
	QueryToolView *v;
	
	v = (QueryToolView*) workspace->activeWindow();
	
	// is there a view / a connection ?
	
	if (v == 0)
	{
		KMessageBox::error (this, i18n("There is no connections."));
		return;
	}
	
	// is there a batch running ?
	
	if ((v->isBatchRunning()) == true)
	{
		KMessageBox::error (this, i18n("Error : a batch is running !"));
		return;
	}

	signed char activeTabTmp;
	QString directoryTmp, fileNameTmp;

	activeTabTmp = v->getActiveTab();

	if (activeTabTmp == 0)
	{
		// save the query batch
		if (config->getBool("SameFileSaveLoad") == true)
		{
			directoryTmp = v->getDirectoryOpen();
			fileNameTmp = v->getFileNameOpenInput();
		}
		else
		{
			directoryTmp = v->getDirectorySaveInput();
			fileNameTmp = v->getFileNameSaveInput();
		}
		
		if ((fileNameTmp == QString::null)||(fileNameTmp.isEmpty()))
		{
			saveAsQueryResult();
			return;
		}
		
		v->saveInput(directoryTmp,fileNameTmp);
	}
	else if (activeTabTmp == 1)
	{
		// save the output
		directoryTmp = v->getDirectorySaveOutput();
		fileNameTmp = v->getFileNameSaveOutput();
		
		if ((fileNameTmp == QString::null)||(fileNameTmp.isEmpty()))
		{
			saveAsQueryResult();
			return;
		}
		
		v->saveOutput(directoryTmp,fileNameTmp);
	} else
	{
		// warning
		KMessageBox::sorry (this, i18n("No save done. Focus on input or output."));
	}

	return;	
}


void QueryTool::saveAsQueryResult()
{
	QueryToolView *v;
	
	v = (QueryToolView*) workspace->activeWindow();
	
	// is there a view / a connection ?
	
	if (v == 0)
	{
		KMessageBox::error (this, i18n("There is no connections."));
		return;
	}
	
	// is there a batch running ?
	
	if ((v->isBatchRunning()) == true)
	{
		KMessageBox::error (this, i18n("Error : a batch is running !"));
		return;
	}
	
	// ask for directory and file name
	KURL urlTmp;
	QString directoryTmp, fileNameTmp, filterTmp, errorMessageTmp;
	signed char activeTabTmp;
	
	activeTabTmp = v->getActiveTab();
	
	if (activeTabTmp == 0)
	{
		// get date for the query batch
		filterTmp = i18n("*.sql|SQL Files\n*|All files");
		directoryTmp = v->getDirectorySaveInput();
	}
	else if (activeTabTmp == 1)
	{
		// get data for the output
		filterTmp = i18n("*.txt|Text Files\n*|All files");
		directoryTmp = v->getDirectorySaveOutput();
	}
	else
	{
		// warning
		KMessageBox::sorry (this, i18n("No save done. Focus on input or output."));
		return;
	}
	
	if (replaceTildaByHomeDirectory(directoryTmp, errorMessageTmp) == false)
	{
		KMessageBox::error (this, errorMessageTmp);
		return;
	}
	
#ifdef DEBUG_APP
	printf("DEBUG_APP QueryTool::saveAsQueryResult> path: %s\n",directoryTmp.ascii());
#endif
	
	urlTmp = KFileDialog::getOpenURL(directoryTmp, filterTmp, this, i18n("Save File..."));
	
	if(urlTmp.isEmpty())
	{
		// CANCEL pressed
		return;
	}
	
	directoryTmp = urlTmp.directory();
	fileNameTmp = urlTmp.fileName();
	
	if (activeTabTmp == 0)
	{
		// save the query batch
		v->saveInput(directoryTmp, fileNameTmp);
	}
	else if (activeTabTmp == 1)
	{
		// save the output
		v->saveOutput(directoryTmp, fileNameTmp);
	}
	
	return;	
}


void QueryTool::requestTool()
{
	return;	
}


void QueryTool::executeInput()
{
	QueryToolView *v = (QueryToolView*) workspace->activeWindow();
	
	// is there a view / a connection ?
	
	if (v == 0)
	{
		KMessageBox::error (this, i18n("There is no connections."));
		return;
	}
	
	v->executeInput();

	return;	
}

void QueryTool::showConnectionCreationDate()
{
	QMap<QString,QueryToolView*>::iterator tmpIteratorViewsMap;
	QString tmpConnectionName;
	QDateTime tmpCreationDate;

	for (tmpIteratorViewsMap = viewsMap->begin();
		tmpIteratorViewsMap != viewsMap->end();++tmpIteratorViewsMap)
	{
		tmpConnectionName = tmpIteratorViewsMap.key();
		tmpCreationDate = tmpIteratorViewsMap.data()->getCreationDate();
		printf("\t%s -> %s\n",tmpConnectionName.ascii(), 
			tmpCreationDate.toString(Qt::ISODate).ascii());
	}

	return;
}

void QueryTool::newConnection()
{
	QueryToolView *viewTmp;
	QueryToolDoc *docTmp;
	ConnectionParameters *cp;
	ConnectionDialog *cd;
	QString errorMessage;
	ConnectionParameters *cpIn;
	QueryEngine *tmpQueryEngine;

	cpIn = 0;

	if ((config->getBool("FillConnParamWithLastActiveConnParam") == true) 
		&& (lastViewCreated != 0))
	{
		// if there is at least one active connection, we get the last one
		// in order to get the corresponding connection parameters

		docTmp = lastViewCreated->getDocument();
		tmpQueryEngine = docTmp->getEngine();
		cpIn = new ConnectionParameters(tmpQueryEngine->getConnectionParameters());

		docTmp = 0;
		tmpQueryEngine = 0;
	}

	cd = new ConnectionDialog(this, config, cpIn);

	if (cpIn != 0)
	{
		delete cpIn;
	}

	if  (cd->exec() == 0)
	{
		// connection cancel
		delete cd;
		return;
	}

	cp = cd->getConnectionParameters();
	delete cd;

	if (cp == 0)
	{
		// error
		delete cd;
		return;
	}

	docTmp = new QueryToolDoc(this);
	
	if ((docTmp->connect(*cp, errorMessage)) == false)
	{
		KMessageBox::error (this, errorMessage);
		// error
		delete cp;
		delete docTmp;
		return;
	}
	
	delete cp;

	viewTmp = createClient(docTmp);

	if (viewTmp == 0)
	{
		// fails to create view
		docTmp->disconnect();
		delete docTmp;
		KMessageBox::error (this, i18n("Error while creating view"));
		return;
	}

	lastViewCreated = viewTmp;
	
	return;	
}


void QueryTool::disconnect()
{
	QueryToolView *v;
	
	v  = (QueryToolView*) workspace->activeWindow();
	
	if (!v)
	{
		KMessageBox::error (this, i18n("There is no connections."));
		return;
	}
	
	// question : continue to disconnect or do nothing

	if (v->askIfDisconnect() == false)
	{
		return;
	}
	
	// is there a batch running ?
	
	if ((v->isBatchRunning()) == true)
	{
		KMessageBox::error (this, i18n("Error : a batch is running !"));
		return;
	}
	
	delete v;
	
	// auto switch to another view ?
	
	return;
}


void QueryTool::byebye()
{
	QString messageTmp;
	int result;

	if (viewsMap->count() > 0)
	{
		// check if there are some batchs running

		if (isThereBatchRunning() > 0)
		{
			messageTmp = i18n("Error : a batch is running !");
			KMessageBox::error(this, messageTmp, i18n("Can not quit"));
			return;
		}

		messageTmp = i18n("There are still connections.");
		messageTmp += " " + i18n("If you quit, you will lose them.");
		messageTmp += " " + i18n("Do you want to quit ?");

		// ask for a confirmation
		result = KMessageBox::warningYesNo(this, messageTmp,
			i18n("Warning : quitting"));

		if (result == KMessageBox::No)
		{
			// No pressed
			return;
		}
	}

	internalCanQuit = true;
	KMainWindow::close();
	
	return;	
}


void QueryTool::changeConnectionView(int pIndex)
{
#ifdef DEBUG_APP
	printf ("DEBUG_APP QueryTool::changeConnectionView> changeConnectionView %d\n",pIndex);
#endif

	QString sTmp;
	QueryToolDoc *docTmp;

	if ((pIndex==-1)||(connections->count() <= FIRST_CONN))
	{
		// case where this at least one connection
		// so using previous or next on nothing
		// or on one view means no change.
		return;
	}

	if ((pIndex == 1) || (pIndex == 2))
	{
		// previous or previous actions
		QueryToolView *viewTmp = (QueryToolView*) workspace->activeWindow();
		docTmp = viewTmp->getDocument();
		sTmp = docTmp->getName();
		QString sTmp2;
		bool notFound = true;
		unsigned int i,idx;

		for (i=FIRST_CONN;(i<connections->count())&&(notFound);i++)
		{
			sTmp2 = connections->text(connections->idAt(i));
			if (sTmp2 == sTmp)
			{
				notFound = false;
				idx = i;
			}
		}
		if (notFound == true)
		{
			// error
			return;
		}

		if (pIndex == 1)
		{
			// previous
			if (idx == FIRST_CONN)
			{
				// go to the last
				pIndex = connections->count() - 1;
			}
			else
			{
				pIndex = idx - 1;
			}
		}
		else
		{
			// next
			if (idx == connections->count() - 1)
			{
				// go to the first
				pIndex = FIRST_CONN;
			}
			else
			{
				pIndex = idx + 1;
			}
		}
	}

	// we had to set focus on view described by index.
	sTmp = connections->text(connections->idAt(pIndex));
	setActiveView(sTmp);

	return;
}


void QueryTool::previousConnectionView()
{
	return changeConnectionView(1);	
}


void QueryTool::nextConnectionView()
{
	return changeConnectionView(2);	
}


void QueryTool::configureApp()
{
	ConfigDialog *kcd = new ConfigDialog(this, config);
	
	kcd->exec();
	
	delete kcd;
	
	// apply changes
	
	toggleMainToolbar();
	
	return;
}


void QueryTool::switchViewToInput()
{
	QueryToolView *v = (QueryToolView*) workspace->activeWindow();
	
	// is there a view / a connection ?
	
	if (v == 0)
	{
		KMessageBox::error (this, i18n("There is no connections."));
		return;
	}
	
	v->switchToInput();
	
	return;
}


void QueryTool::switchViewToOutput()
{
	QueryToolView *v = (QueryToolView*) workspace->activeWindow();
	
	// is there a view / a connection ?
	
	if (v == 0)
	{
		KMessageBox::error (this, i18n("There is no connections."));
		return;
	}
	
	v->switchToOutput();
	
	return;
}


void QueryTool::switchViewToNoticeMessages()
{
	QueryToolView *v = (QueryToolView*) workspace->activeWindow();
	
	// is there a view / a connection ?
	
	if (v == 0)
	{
		KMessageBox::error (this, i18n("There is no connections."));
		return;
	}
	
	v->switchToNoticeMessages();
	
	return;
}


void QueryTool::configureKeyShortcuts()
{
	KKeyDialog keyDialog(false,this);
	
	keyDialog.configure(actionCollection());
	
	return;
}


void QueryTool::stopBatch()
{
	QueryToolView *v = (QueryToolView*) workspace->activeWindow();
	
	// is there a view / a connection ?
	
	if (v == 0)
	{
		KMessageBox::error (this, i18n("There is no connections."));
		return;
	}
	
	v->stopBatch();
	
	return;
}


void QueryTool::toggleMainToolbar()
{
	if (config->getBool("ShowToolBar") == true)
	{
		toolBar()->show();
	}
	else
	{
		toolBar()->hide();
	}
	
	return;	
}

void QueryTool::initView()
{
	// Workspace	
	
	QVBox* viewBack = new QVBox(this);
	viewBack->setFrameStyle( QFrame::StyledPanel | QFrame::Sunken );
	workspace = new QWorkspace(viewBack);
	setCentralWidget(viewBack);
	
	return;	
}

void QueryTool::initActions()
{
	KShortcut shortcutDummy;

	// Main Menu
	KShortcut shortcutNewConnection(CTRL+Key_N);
	actionNewConnection = new KAction(i18n("&New Connection"), shortcutNewConnection,
		this, SLOT(newConnection()), actionCollection(),"file_new_connection");
	actionNewConnection->setWhatsThis(i18n("Create a new connection to a database"));
	actionNewConnection->setIcon("connect_creating");
	
	KShortcut shortcutDisconnect(CTRL+Key_D);
	actionDisconnect = new KAction(i18n("&Disconnect"), shortcutDisconnect,
		this, SLOT(disconnect()), actionCollection(), "file_disconnect");
	actionDisconnect->setWhatsThis(i18n("Disconnect from current connection"));
	actionDisconnect->setIcon("fileclose");
	
	KShortcut shortcutExecute(CTRL+Key_E);
	actionExecute = new KAction (i18n("&Execute"), shortcutExecute,
		this, SLOT(executeInput()), actionCollection(),
		"file_execute");
	actionExecute->setWhatsThis(i18n("Execute the batch of the input window"));
	actionExecute->setIcon("exec");
	
	KShortcut shortcutOpenQuery(CTRL+Key_O);
	actionOpen = new KAction (i18n("&Open"), shortcutOpenQuery,
		this, SLOT(openQuery()), actionCollection(),
		"file_open");
	actionOpen->setWhatsThis(i18n("Load a batch into the input window"));
	actionOpen->setIcon("fileopen");
	
	KShortcut shortcutSaveQueryResult(CTRL+Key_S);
		actionSave = new KAction (i18n("&Save"), shortcutSaveQueryResult,
			this, SLOT(saveQueryResult()), actionCollection(),
			"file_save");
	actionSave->setWhatsThis(i18n("Save the batch or the results"));
	actionSave->setIcon("filesave");
	
	actionSaveAs = new KAction (i18n("&Save As"), shortcutDummy,
		this, SLOT(saveAsQueryResult()), actionCollection(),
		"file_save_as");
	actionSaveAs->setWhatsThis(i18n("Save the batch or the results"));
	actionSaveAs->setIcon("filesaveas");
	
	// Connection menu
	
	KShortcut shortcutPreviousConnections(ALT+Key_Left);
	
	actionPreviousConnection = new KAction (i18n("Previous Connection"),
		shortcutPreviousConnections, this,
		SLOT(previousConnectionView()),actionCollection(),
		"connections_previous_connection");
	actionPreviousConnection->setWhatsThis(i18n("Show previous connection data"));
	actionPreviousConnection->setIcon("previous");
	
	KShortcut shortcutNextConnections(ALT+Key_Right);
	
	actionNextConnection = new KAction (i18n("Next Connection"),
		shortcutNextConnections, this,
		SLOT(nextConnectionView()),actionCollection(),
		"connections_next_connection");
	actionNextConnection->setWhatsThis(i18n("Show next connection data"));
	actionNextConnection->setIcon("next");


	// Settings menu
	
	actionSettingsConfigureApp = new KAction(i18n("Configure ") + applicationName,
		shortcutDummy, this,
		SLOT(configureApp()), actionCollection(),
		"settings_configure_app");
	actionSettingsConfigureApp->setWhatsThis(i18n("Acces to options"));
	actionSettingsConfigureApp->setIcon("configure");
	
	actionSettingsConfigureKeyShortcuts = new KAction (i18n("Configure key shortcuts"),
		shortcutDummy, this, SLOT(configureKeyShortcuts()),
		actionCollection(), "settings_configure_key_shortcuts");
	actionSettingsConfigureKeyShortcuts->setIcon("configure_shortcuts");
	
	actionSettingsConfigureKeyShortcuts->setWhatsThis(
		i18n("Acces to Key Shortcuts Configuration"));

	// Key shortcut
	
	KShortcut shortcutSwitchToInput(CTRL+Key_1);
	
	actionSwitchToInput = new KAction(i18n("Switch to Input"),
		shortcutSwitchToInput, this, SLOT(switchViewToInput()),
		actionCollection(),"switch_to_input");
	
	KShortcut shortcutSwitchToOutput(CTRL+Key_2);
	
	actionSwitchToOutput = new KAction(i18n("Switch to Output"),
		shortcutSwitchToOutput, this, SLOT(switchViewToOutput()),
		actionCollection(),"switch_to_output");
	
	KShortcut shortcutSwitchToNoticeMessages(CTRL+Key_3);
	
	actionSwitchToNoticeMessages = new KAction(i18n("Switch to Notice Messages"),
		shortcutSwitchToNoticeMessages, this, SLOT(switchViewToNoticeMessages()),
		actionCollection(),"switch_to_notice_messages");
	
	KShortcut shortcutStopBatch(CTRL+Key_T);
	
	actionStopBatch = new KAction(i18n("Stop batch"), shortcutStopBatch,
		this, SLOT(stopBatch()), actionCollection(),"stop_batch");
	actionStopBatch->setWhatsThis(i18n("Stop a running batch in the active window"));
	actionStopBatch->setIcon("stop");
	
	return;	
}

void QueryTool::updateConnectionMenu()
{
	QMap<QString,QueryToolView*>::Iterator i;
	unsigned int j;
	
	connections->clear();
	actionPreviousConnection->plug(connections);
	actionNextConnection->plug(connections);
	connections->insertSeparator();
	
	if (viewsMap == 0)
	{
		return;
	}
	
	j=connections->count();
	
	for (i=viewsMap->begin();i!=viewsMap->end();++i)
	{
		connections->insertItem(i.key().ascii(), this,
			SLOT(changeConnectionView(int)), 0,j);
		j++;
	}
	
	return;
}

void QueryTool::updateLastView()
{
	QMap<QString,QueryToolView*>::Iterator i;
	QWidgetList listTmp;
	QueryToolView *viewTmp;
	QDateTime maxDateTime;
	
	lastViewCreated = 0;

	if ((viewsMap == 0) || (viewsMap->empty() == true))
	{
		return;
	}

	i=viewsMap->begin();
	viewTmp = i.data();
	maxDateTime = viewTmp->getCreationDate();

	i++;

	while (i!=viewsMap->end())
	{
		if ((i.data()->getCreationDate()) > maxDateTime)
		{
			viewTmp = i.data();
			maxDateTime = viewTmp->getCreationDate();
		}
		i++;
	}

	lastViewCreated = viewTmp;

	viewTmp = 0;
	
	return;
}

unsigned int QueryTool::isThereBatchRunning()
{
	unsigned int nbRunningBatches;
	QMap<QString,QueryToolView*>::Iterator i;
	QueryToolView *viewTmp;
	
	nbRunningBatches = 0;
	viewTmp = 0;
	
	for (i=viewsMap->begin();i!=viewsMap->end();++i)
	{
		viewTmp = i.data();
		if (viewTmp->isBatchRunning() == true)
		{
			nbRunningBatches++;
		}
	}
#ifdef DEBUG_APP
	printf("DEBUG_APP QueryTool::isThereBatchRunning> Nb running batches: %d\n", nbRunningBatches);
#endif

	return nbRunningBatches;
}

ToolSuiteConfig * QueryTool::getConfig()
{
	return config;
}

