/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2002 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */

#include <stdio.h>

#include "structures.h"
#include "../common/string_map.h"
#include "../compiler/epsilon.h"

struct string_map *defines_map; /* name -> index of table */

#define TABLE_INITIAL_SIZE 1

struct defines_element{
  char identifier[IDENTIFIER_LENGTH+1];
  char *definition; /* dynamically allocated */
  int is_active;
} *defines_table;
int defines_table_pointer;
int defines_table_size;

void create_repl_data_structures(){
  int i;

  defines_map=create_string_map(sizeof(int)); // updated in 2003
  defines_table=malloc(sizeof(struct defines_element)
		                        *(defines_table_size=TABLE_INITIAL_SIZE));
  defines_table_pointer=0;
  for(i=0;i<defines_table_size;i++){
    defines_table[i].definition=(char*)malloc(sizeof(char));
    defines_table[i].definition[0]='\0';
    defines_table[i].is_active=1;
  }
}

void destroy_repl_data_structures(){
  destroy_string_map(defines_map);
}

void insert_define(char* name, char* value){
  int value_length;

  if(is_define_present(name)){
    int* pos;
    /* If the define is not active then don't overwite it */
    pos=access_string_map(defines_map,name);
    if(defines_table[*pos].is_active){
      printf("\aWarning: redefining identifier %s.\n",name);
      remove_define(name);
    }
  }
  
  if(defines_table_pointer==defines_table_size){
    int i;
    defines_table_size*=2;
    defines_table=realloc(defines_table,
			  sizeof(struct defines_element)*defines_table_size);
    for(i=defines_table_size/2;i<defines_table_size;i++){
      defines_table[i].definition=(char*)malloc(sizeof(char));
      defines_table[i].definition[0]='\0';
      defines_table[i].is_active=1;
    }
  }

  strcpy(defines_table[defines_table_pointer].identifier,name);
  value_length=strlen(value);
  defines_table[defines_table_pointer].definition=realloc(defines_table[defines_table_pointer].definition,
							  sizeof(char)*value_length);
  strcpy(defines_table[defines_table_pointer].definition,value);
  insert_into_string_map(defines_map,name,&defines_table_pointer);
  defines_table_pointer++;
}

int is_define_present(char* name){
  return access_string_map(defines_map,name) != NULL;
}

void remove_define(char* name){
  int* x=access_string_map(defines_map,name);

  if(x==NULL){
    printf("\aThe identifier %s has not been defined.\n",name);
    return;
  }

  /* If we arrived here then name is bound. */

  //printf("\t{%i}\n",*x);

  /* Clear identifier and definition fields in the table entry: */
  defines_table[*x].identifier[0]='\0';
  
  defines_table[*x].definition=(char*)realloc(defines_table[*x].definition,
					      sizeof(char));
  defines_table[*x].definition[0]='\0';
  
  /* Remove the map entry: */
  remove_from_string_map(defines_map,name);
  //printf("A\n");
}

void activate_define(char* name){
  int* pos;
  
  if((pos=access_string_map(defines_map,name))!=NULL)
    defines_table[*pos].is_active=1;
}

void deactivate_define(char* name){
  int* pos;

  if((pos=access_string_map(defines_map,name))!=NULL)
    defines_table[*pos].is_active=0;
}

void dump_defines(FILE* f){
  int i,empty=1;

  fprintf(f,"/* Bindings in the global environment: */\n\n");
  for(i=0;i<defines_table_pointer;i++)
    if((strcmp(defines_table[i].identifier,""))&&(defines_table[i].is_active)){
      if(defines_table[i].is_active)
        fprintf(f,"define %s = %s\n\n",defines_table[i].identifier,defines_table[i].definition);
      else
        fprintf(f,"/* define %s = %s */\n\n",defines_table[i].identifier,defines_table[i].definition);

      empty=0;
    }
  if(empty)
    fprintf(f,"/* [none] */\n");
  fprintf(f,"\n\n");
}



void remove_active_of_not_active_define(char* name,int active){
  int i;
  
  for(i=0;i<defines_table_pointer;i++)
    if(/* (defines_table[i].is_active XOR active) AND !strcmp(defines_table[i].identifier,name) */
       ((defines_table[i].is_active&&active)||((!defines_table[i].is_active)&&(!active)))
       &&(!strcmp(defines_table[i].identifier,name))){
      /* Remove the entry from the table: */
      defines_table[i].definition=(char*)realloc(defines_table[i].definition,sizeof(char));
      defines_table[i].definition[0]='\0';
      strcpy(defines_table[i].identifier,"");
      
      /* Fix the hash table entry if needed: */
      if(*(int*)access_string_map(defines_map,name)==i){
	int j;

	remove_from_string_map(defines_map,name);
	for(j=0;j<defines_table_pointer;j++)
	  if(!strcmp(defines_table[j].identifier,name)){
	    insert_into_string_map(defines_map,name,&j);
	    break;
	  }
      }

      break;
    }
}

void remove_active_define(char* name){
  remove_active_of_not_active_define(name,1);
}

void remove_not_active_define(char* name){
  remove_active_of_not_active_define(name,0);
}
