/* This file is part of GNU epsilon, a functional language implementation

Copyright (C) 2003 Luca Saiu

GNU epsilon is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published
by the Free Software Foundation; either version 2, or (at your
option) any later version.

GNU epsilon is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with epsilon; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <malloc.h>
#include <limits.h>
#include <string.h>

#include "../../common/command_line.h"
#include "../../common/malloc.h"
#include "heap.h"
#include "set.h"


/* This implementation could be optimized replacing malloc() and free() with
   allocation and freeing from homogenoeus pages; however this would create
   a circular dependence set->homogeneous_page->set (dirty but we could live
   with that), and I think the performance improvement would be low, only
   related to page creation/deletion. */

/* The set consists of a buffer whose first part is a struct set; the second 
   part contains the array of buckets. */
struct set{
  integer_t size;
};

/* Returns the array of buckets (i.e. the payload); this should cost just an
   assembler instruction: */
#define SET_TO_BUCKETS(s) \
  ((word_t*)(((char*)(s)) + sizeof(struct set)))

typedef struct bucket_element* bucket_element_t;
struct bucket_element{
  word_t datum;
  bucket_element_t next;
};

unsigned_integer_t hash_function1(word_t pointer, integer_t size){
  return (unsigned_integer_t)pointer % (unsigned_integer_t)size;
}

integer_t hash_function(word_t pointer, integer_t size){
  integer_t x = (integer_t) hash_function1(pointer, size);
  //printf("hash_function(): %p -> "INTEGER_T_FORMAT"\n", pointer, x);
  return x;
}

set_t create_set(integer_t size){
  set_t r;
  integer_t i;
  word_t* buckets;
  
  /* Allocate and set fields: */
  r = (set_t) xmalloc(sizeof(struct set) + size * sizeof(word_t));
  r->size = size;

  /* Start making all the content empty: */
  buckets = SET_TO_BUCKETS(r);
  for(i = 0; i < size; i++)
    buckets[i] = NULL;
  
  return r;
}

void destroy_set(set_t set){
  /* Free the buffer holding both the header and the payload: */
  free(set);
}

void insert_pointer_into_set(word_t pointer, set_t set){
  /* Apply the hash function to find the address: */
  integer_t address;
  word_t* buckets = SET_TO_BUCKETS(set);
  bucket_element_t new_bucket_element;
  
  /* Check whether pointer already belongs to set; if it does then do nothing 
     and return: */
  if(does_pointer_belong_to_set(pointer, set))
    return;
  
  /* Ok, the pointer does not belong to set; insert it: */
  address = hash_function(pointer, set->size);

  /* Create the new bucket element, to be prepended to the current bucket: */
  new_bucket_element = xmalloc(sizeof(struct bucket_element));
  new_bucket_element->datum = pointer;
  new_bucket_element->next = buckets[address];

  /* Prepend the new bucket element: */
  buckets[address] = new_bucket_element;
}

integer_t does_pointer_belong_to_set(word_t pointer, set_t set){
  integer_t address = hash_function(pointer, set->size);
  word_t* buckets = SET_TO_BUCKETS(set);
  bucket_element_t bucket_element = buckets[address];
  
  /* Scan the bucket searching for pointer: */
  while(bucket_element != NULL){
    if(bucket_element->datum == pointer)
      return (integer_t)1; /* we found pointer */
    bucket_element = bucket_element->next;
  };
  
  /* If we arrived here then we scanned the whole bucket without finding
     pointer: */
  return (integer_t)0; /* we did not find pointer */
}

/* Return 0 if the element was present, -1 otherwise: */
integer_t remove_pointer_from_set(word_t pointer, set_t set){
  integer_t address = hash_function(pointer, set->size);
  word_t* buckets = SET_TO_BUCKETS(set);
  /* The following two pointers scan the bucket in parallel, always staying
     at the distance of 1 element: */
  bucket_element_t bucket_element_1 = buckets[address];
  bucket_element_t bucket_element_2;
  
  /* If the bucket is empty then do nothing and return;*/
  if(bucket_element_1 == NULL)
    return (integer_t)-1; /* we did not find pointer */
  
  /* Particular case: check whether pointer is the *first* element of
     the bucket (if we arrived here bucket_element_1 != NULL): */
  if(bucket_element_1->datum == pointer){ 
    /* We found pointer in the first element; remove it: */
    buckets[address] = bucket_element_1->next;
    free(bucket_element_1);
    return (integer_t)0; /* we found pointer and removed it */
  }
  
  /* Scan the bucket searching for pointer (we can use do..while instead of
     while since we are sure that the first element is not NULL, and do..while
     is slightly faster: */
  do{
    bucket_element_2 = bucket_element_1->next;
    
    if(bucket_element_2 == NULL)
      return (integer_t)-1; /* we did not find pointer */
    else if(bucket_element_2->datum == pointer){
      /* We found pointer in bucket_element_2; remove it and return: */
      bucket_element_1->next = bucket_element_2->next;
      free(bucket_element_2);
      return (integer_t)0; /* we found pointer and removed it */
    }
      
    bucket_element_1 = bucket_element_2; /* advance the pointer */
  } while(bucket_element_1 != NULL);
  
  /* If we arrived here then we scanned the whole bucket without finding
     pointer: */
  return (integer_t)-1; /* we did not find pointer */
}

/* Only useful for debugging (complexity is O(max{m, n})) */
void dump_set(set_t s){
  integer_t i;
  word_t* buckets = SET_TO_BUCKETS(s);
  bucket_element_t bucket_element;
  
  fprintf(stderr, "{ ");
  for(i = 0; i < s->size; i++){
    bucket_element_t bucket_element = buckets[i];
    while(bucket_element != NULL){
      fprintf(stderr, "%p ", bucket_element->datum);
      bucket_element = bucket_element->next;
    }
  }
  
  fprintf(stderr, "}\n");
}
/*
int main(){
  integer_t i;
  set_t s;
  
  s = create_set(1000000);

  for(i = 0; i < 100000; i++)
    insert_pointer_into_set((word_t)(i * 4), s);

  insert_pointer_into_set(0xde, s);

  for(i = 0; i < 100000; i++)
    remove_pointer_from_set((word_t)(i * 4), s);

  fprintf(stderr, "%i\n",does_pointer_belong_to_set(0xde, s));
  fprintf(stderr, "%i\n",does_pointer_belong_to_set(0xb01ade, s));
  //dump_set(s);
  destroy_set(s);
  return 0;
}
*/
